/*
 * Copyright 1997, Steven M. Robbins <steve@nyongwa.montreal.qc.ca>
 * $Id: DepTable.java,v 1.12.2.1 1998/11/09 22:11:02 steve Exp $
 */

package smr.JavaDeps;


import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Stack;
import java.io.File;


/**
 * Node of the dependency graph.
 *
 * <p>In the simple case, a node corresponds to single class, and the
 * links correspond to other classes that must be built before this one.
 * Hence, the linked-to class files will end up being the dependencies for
 * the class file of the current node.
 *
 * <p>Unfortunately, java classes are frequently mutually-dependent.  Think
 * of a linked-list and list-item classes.  If they are both public,
 * they must appear in different .java files, yet they will be dependent
 * on the other.
 *
 * <p>Sometimes, therefore, a node will contain a <em>set</em> of such
 * mutually-dependent classes.  Their .java files will all be compiled
 * together.  The dependencies for such a node will be the union of the
 * dependencies of each class in the set.
 *
 * <p>Finally, a word about the links.  In the first, "discovery" phase
 * of operation, the links will be stored as class names, some of which
 * (like, say, java.awt.Frame) may not even be part of the project being
 * built.  Later, a proper graph is built, when these links are turned into
 * references to other DepNodes.  The non-project dependencies are weeded
 * out at this time.
 **/
 
class DepNode
{
    /**
     * The class and file names are either a single String, (in the
     * case of simple nodes) or a Vector of them, (for multi-nodes).
     **/
    private Object className, classFileName, sourceFileName;
     
    /**
     * A list of other symbols that it depends on.
     * The vector elements are Strings or DepNodes.
     **/
    private Vector links;

    /**
     * True iff the class has one or more native methods.  For multi-nodes,
     * this is set if any of the classes have a native method.  This will
     * cause javah to be run on all the classes, even if some have no
     * native method.  This causes extra .h files.  FIXME
     **/
    private boolean hasNative;
    
    /**
     * A status indicator.
     * This is used in the depth-first searches to indicate status during
     * the search.  The colour is either WHITE, GRAY, or BLACK, after
     * Cormen, Lieserson, and Rivest.
     **/
    byte colour;

    final static byte WHITE = 0;
    final static byte GRAY = 1;
    final static byte BLACK = 2;

    static String statementDelim = "";
    static String blockStart = "";
    static String blockEnd = "";
    
    /**
     * If true, output some comments.
     **/
    static boolean emitComments;

    /**
     * String to use as build rule for compiling one or more java files.
     **/
    static String buildCommand;

    /**
     * If true, build subs as well as headers, for native methods.
     **/
    static boolean buildStubs;

    /**
     * String to use as header build rule; used for native methods.
     **/
    static String headerBuildCommand;
    

    // For debugging purposes only
    public String toString()
    {
	return "{classname=" + className +
	    ",files=(" + sourceFileName +
	    " --> " + classFileName + ")}";
    }
    
    /**
     * Constructor during phase one.  This creates a simple (not multi)
     * node.
     **/
    DepNode( String className, String classFileName, String sourceFileName )
    {
	this.className = className;
	this.classFileName = classFileName;
	this.sourceFileName = sourceFileName;
	links = new Vector();
	hasNative = false;
	colour = WHITE;
    }

    /**
     * Constructor for multinode.
     *
     * @param nodeSet the set of DepNodes to be collapsed into a multi-node
     **/
    DepNode( Vector nodeSet )
    {
	className = new Vector();
	classFileName = new Vector();
	sourceFileName = new Vector();
	links = new Vector();
	colour = WHITE;
	hasNative = false;

	for ( Enumeration e = nodeSet.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();
	    if ( dn.isSimple() ) {
		addClass( (String)dn.className );
		addClassFile( (String)dn.classFileName );
		addSourceFile( (String)dn.sourceFileName );
	    } else {
		addClass( (Vector)dn.className );
		addClassFile( (Vector)dn.classFileName );
		addSourceFile( (Vector)dn.sourceFileName );
	    }
	    addLink( dn.links );
	    if ( dn.hasNative )
		hasNative = true;
	}
    }

    /**
     * Clear links; in preparation to replace them.
     **/
    void clearLinks()
    {
	links = new Vector();
    }

    /**
     * Iterate over the links to adjacent nodes.
     **/
    Enumeration adjacent()
    {
	return links.elements();
    }

    /**
     * Indicate the class has a native method.
     **/
    void setHasNative()
    {
	hasNative = true;
    }
    
    /**
     * Build a list of source (.java) files.
     *
     * @return the space-separated list of source filenames associated
     * with the class or classes of this node
     **/
    String javaFiles()
    {
	if ( isSimple() )
	    return quote((String)sourceFileName);

	String cf = "";
	for ( Enumeration e = ((Vector)sourceFileName).elements();
	      e.hasMoreElements(); )
	    {
		cf += quote((String)e.nextElement()) + " ";
	    }
	return cf;
    }

    /**
     * Build a list of object (.class) files.
     *
     * @return the space-separated list of object filenames associated
     * with the class or classes of this node
     **/
    String classFiles()
    {
	if ( isSimple() )
	    return quote((String)classFileName);

	String cf = "";
	for ( Enumeration e = ((Vector)classFileName).elements();
	      e.hasMoreElements(); )
	    {
		cf += quote((String)e.nextElement()) + " ";
	    }
	return cf;
    }

    String quote( String anArg ) {
        if( DepTable.jmkSyntax ) {
            return '"' + anArg.trim().replace('\\', '/') + '"';
        }
	else {
	   // escape $ signs
	   StringBuffer s = new StringBuffer(anArg.length() + 10);
	   int start = 0;
	   int end = 0;
	   while ((end = anArg.indexOf('$', start)) >= 0)
	   {
	      s.append(anArg.substring(start, end + 1));
	      s.append('$');
	      start = end + 1;
	   }
	   s.append(anArg.substring(start));
	   return s.toString();
	}
    }
    /**
     * Build a list of classes.
     *
     * @return the space-separated list of classes associated
     * with this node
     **/
    String classNames()
    {
	if ( isSimple() )
	    return (String)className;

	String cf = "";
	for ( Enumeration e = ((Vector)className).elements();
	      e.hasMoreElements(); )
	    {
		cf += (String)e.nextElement() + " ";
	    }
	return cf;
    }

    /**
     * Output a makefile rule.
     *
     * Prints to standard output, a makefile rule.  The target and
     * dependency strings must be suplied.  The build string may be
     * null, in which case only the dependency is output.
     **/
    private void printRule( String target, String deps, String build )
    {
	JavaDeps.out.print( target + ": " + deps + statementDelim );
	if ( build != null ) {
	    JavaDeps.out.println( blockStart );
            JavaDeps.out.println( "\t" + build + statementDelim );
            if( DepTable.jmkSyntax ) {
                JavaDeps.out.println( blockEnd );
            }
        }
        else {
            JavaDeps.out.println();
        }
	JavaDeps.out.println();
    }

    /**
     * Convert a class name into prefix for native methods' header/stubs
     * file.
     **/
    private static String stubPrefix( String className )
    {
	return className.replace( '.', '_' );
    }


    /**
     * Build a list of stub files.
     *
     * @return the space-separated list of .h and .c files associated
     * with the native methods of this node
     **/
    String stubFiles()
    {
	if ( ! hasNative )
	    return null;
	
	if ( isSimple() ) {
	    String pref = stubPrefix( (String)className );
	    String s = pref + ".h";
	    if ( buildStubs )
		s += " " + pref + ".c";
	    return s;
	}

	String s = "";
	for ( Enumeration e = ((Vector)className).elements();
	      e.hasMoreElements(); )
	    {
		String pref = stubPrefix( (String)e.nextElement() );
		s += pref + ".h ";
		if ( buildStubs )
		    s += pref + ".c ";
	    }
	return s;
    }

    /**
     * Dump rules for building native-method header & stubs file.
     * @param cn the class name
     * @param cfn class file name
     **/
    private void dumpNative( String cn, String cfn )
    {
	String pref = stubPrefix( cn );

	printRule( pref + ".h", cfn, headerBuildCommand + " " + cn );

	if ( buildStubs )
	    printRule( pref + ".c", cfn,
		       headerBuildCommand + " -stubs " + cn );
    }

    /**
     * Output all makefile rules for this node.
     **/
    public void dump()
    {
	String left = classFiles();
	String right = javaFiles();

	if ( emitComments ) {
	    JavaDeps.out.println( "# Rules for " + classNames() );
	    JavaDeps.out.println( "#" );
	}

	printRule( left, right, (buildCommand == null ? null :
				 buildCommand + " " + right) );
	for ( Enumeration e = links.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();
	    printRule( left, dn.classFiles(), null );
	}

	// Deal with creating header files, for native methods.
	//
	if ( hasNative && headerBuildCommand != null ) {
	    if ( isSimple() )
		dumpNative( (String)className, (String)classFileName );
	    else {
		for ( int i = 0; i < ((Vector)className).size(); ++i ) {
		    dumpNative( (String)((Vector)className).elementAt( i ),
				(String)((Vector)classFileName).elementAt( i )
				);
		}
	    }
	}	    

	JavaDeps.out.println();
    }

    boolean isSimple()
    {
	return className instanceof String;
    }

    boolean isMulti()
    {
	return className instanceof Vector;
    }

    private void becomeMulti()
    {
	if ( isMulti() )
	    return;

	Vector tmp = new Vector();
	tmp.addElement( className );
	className = tmp;

	tmp = new Vector();
	tmp.addElement( classFileName );
	classFileName = tmp;

	tmp = new Vector();
	tmp.addElement( sourceFileName );
	sourceFileName = tmp;
    }

    void addClass( String className, String classFileName )
    {
	becomeMulti();
	addClass( className );
	addClassFile( classFileName );
    }

    private void addClass( String name )
    {
	if ( ! ((Vector)className).contains( name ) )
	    ((Vector)className).addElement( name );
    }

    private void addClass( Vector v )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    addClass( (String)e.nextElement() );
    }

    private void addClassFile( String name )
    {
	if ( ! ((Vector)classFileName).contains( name ) )
	    ((Vector)classFileName).addElement( name );
    }

    private void addClassFile( Vector v )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    addClassFile( (String)e.nextElement() );
    }

    private void addSourceFile( String name )
    {
	if ( ! ((Vector)sourceFileName).contains( name ) )
	    ((Vector)sourceFileName).addElement( name );
    }

    private void addSourceFile( Vector v )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    addSourceFile( (String)e.nextElement() );
    }

    void addLink( String ref )
    {
	if ( ! links.contains( ref ) )
	    links.addElement( ref );
    }

    void addLink( DepNode dn )
    {
	if ( dn != this && ! links.contains( dn ) )
	    links.addElement( dn );
    }

    void addLink( Vector v )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    addLink( (DepNode)e.nextElement() );
    }

    /**
     * Replace selected link.
     *
     * If this node contains a link to the old node, it is replace by
     * a link to the new node.
     **/
    void replaceLink( DepNode old, DepNode newNode )
    {
	int i = links.indexOf( old );
	if ( i >= 0 ) {
	    if ( this == newNode || links.contains( newNode ) )
		links.removeElementAt( i );
	    else
		links.setElementAt( newNode, i );
	}
    }

    /**
     * Replace selected links.
     *
     * If this node contains a link to <em>any</em> of a set of nodes,
     * it is replaced by a link to the new node.
     **/
    void replaceLink( Vector v, DepNode newNode )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    replaceLink( (DepNode)e.nextElement(), newNode );
    }
}



/**
 * Table of symbols and dependencies.
 *
 * This is the interface to the parser.  It holds a graph of the DepNodes,
 * above.
 **/

public class DepTable
{
    /**
     * Remember the current state as we parse multiple files.
     **/
    private String sourceFileName, packageName;

    /**
     * Maps a class name to a fully-qualified one.
     * An import of "a.b.c" creates an alias for "c", with value "a.b.c".
     **/
    private Hashtable alias;

    /**
     * Store the wildcard imports.
     * These are used, as a last resort, for unqualified references.
     * We simply insert all possible strings formed by concatenating
     * all the wildcard imports with the reference name.  Vector elements
     * are Strings.
     *
     * @see #addReference
     **/
    private Vector wildImport;

    /**
     * Table mapping String (type name) to DepNode (list of dependencies)
     * during first phase, when the links are stored as classnames.
     **/
    private Hashtable table;
    private String curKey;
    private DepNode curValue;

    /**
     * Flag to indicate the first definition in a source file.
     **/
    private boolean isFirstDefinition;
    
    /**
     * After first phase, when a proper graph is built,
     * the list of nodes is stored here.
     **/
    private Vector graphNodes;

    /**
     * Record the destination directory if the "-d destdir"
     * option is used.
     **/
    private String destDir;

    /**
     * Topologically-sorted final list of classes, output as assignment
     * to "CLASSES" makefile variable.  Stub list is output as STUBS variable.
     *
     **/
    private String classList;
    private String stubList;

    /**
     * Cache the line separator string here.
     **/
    private String lineSeparator;

    /**
     * Build a stack of current classNames in scope in order to be able
     * to know what prefix to use for inner classes
     */
    private Stack classNames;
    /**
     * Another stack of the last number used to name the last anonymous
     * inner class parsed
     */
    private Stack annonInnerNum;

    /**
     * If true, generate makefile rules with jmk syntax
     **/
    static boolean jmkSyntax = false;
    static String statementDelim = "";
    static String blockStart = "";
    static String blockEnd = "";
    
    public static void setJmk( boolean b ) {
        jmkSyntax = b;
        if( jmkSyntax ) {
            DepNode.statementDelim = ";";
            DepNode.blockStart = " {";
            DepNode.blockEnd = "}";
        }
    }
    
    public DepTable( String destDir )
    {
	table = new Hashtable();
	this.destDir = destDir;
	lineSeparator = System.getProperty( "line.separator" );
    }

    /**
     * Parser calls this before a new source file is started.
     **/
    public void startFile( String fname )
    {
	sourceFileName = fname;
	packageName = null;
	alias = new Hashtable();
	wildImport = new Vector();
	classNames = new Stack();
	annonInnerNum = new Stack();

	curKey = null;
	curValue = null;
	isFirstDefinition = true;
    }

    /**
     * Parser calls this when "package x.y.z" is found.
     **/
    public void startPackage( String pname )
    {
	packageName = pname;
    }

    /**
     * Parser calls this when "import a.b.c" is found.
     **/
    public void addImport( String iname )
    {
	int i = iname.lastIndexOf( '.' );

	// Ignore non-qualified import.  What does it mean, anyway??
	if ( i == -1 )
	    return;

	String prefix = iname.substring( 0, i );
	String suffix = iname.substring( i+1 );
	
	if ( suffix.equals( "*" ) ) {
	    wildImport.addElement( prefix );
	} else {
	    alias.put( suffix, iname );
	}
    }

    /**
     * Parser class this when encountering an annonymous inner class
     */
    public void addAnnonInnerClass()
    {
        int classNum;
        classNum = ((Integer)annonInnerNum.pop()).intValue();
        annonInnerNum.push(new Integer(++classNum));
        startDefinition(String.valueOf(classNum), false, false);
    }
    
    /**
     * Parser calls this when exiting a class/interface scope
     **/
    public void endDefinition()
    {
        classNames.pop();
        annonInnerNum.pop();
    }
    
    /**
     * Parser calls this when "class x" or "interface x" is found.
     **/
    public void startDefinition( String dname , boolean isTopLevel, boolean isLocal )
    {
        String className;

        if ( ! isTopLevel ) {
            className = (String)classNames.peek();
	    if (!isLocal)
	    {
	       className += '$' + dname;
	    }
	    else
	    {
	       int classNum = ((Integer)annonInnerNum.pop()).intValue();
	       annonInnerNum.push(new Integer(++classNum));
	       className += '$' + String.valueOf(classNum) + '$' + dname;
	    }
        } else {
            className =
                ( packageName != null ? packageName + "." : "" ) + dname;
        }
	String classFileName = "";
	
	if ( destDir == null ) {
	    // Class file is named for the last component of the class name,
	    // and appears in the same directory as the source file.
	    File sourceFile = new File( sourceFileName );
	    if ( sourceFile.getParent() != null )
		classFileName = sourceFile.getParent() + File.separator;

	    int i = className.lastIndexOf( '.' );
	    classFileName += className.substring( i+1 ) + ".class";
	} else {
	    // Class file appears in a subdirectory of destDir
	    classFileName = destDir + File.separator +
		className.replace( '.', File.separatorChar ) + ".class";
	}

	curKey = className;

	if ( isFirstDefinition ) {

	    isFirstDefinition = false;
	    curValue = new DepNode( className, classFileName, sourceFileName );

	    // Stick in all the imports as dependencies.  What do we do about
	    // wildcards?  Nothing at the moment ...
	    for ( Enumeration e = alias.elements(); e.hasMoreElements(); )
		addReference( (String)e.nextElement() );
	} else {
	    curValue.addClass( className, classFileName );
	}

        classNames.push(className);
        annonInnerNum.push(new Integer(0));
	table.put( curKey, curValue );
    }

    /**
     * Parser calls this whenever some type name is referenced.
     * This can be: member variable type, static method call, cast,
     * etc, etc.
     **/
    public void addReference( String rname )
    {
	boolean qualified = rname.lastIndexOf( '.' ) >= 0;

	// If not qualified, check aliases
	if ( !qualified ) {
	    String s = (String)alias.get( rname );
	    if ( s != null ) {
		rname = s;
		qualified = true;
	    }
	}

	if ( qualified ) {
	    curValue.addLink( rname );
	    return;
	}

	// Guess that it must be either in the package, or
	// one of the wildcard imports.  Insert all possibilities now
	// and sort it out later (in dump()).

	if ( packageName == null )
	    curValue.addLink( rname );
	else
	    curValue.addLink( packageName + "." + rname );

	for ( Enumeration e = wildImport.elements(); e.hasMoreElements(); ) {
	    String prefix = (String)e.nextElement();
	    curValue.addLink( prefix + "." + rname );
	}
    }

    /**
     * This is used when a whole lot of references are parsed all at
     * once.  For example, a list of "implements".
     **/
    public void addReference( Vector v )
    {
	for ( Enumeration e = v.elements(); e.hasMoreElements(); )
	    addReference( (String)e.nextElement() );
    }

    /** 
     * Used for a name found in a primary expression, where we may need to
     * strip off the last identifier, in case of reference to static variable
     * or method, (i.e. big.bad.bob.finger("booger") or not, in case of class
     * literal reference (i.e. java.lang.String.class).
     **/
    public void addReference( String pn, boolean stripLastIdent )
    {
	if ( stripLastIdent ) {
	    int i = pn.lastIndexOf( '.' );
	    if ( i > 0 ) {
		addReference( pn.substring( 0, i ) );
	    }
	} else {
	    addReference( pn );
	}
    }

    /**
     * Called to indicate that the current class has a native method.
     **/
    public void hasNative()
    {
	curValue.setHasNative();
    }

    
    /**
     * Set all nodes to white, preparation for a graph traversal.
     **/
    private void setGraphToWhite()
    {
	for ( Enumeration e = graphNodes.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();
	    dn.colour = DepNode.WHITE;
	}
    }

    private void addGraphNode( DepNode dn )
    {
	graphNodes.addElement( dn );
    }
    
    private void removeGraphNode( DepNode dn )
    {
	graphNodes.removeElement( dn );
    }
    
    private void removeGraphNode( Vector nodeSet )
    {
	for ( Enumeration e = nodeSet.elements(); e.hasMoreElements(); ) {
	    graphNodes.removeElement( e.nextElement() );
	}
    }

    /**
     * Replace links.
     * Search all edges of the graph.  Replace any links to a given set of
     * nodes with a link to a new node.  Any edge where the target is in
     * the old set is replaced with an edge, using the new target node.
     *
     * @param oldSet set of old nodes about to be replaced
     * @param newNode the new target node
     **/
    private void replaceGraphLinks( Vector oldSet, DepNode newNode )
    {
	for ( Enumeration e = graphNodes.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();
	    dn.replaceLink( oldSet, newNode );
	}
    }
    
    /**
     * Replace all the links of String with links of DepNode.  Put the
     * set of nodes into the vector graphNodes, as well.
     **/
    private void buildGraph( boolean debug )
    {
	if ( debug ) {
	    System.err.println( "Building graph ..." );
	    System.err.println();
	}
	
	graphNodes = new Vector();
	
	for ( Enumeration e = table.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();

	    if ( dn.colour != DepNode.WHITE )
		continue;

	    dn.colour = DepNode.BLACK;
	    addGraphNode( dn );

	    Enumeration adj = dn.adjacent();
	    dn.clearLinks();

	    while ( adj.hasMoreElements() ) {
		String ref = (String)adj.nextElement();
		DepNode newLink = (DepNode)table.get( ref );

		// Only make the link if the reference is part of the
		// project being built (hence is non-null).

		if ( newLink != null )
		    dn.addLink( newLink );
	    }

	    if ( debug ) {
		System.err.println( "NODE: " + dn );
		System.err.println();
	    }
	}

	// We should not use table henceforth, so null it to trap programming
	// errors 
	table = null;
    }

    /**
     * Collapse cycles in graph into multi-nodes.
     * Make can't deal with circular dependencies, so we make the graph
     * acyclic by changing all cycles into "multi-nodes".  The approach used
     * here does a DFS to find the cycle, and stupidly re-starts the DFS
     * whenever a cycle is detected.  Yes, I know this is nothing more than
     * finding the strongly-connected components.  To be improved.
     **/
    private void collapseCycles( boolean debug )
    {
	boolean cycleDetected;

	do {
	    cycleDetected = false;
	    setGraphToWhite();
	    try {
		for ( Enumeration e = graphNodes.elements();
		      e.hasMoreElements(); )
		    {
			DepNode dn = (DepNode)e.nextElement();
			if ( dn.colour == DepNode.WHITE )
			    collapseVisit( dn );
		    }
	    } catch ( GraphCycleException gce ) {
		cycleDetected = true;
		DepNode dn = new DepNode( gce.nodeSet );
		if ( debug ) {
		    System.err.println( "CYCLE: " + gce.nodeSet );
		    System.err.println();
		    System.err.println( "Created node: " + dn );
		    System.err.println();
		    System.err.println();
		}
		removeGraphNode( gce.nodeSet );
		addGraphNode( dn );
		replaceGraphLinks( gce.nodeSet, dn );
	    }
	} while ( cycleDetected );
    }

    /**
     * DFS VISIT step, during cycle collapsing.
     **/    
    private void collapseVisit( DepNode dn )
	 throws GraphCycleException
    {
	dn.colour = DepNode.GRAY;
	for ( Enumeration e = dn.adjacent(); e.hasMoreElements(); ) {
	    DepNode v = (DepNode)e.nextElement();

	    if ( v.colour == DepNode.WHITE )
		collapseVisit( v );
	    else if ( v.colour == DepNode.GRAY )
		throw new GraphCycleException( v );
	}
	dn.colour = DepNode.BLACK;
    }

    /**
     * DFS VISIT step, during dump.
     * Since output happens at the end of visiting a node, we are doing
     * a topological sort of the output.
     **/
    private void dumpVisit( DepNode dn  )
    {
	dn.colour = DepNode.GRAY;
	for ( Enumeration e = dn.adjacent(); e.hasMoreElements(); ) {
	    DepNode v = (DepNode)e.nextElement();

	    if ( v.colour == DepNode.WHITE )
		dumpVisit( v );
	    else if ( v.colour == DepNode.GRAY )
		throw new RuntimeException( "graph should be DAG!" );

	}
	dn.dump( );
        if( jmkSyntax ) {
            classList += dn.classFiles();
        }
        else {
            classList += " \\" + lineSeparator + "\t" + dn.classFiles();
        }

	String sf = dn.stubFiles();
	if ( sf != null )
	    stubList += "\t" + sf + " \\" + lineSeparator;
	dn.colour = DepNode.BLACK;
    }
    
    public void dump( boolean debug, boolean emitComments, boolean buildStubs,
		      String buildCommand, String headerBuildCommand,
		      String classVariable )
    {
	DepNode.emitComments = emitComments;
	DepNode.buildStubs = buildStubs;
	DepNode.buildCommand = buildCommand;
	DepNode.headerBuildCommand = headerBuildCommand;
	
	// Output makefile commands so that the variable $(CLASSES) holds all
	// the buildable class files.
	if ( emitComments ) {
	    JavaDeps.out.println(
			  "# -- Automatically generated by JavaDeps --" );
	    JavaDeps.out.println();
	}

	if ( debug )
	    System.err.println( "Building dependencies graph" );
	buildGraph( debug );

	if ( debug )
	    System.err.println( "Collapsing dependency cycles" );
	collapseCycles( debug );

	if ( debug )
	    System.err.println( "Dumping..." );
	
	setGraphToWhite();
	classList = "";
	stubList = "";
	for ( Enumeration e = graphNodes.elements(); e.hasMoreElements(); ) {
	    DepNode dn = (DepNode)e.nextElement();
	    if ( dn.colour == DepNode.WHITE )
		dumpVisit( dn );
	}

	JavaDeps.out.println();
	if ( headerBuildCommand != null )
	    JavaDeps.out.println( "STUBS = \\ " + lineSeparator +
				stubList + "# end of stubs list" +
				lineSeparator );
        if( jmkSyntax ) {
	    JavaDeps.out.println( classVariable + " = " +
			        classList + ";" + lineSeparator +
                                "# end of classes list" );
        }
        else {
	    JavaDeps.out.println( classVariable + " = " +
			        classList + lineSeparator +
                                "# end of classes list" );
        }
    }

} // public class DepTable


/**
 * Thrown when the DFS discovers a cycle.  We carry the list of nodes
 * in the cycle.
 **/

class GraphCycleException extends Exception
{
    // Retain the set of nodes in the cycle
    Vector nodeSet;

    GraphCycleException( DepNode cycleStart )
    {
	nodeSet = new Vector();

	// Find the cycle by following the path of gray nodes
	DepNode dn = cycleStart;
	do {
	    nodeSet.addElement( dn );
	    for ( Enumeration e = dn.adjacent(); e.hasMoreElements(); )
		{
		    dn = null; // cause an exception if a gray node not found
		    DepNode d = (DepNode)e.nextElement();
		    if ( d.colour == DepNode.GRAY ) {
			dn = d;
			break;
		    }
		}
	} while ( dn != cycleStart );
    }
}
