package io

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestParseDataSpec(t *testing.T) {
	cases := []struct {
		input     string
		output    *DataSpec
		mustError bool
	}{
		{
			input:  "/foo/bar",
			output: &DataSpec{FilePath: "/foo/bar"},
		},
		{
			input:  "./foo/bar.txt",
			output: &DataSpec{FilePath: "./foo/bar.txt"},
		},
		{
			input:  "json:/foo/bar",
			output: &DataSpec{Format: "json", FilePath: "/foo/bar"},
		},
		{
			input:     "json:::/foo/bar",
			output:    nil,
			mustError: true,
		},
		{
			input:     "json:foo:/foo/bar",
			output:    nil,
			mustError: true,
		},
		{
			input:     "json:foo=6=baz:/foo/bar",
			output:    nil,
			mustError: true,
		},
		{
			input: "csv:csv.headers=true;csv.separator=,;rego.path=foo:/foo/bar",
			output: &DataSpec{
				Format:   "csv",
				FilePath: "/foo/bar",
				Options: map[string]string{
					"csv.headers":   "true",
					"csv.separator": ",",
					"rego.path":     "foo",
				},
			},
		},
		{
			input: "csv.headers=true;csv.separator=,;rego.path=foo:/foo/bar",
			output: &DataSpec{
				Format:   "",
				FilePath: "/foo/bar",
				Options: map[string]string{
					"csv.headers":   "true",
					"csv.separator": ",",
					"rego.path":     "foo",
				},
			},
		},
		{
			input: "csv.headers=true;csv.separator=\\t;rego.path=foo:/foo/bar",
			output: &DataSpec{
				Format:   "",
				FilePath: "/foo/bar",
				Options: map[string]string{
					"csv.headers":   "true",
					"csv.separator": "\\t",
					"rego.path":     "foo",
				},
			},
		},
		{
			input: "csv.headers=true;csv.separator=,;rego.path=foo:",
			output: &DataSpec{
				Format:   "",
				FilePath: "",
				Options: map[string]string{
					"csv.headers":   "true",
					"csv.separator": ",",
					"rego.path":     "foo",
				},
			},
		},
		{
			input: `{"format": "csv", "file_path": "/foo/bar", "options": {"csv.headers": "true", "csv.separator": ",", "rego.path": "foo"}}`,
			output: &DataSpec{
				Format:   "csv",
				FilePath: "/foo/bar",
				Options: map[string]string{
					"csv.headers":   "true",
					"csv.separator": ",",
					"rego.path":     "foo",
				},
			},
		},
		{
			input: `{"format": "csv", "file_path": "/foo/bar", "options": {"bool": true, "int": 7}}`,
			output: &DataSpec{
				Format:   "csv",
				FilePath: "/foo/bar",
				Options: map[string]string{
					"bool": "true",
					"int":  "7",
				},
			},
		},
	}

	for _, c := range cases {
		actual, err := ParseDataSpec(c.input)

		if c.mustError {
			assert.NotNil(t, err)
		} else {
			assert.Nil(t, err)
		}

		// Needed because a nil map isn't the same thing as an empty
		// map.
		if c.output != nil && c.output.Options == nil {
			c.output.Options = make(map[string]string)
		}

		assert.Equal(t, c.output, actual)
	}
}

func TestResolveDefaults(t *testing.T) {
	cases := []struct {
		input    *DataSpec
		defaults *DataSpec
		expected *DataSpec
	}{
		{
			input: &DataSpec{},
			defaults: &DataSpec{
				FilePath: "foo",
				Format:   "bar",
				Options: map[string]string{
					"spam": "ham",
				},
			},
			expected: &DataSpec{
				FilePath: "foo",
				Format:   "bar",
				Options: map[string]string{
					"spam": "ham",
				},
			},
		},
		{
			input: &DataSpec{FilePath: "foo.bar"},
			defaults: &DataSpec{
				Options: map[string]string{
					"spam": "ham",
				},
			},
			expected: &DataSpec{
				FilePath: "foo.bar",
				Format:   "bar",
				Options: map[string]string{
					"spam": "ham",
				},
			},
		},
	}

	for _, c := range cases {
		c.input.ResolveDefaults(c.defaults)
		assert.Equal(t, c.expected, c.input)
	}
}
