/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.htmlparser.Node;
import org.htmlparser.NodeFilter;
import org.htmlparser.Parser;
import org.htmlparser.filters.HasAttributeFilter;
import org.htmlparser.filters.OrFilter;
import org.htmlparser.filters.TagNameFilter;
import org.htmlparser.nodes.TextNode;
import org.htmlparser.util.NodeList;
import org.htmlparser.util.ParserException;
import org.videolan.vlma.daemon.Daemon;
import org.videolan.vlma.dao.VLMaDao;
import org.videolan.vlma.model.Command;
import org.videolan.vlma.model.DTTChannel;
import org.videolan.vlma.model.FilesChannel;
import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.Program;
import org.videolan.vlma.model.SatChannel;
import org.videolan.vlma.model.Satellite;
import org.videolan.vlma.model.Server;
import org.videolan.vlma.model.StreamChannel;
import org.videolan.vlma.model.StreamingStrategy;
import org.videolan.vlma.order.CommandLogger;

/**
 * Implementation of the Data interface.
 *
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class DataImpl implements Data {

    private static final Logger logger = Logger.getLogger(DataImpl.class);

    private CommandLogger commandLogger;
    private Configuration configuration;
    private Daemon daemon;
    private IpBank ipBank;
    private ProgramFactory programFactory;
    private VLMaDao vlmaDao;
    private VLMaService vlmaService;

    /*************************************************************************
     *
     * Interaction with VLMa daemon
     *
     *************************************************************************/

    public void reload() {
        daemon.reload();
    }

    public void stop() {
        daemon.stop();
    }

    /*************************************************************************
     *
     * Management of VLMa configuration
     *
     *************************************************************************/

    /*
     * Configuration is not exposed directly because it is not serializable
     */

    public String getString(String key) {
        synchronized (configuration) {
            return configuration.getString(key);
        }
    }

    public Integer getInt(String key) {
        synchronized (configuration) {
            return configuration.getInt(key);
        }
    }

    public List getList(String key) {
        synchronized (configuration) {
            return configuration.getList(key);
        }
    }

    public void setProperty(String key, Object value) {
        synchronized (configuration) {
            configuration.setProperty(key, value);
        }
    }

    public void clearProperty(String key) {
        synchronized (configuration) {
            configuration.clearProperty(key);
        }
    }

    /*************************************************************************
     *
     * Data Access Objects
     *
     *************************************************************************/

    public List<Media> getMedias() {
        return vlmaDao.getMedias();
    }

    public List<Satellite> getSatellites() {
        return vlmaDao.getSatellites();
    }

    public List<Server> getServers() {
        return vlmaDao.getServers();
    }

    public Media getMedia(int id) {
        return vlmaDao.getMedia(id);
    }

    public Satellite getSatellite(int id) {
        return vlmaDao.getSatellite(id);
    }

    public Server getServer(int id) {
        return vlmaDao.getServer(id);
    }

    public void update(Media media) {
        Media m = vlmaDao.getMedia(media.getId());
        if (m.getProgram() != null
                && m.getProgram().getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)
                && (media.getProgram() == null
                        || !media.getProgram().getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST))) {
            ipBank.releaseIp(m.getProgram().getIp());
        } else if (media.getProgram() != null
                && (m.getProgram() == null
                        || !m.getProgram().getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST))
                && media.getProgram().getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)) {
            media.getProgram().setIp(ipBank.getIp());
        }
        // Update sap group
        if (media.getProgram() != null) {
            String sapGroupId = null;
            if (media instanceof FilesChannel) {
                sapGroupId = "vlma.announcement.sap.files.group";
            } else if (media instanceof StreamChannel) {
                sapGroupId = "vlma.announcement.sap.stream.group";
            } else if (media instanceof DTTChannel) {
                sapGroupId = "vlma.announcement.sap.dtt.group";
            } else if (media instanceof SatChannel) {
                String category = ((SatChannel) m).getCategory();
                if ("R-DIG".equals(category) || "R-DIG-CRYPT".equals(category)) {
                    sapGroupId = "vlma.announcement.sap.radio.group";
                } else {
                    sapGroupId = "vlma.announcement.sap.satellite.group";
                }
            }
            if (sapGroupId != null)
                media.getProgram().setSapGroup(configuration.getString(sapGroupId));
        }
        vlmaDao.update(media);
    }

    public void update(Satellite satellite) {
        vlmaDao.update(satellite);
    }

    public void update(Server server) {
        vlmaDao.update(server);
    }

    public void add(Media media) {
        vlmaDao.add(media);
    }

    public void add(Satellite satellite) {
        vlmaDao.add(satellite);
    }

    public void add(Server server) {
        vlmaDao.add(server);
    }

    public void remove(Media media) {
        vlmaDao.remove(media);
    }

    public void remove(Satellite satellite) {
        vlmaDao.remove(satellite);
    }

    public void remove(Server server) {
        vlmaDao.remove(server);
    }

    /*************************************************************************
     *
     * Various utilities
     *
     *************************************************************************/

    public String getUrlOfProgram(Program program) {
        if (program != null) {
            if (program.getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.HTTP)) {
                return "http://" + program.getPlayer().getHostAddress() + ":" + configuration.getInt("vlma.streaming.http.port") + "/" + program.hashCode();
            } else if (program.getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)) {
                return "udp://@" + program.getIp().getHostAddress();
            }
        }
        return null;
    }

    public Program newProgram() {
        return programFactory.newProgram();
    }

    /**
     * Extract data from a HTML row where, in each cell, several <br />
     * -separated `rows' can be found.
     *
     * @param nodes
     *            the list of nodes which represent the row
     * @param column
     *            the column (beginning with 0) where to find data
     * @param row
     *            the row (beginning with 0) where to find data
     * @param maxRow
     *            the expected number of rows - 1 (in fact the highest row
     *            number)
     */
    private static String htmlCellExtract(NodeList nodes, int column, int row, int maxRow) {
        String result = "";
        if (maxRow == 0) {
            Node text = nodes.elementAt(column);
            result = text.toPlainTextString();
        }
        else {
            NodeFilter brTagFilter = new TagNameFilter("br");
            NodeList breakLines = nodes.elementAt(column).getChildren().extractAllNodesThatMatch(brTagFilter);
            if (breakLines.size() == maxRow) {
                if (row == 0) {
                    Node text = breakLines.elementAt(0).getPreviousSibling();
                    while (!(text instanceof TextNode) || "".equals(((TextNode) text).getText().trim())) {
                        text = text.getPreviousSibling();
                        if (text == null) {
                            return "";
                        }
                    }
                    result = ((TextNode) text).getText();
                }
                else {
                    Node text = breakLines.elementAt(row - 1).getNextSibling();
                    while (!(text instanceof TextNode) || "".equals(((TextNode) text).getText().trim())) {
                        text = text.getNextSibling();
                        if (text == null) {
                            return "";
                        }
                    }
                    result = ((TextNode) text).getText();
                }
            }
        }
        result = result.replaceAll("&nbsp;", " ");
        return result.trim();
    }

    synchronized public void updateSatChannels(URL source) throws IOException {
        List<String> tvFilter = new ArrayList<String>();
        Set<String> coverages = new HashSet<String>();
        tvFilter.add("TV-DIG-CRYPT");
        tvFilter.add("TV-DIG");
        tvFilter.add("R-DIG");
        tvFilter.add("R-DIG-CRYPT");
        tvFilter.add("TV-HD");
        Set<Media> newMedias = new HashSet<Media>();
        newMedias.clear();
        logger.info("Downloading " + source.toString());

        try {
            Parser parser = new Parser(source.openConnection());
            NodeList rows = parser.parse(new TagNameFilter("tr"));
            logger.info(source.toString() + " downloaded and parsed.");
            int frequency = 0;
            char polarisation = 'A';
            String coverage = "";
            NodeFilter thTagFilter = new OrFilter(new TagNameFilter("th"), new HasAttributeFilter("rowspan"));
            NodeFilter tdTagFilter = new TagNameFilter("td");
            for (int i = 0; i < rows.size(); ++i) {
                Node node = rows.elementAt(i);
                NodeList thCells = node.getChildren().extractAllNodesThatMatch(thTagFilter);
                NodeList tdCells = node.getChildren().extractAllNodesThatMatch(tdTagFilter);
                if (tdCells.size() == 8) {
                    if (thCells.size() == 2) {
                        frequency = (int) (Double.parseDouble(htmlCellExtract(thCells, 0, 1, 2).split(" ")[0]) * 1000);
                        polarisation = htmlCellExtract(thCells, 0, 2, 2).charAt(0);
                        coverage = htmlCellExtract(thCells, 1, 0, 0);
                    }
                    String name = htmlCellExtract(tdCells, 1, 0, 3);
                    String category = htmlCellExtract(tdCells, 0, 0, 2);
                    String encryption = htmlCellExtract(tdCells, 2, 1, 3);
                    int symbolRate = 27500;
                    try {
                        symbolRate = (int) (Double.parseDouble(htmlCellExtract(tdCells, 3, 0, 3))) * 1000;
                    } catch (ArrayIndexOutOfBoundsException e) {
                    } catch (NumberFormatException e) {
                    }
                    int errorCorrection = 9;
                    try {
                        errorCorrection = Integer.parseInt((htmlCellExtract(tdCells, 3, 0, 3).substring(0, 1)));
                    } catch (ArrayIndexOutOfBoundsException e) {
                    } catch (StringIndexOutOfBoundsException e) {
                    } catch (NumberFormatException e) {
                    }

                    int sid = 0;
                    try {
                        sid = Integer.parseInt(htmlCellExtract(tdCells, 4, 1, 3));
                    } catch (NumberFormatException e) {
                    } catch (ArrayIndexOutOfBoundsException e) {
                    }
                    String country = htmlCellExtract(tdCells, 5, 0, 3);
                    SatChannel ch = new SatChannel();
                    ch.setFrequency(frequency);
                    ch.setPolarisation(polarisation);
                    ch.setCoverage(coverage);
                    ch.setName(name);
                    ch.setCategory(category);
                    ch.setEncryption(encryption);
                    ch.setSymbolRate(symbolRate);
                    ch.setErrorCorrection(errorCorrection);
                    ch.setSid(sid);
                    ch.setCountry(country);
                    if (tvFilter.contains(ch.getCategory())) {
                        newMedias.add(ch);
                    }
                }
            }
        }
        catch (ParserException e) {
            throw new IOException();
        }

        List<Media> medias = vlmaDao.getMedias();
        for (Media ch : newMedias) {
            logger.debug("Adding " + ch.getName());
            coverages.add(((SatChannel)ch).getCoverage());
            for (Media media : medias) {
                if (ch.equals(media))
                    ch.setProgram(media.getProgram());
            }
        }

        logger.debug("Got " + newMedias.size() + " channels");
        List<Media> mediasToRemove = new ArrayList<Media>();
        for (Media ch : medias) {
            if (ch instanceof SatChannel && coverages.contains(((SatChannel)ch).getCoverage()))
                mediasToRemove.add(ch);
        }
        vlmaDao.removeAll(mediasToRemove);
        vlmaDao.addAll(newMedias);
        logger.info("Ending analysis of " + source.toString());
    }

    public void giveOrders() {
        vlmaService.giveOrders();
    }

    public List<Command> getCommands() {
        return commandLogger.getCommands();
    }

    public void startOrderMonitoring() {
        vlmaService.startOrderMonitoring();
    }

    public void startCheckAllVLCs() {
        vlmaService.startCheckAllVLCs();
    }

    /*************************************************************************
     *
     * Setters
     *
     *************************************************************************\

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    /**
     * Sets the daemon.
     *
     * @param daemon the daemon to set
     */
    public void setDaemon(Daemon daemon) {
        this.daemon = daemon;
    }

    /**
     * Sets the command logger.
     *
     * @param commandLogger the commandLogger to set
     */
    public void setCommandLogger(CommandLogger commandLogger) {
        this.commandLogger = commandLogger;
    }

    /**
     * Sets the IP bank.
     *
     * @param ipBank the ipBank to set
     */
    public void setIpBank(IpBank ipBank) {
        this.ipBank = ipBank;
    }

    /**
     * Sets the program factory
     *
     * @param programFactory the programFactory to set
     */
    public void setProgramFactory(ProgramFactory programFactory) {
        this.programFactory = programFactory;
    }

    /**
     * Sets VLMa DAO
     *
     * @param vlmaDao the vlmaDao to set
     */
    public void setVlmaDao(VLMaDao vlmaDao) {
        this.vlmaDao = vlmaDao;
    }

    /**
     * Sets the VLMa Service
     *
     * @param vlmaService
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

}
