# Pipistrel EPSI 430 EIS by Benedikt Wolf (D-ECHO) based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (it0uchpods)
#######################################

var epsi430_sysoff = nil;
var epsi430_syson = nil;
var epsi430_common = nil;
var epsi430_start = nil;
var epsi430_display = nil;

var state = 0;	# 0 = off, 1 = starting, 2 = on

var base = props.globals.initNode("/instrumentation/epsi430/");

var path = "Aircraft/Taurus/Models/Instruments/epsi430/";

var page_prop = base.initNode("page", "", "STRING");

var start_prop = base.initNode("start", 0.0, "DOUBLE");
var volt_prop = props.globals.getNode( "/systems/electrical/outputs/epsi" );
var brightness_prop = base.initNode("brightness", 0.0, "DOUBLE");
var accel = base.getNode("accel-g", 1 );

var battery_charge = [
	taurus_electro.battery_front.charge_percent,
	taurus_electro.battery_rear.charge_percent,
];
var hv_amps = props.globals.getNode("/systems/electrical/hv-amps", 1);
var hv_volts = props.globals.getNode("/systems/electrical/hv-volts", 1);
var lv_amps = props.globals.getNode("/systems/electrical/lv-amps", 1);
var lv_volts = props.globals.getNode("/systems/electrical/lv-volts", 1);

var battery_volts = props.globals.getNode( "/systems/electrical/hv-volts", 1 );

var solar_power = props.globals.getNode( "/systems/electrical/solar[0]/power-output" );

var motor_temp = props.globals.getNode( "/engines/engine[0]/engine-temp-degc", 1);
var power_controller_temp = props.globals.getNode( "/engines/engine[0]/power-controller-temp-degc", 1);

var shown_message = nil;

var colors = {
	red:		[ 0.9, 0.05, 0.05 ],
	amber:	[ 1.0, 0.5, 0.0 ],
	green:	[ 0.05, 0.9, 0.05 ],
	grey: 	[ 0.5, 0.5, 0.5 ],
	black:	[ 0, 0, 0 ],
	white:	[ 1, 1, 1 ],
	blue: 	[ 0.0, 0.7, 0.94 ],
	lightblue:	[ 0.776, 0.902, 0.918 ],
};

var rpm_p = props.globals.getNode("/engines/engine[0]/rpm", 1);
var power_p = props.globals.getNode("/fdm/jsbsim/propulsion/engine[0]/power-hp", 1);
var charge_avg		=	props.globals.getNode("/systems/electrical/average-charge", 1);

var canvas_epsi430_base = {
	init: func(canvas_group, file) {
		
		canvas.parsesvg(canvas_group, file, {'font-mapper': global.canvas.FontMapper});

		var svg_keys = me.getKeys();
		
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
			var clip_el = canvas_group.getElementById(key ~ "_clip");
			if (clip_el != nil) {
				clip_el.setVisible(0);
				var tran_rect = clip_el.getTransformedBounds();
				var clip_rect = sprintf("rect(%d,%d, %d,%d)", 
				tran_rect[1], # 0 ys
				tran_rect[2], # 1 xe
				tran_rect[3], # 2 ye
				tran_rect[0]); #3 xs
				#   coordinates are top,right,bottom,left (ys, xe, ye, xs) ref: l621 of simgear/canvas/CanvasElement.cxx
				me[key].set("clip", clip_rect);
				me[key].set("clip-frame", canvas.Element.PARENT);
			}
		}

		me.page = canvas_group;

		return me;
	},
	getKeys: func {
		return [];
	},
	update: func {
		var volts = volt_prop.getDoubleValue();
		if ( volts > 10 and state == 2 ) {
			epsi430_start.page.hide();
			if( taurus_electro.engine.sys_en.getBoolValue() ){
				epsi430_syson.page.show();
				epsi430_syson.update();
				epsi430_sysoff.page.hide();
			} else {
				epsi430_sysoff.page.show();
				epsi430_sysoff.update();
				epsi430_syson.page.hide();
			}
			epsi430_common.page.show();
			epsi430_common.update();
		} else if ( volts > 10 and state == 1 ){
			epsi430_syson.page.hide();
			epsi430_sysoff.page.hide();
			epsi430_common.page.hide();
			epsi430_start.page.show();
		} else {
			epsi430_common.page.hide();
			epsi430_syson.page.hide();
			epsi430_sysoff.page.hide();
			epsi430_start.page.hide();
		}
	},
	update_common: func {
		
	},
};

var color_state = {	# 0 = green, 1 = amber, 2 = red
	rpm: -1,
	power: -1,
	motor_temp: 0,
	power_controller_temp: 0,
	batt_temp: [ 0, 0 ],
};

var flag_state = {
	arm: -1, # 0 = down, 1 = up TODO extend
	sysen: -1,	# 0 = off, 1 = on
	proppos: -1,	# 0 = not positioned, 1 = positioned (vertical for retraction)
	heat: -1,	# 0 = off, 1 = on (either battery pack)
};

var canvas_epsi430_common = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_epsi430_common , canvas_epsi430_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ 
			"charge.bar","charge.digits",
			"arm.flag", "sys_status.flag",
			"message_box","message_box.category","message_box.text","message_box.bg",
			"bat_volts", "sys_volts",
			"prop_positioned.flag",
			"battery_heating.flag","battery_balancing.flag",
		];
	},
	update: func() {
		
		var avg_chg = charge_avg.getDoubleValue();
		
		me["charge.digits"].setText( sprintf("%3d%%", avg_chg * 100 ) );
		me["charge.bar"].setTranslation( -( 1 - avg_chg ) * 278, 0 );
		
		me["bat_volts"].setText( sprintf("%3d", math.round( hv_volts.getDoubleValue() ) ) ~ " V" );
		me["sys_volts"].setText( sprintf("%3.1f", lv_volts.getDoubleValue() ) ~ " V" );
		
		if( taurus_electro.batt.heating[0].getBoolValue() or taurus_electro.batt.heating[0].getBoolValue() ){
			if( flag_state.heat != 1 ){
				me["battery_heating.flag"].show();
				flag_state.heat = 1;
			}
		} else {
			if( flag_state.heat != 0 ){
				me["battery_heating.flag"].hide();
				flag_state.heat = 0;
			}
		}
		
		if( taurus_electro.engine.prop_pos.getDoubleValue() > 0.99 ){
			if( flag_state.arm != 1 ){
				me["arm.flag"].setText("Up");
				flag_state.arm = 1;
			}
		} else {
			if( flag_state.arm != 0 ){
				me["arm.flag"].setText("Down");
				flag_state.arm = 0;
			}
		}
		
		if( taurus_electro.engine.sys_en.getBoolValue() ){
			if( flag_state.sysen != 1 ){
				me["sys_status.flag"].setText("On");
				flag_state.sysen = 1;
			}
		} else {
			if( flag_state.sysen != 0 ){
				me["sys_status.flag"].setText("Off");
				flag_state.sysen = 0;
			}
		}
		
		if( taurus_electro.returning == 1 and ( taurus_electro.prop_p.getDoubleValue() == 180 or taurus_electro.prop_p.getDoubleValue() == 360 ) ){
			if( flag_state.proppos != 1 ){
				me["prop_positioned.flag"].show();
				flag_state.proppos = 1;
			}
		} else {
			if( flag_state.proppos != 0 ){
				me["prop_positioned.flag"].hide();
				flag_state.proppos = 0;
			}
		}
	},
	update_message: func( show ) {
		if( !show ){
			me["message_box"].hide();
		} else {
			me["message_box"].show();
			me["message_box.text"].setText( shown_message.message );
			if( shown_message.priority == 0 ){
				me["message_box.category"].setText("WARNING");
				me["message_box.bg"].setColorFill( colors.red );
			} elsif( shown_message.priority == 1 ){
				me["message_box.category"].setText("CAUTION");
				me["message_box.bg"].setColorFill( colors.amber );
			} else {
				me["message_box.category"].setText("INFO");
				me["message_box.bg"].setColorFill( colors.blue );
			}
		}
	},
};

var canvas_epsi430_syson = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_epsi430_syson , canvas_epsi430_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ 
			"rpm.digits", "rpm.arc",
			"power.digits.big", "power.digits.small", "power.arc",
			"motor_temp.digits", "inv_temp.digits","bat_temp.digits",
			"motor_temp.needle", "inv_temp.needle","bat_temp.needle",
		];
	},
	update: func() {
		
		var rpm = rpm_p.getDoubleValue();
		me["rpm.digits"].setText(sprintf("%2d", math.round(rpm/100, 1)));
		
		var rpm_deg = math.clamp( ( 1 - ( rpm / 2450 ) ) * -177, -177, 0 );
		me["rpm.arc"].setRotation( rpm_deg * D2R );
		
		if( rpm >= 2450 ) {
			if( color_state.rpm != 2 ){
				me["rpm.digits"].setColor( colors.red );
				me["rpm.arc"].setColor( colors.red );
				color_state.rpm = 2;
			}
		} elsif( rpm >= 2342 ) {
			if( color_state.rpm != 1 ){
				me["rpm.digits"].setColor( colors.amber );
				me["rpm.arc"].setColor( colors.amber );
				color_state.rpm = 1;
			}
		} else {
			if( color_state.rpm != 0 ){
				me["rpm.digits"].setColor( colors.white );
				me["rpm.arc"].setColor( colors.white );
				color_state.rpm = 0;
			}
		}
		
		var power = math.max( power_p.getDoubleValue(), 0 ) * 0.7457; # convert from hp to kW
		me["power.digits.big"].setText(sprintf("%2d", int(power))~".");
		me["power.digits.small"].setText(sprintf("%1d", math.mod( power, 1)));
		
		var power_deg = math.clamp( ( 1 - ( power/35 ) ) * -175, -175, 0 );
		me["power.arc"].setRotation( power_deg * D2R );
		
		if( power >= 35 ) {
			if( color_state.power != 2 ){
				me["power.digits.big"].setColor( colors.red );
				me["power.digits.small"].setColor( colors.red );
				me["power.arc"].setColor( colors.red );
				color_state.power = 2;
			}
		} elsif( power >= 30 ) {
			if( color_state.power != 1 ){
				me["power.digits.big"].setColor( colors.amber );
				me["power.digits.small"].setColor( colors.amber );
				me["power.arc"].setColor( colors.amber );
				color_state.power = 1;
			}
		} else {
			if( color_state.power != 0 ){
				me["power.digits.big"].setColor( colors.white );
				me["power.digits.small"].setColor( colors.white );
				me["power.arc"].setColor( colors.white );
				color_state.power = 0;
			}
		}
		
		var mot_temp = motor_temp.getDoubleValue();
		me["motor_temp.digits"].setText(sprintf("%3d", math.round( mot_temp ) ) );
		if( mot_temp < 100 ){
			me["motor_temp.needle"].setTranslation( 0, - mot_temp * 0.61 );
		} else {
			me["motor_temp.needle"].setTranslation( 0, - math.min( 94, 61 + ( mot_temp - 100 ) * 1.65 ) );
		}
		
		var inv_temp = power_controller_temp.getDoubleValue();
		me["inv_temp.digits"].setText( sprintf("%2d", math.round( inv_temp ) ) );
		if( inv_temp < 76 ){
			me["inv_temp.needle"].setTranslation( 0, - inv_temp * 0.94737 );
		} else {
			me["inv_temp.needle"].setTranslation( 0, - math.min( 94, 72 + ( inv_temp - 76 ) * 2.445 ) );
		}
		
		var batt_temp_v = nil;
		if( taurus_electro.batt.temp[0].getDoubleValue() > taurus_electro.batt.temp[1].getDoubleValue() ){
			batt_temp_v = taurus_electro.batt.temp[0].getDoubleValue();
		} else {
			batt_temp_v = taurus_electro.batt.temp[1].getDoubleValue();
		}
		me["bat_temp.digits"].setText( sprintf("%2d", math.round( batt_temp_v ) ) );
		if( batt_temp_v <= 0 ){
			me["bat_temp.needle"].setTranslation( 0, - math.max( 0, ( batt_temp_v + 10 ) * 1.2 ) );
		} elsif( batt_temp_v <= 15 ){
			me["bat_temp.needle"].setTranslation( 0, - ( 12 + batt_temp_v * 1.133 ) );
		} elsif( batt_temp_v <= 46 ){
			me["bat_temp.needle"].setTranslation( 0, - ( 29 + ( batt_temp_v - 15 ) * 1.1935 ) );
		} else {
			me["bat_temp.needle"].setTranslation( 0, - math.min( 94, 66 + ( batt_temp_v - 46 ) * 3.11 ) );
		}
	},
};

var canvas_epsi430_sysoff = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_epsi430_sysoff , canvas_epsi430_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ 
			"accel.digits","accel.bar","solar.digits","solar.bar"
		];
	},
	update: func() {
		var acc = accel.getDoubleValue();
		me["accel.digits"].setText( sprintf("%2.1f", acc )~" g" );
		me["accel.bar"].setTranslation( 0, ( 1 - ( math.clamp( acc, 0, 7 ) / 7 ) ) * 99 );
		
		var power_frac = solar_power.getDoubleValue() / 22;
		me["solar.digits"].setText( sprintf("%3d", math.round( power_frac * 100 ) ) ~ " %" );
		me["solar.bar"].setTranslation( 0, ( 1 - math.clamp( power_frac, 0, 1 ) ) * 99 );
	},
};

var canvas_epsi430_start = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_epsi430_start , canvas_epsi430_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
	
};


var base_updater = maketimer( 0.1, func(){ canvas_epsi430_base.update() } );
base_updater.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	epsi430_display = canvas.new({
		"name": "EPSI430",
		"size": [512, 256],
		"view": [512, 256],
		"mipmapping": 1
	});
	epsi430_display.addPlacement({"node": "epsi430.display"});
	var groupSysOn = epsi430_display.createGroup();
	var groupSysOff = epsi430_display.createGroup();
	var groupCommon = epsi430_display.createGroup();
	var groupStart = epsi430_display.createGroup();


	epsi430_syson  = canvas_epsi430_syson.new(  epsi430_display.createGroup(), path~"epsi430-system-on.svg");
	epsi430_sysoff = canvas_epsi430_sysoff.new( epsi430_display.createGroup(), path~"epsi430-system-off.svg");
	epsi430_common = canvas_epsi430_common.new( epsi430_display.createGroup(), path~"epsi430-common.svg");
	epsi430_start  = canvas_epsi430_start.new(  epsi430_display.createGroup(), path~"epsi430-start.svg");
	
	base_updater.start();
	epsi430_common.update_message( 0 );
});


var showEPSI430 = func {
	var dlg = canvas.Window.new([256, 128], "dialog").set("resize", 1);
	dlg.setCanvas(epsi430_display);
}

var power_btn = func (){
	if( state == 0 and volt_prop.getDoubleValue() > 10 ){
		state = 1;
		settimer( func{ state = 2 }, 3 );
	}
}

setlistener(volt_prop, func{
	if( volt_prop.getDoubleValue() < 10 ){
		state = 0;
	}
});

var newAnnunciation = func ( x ){
	shown_message = x;
	epsi430_common.update_message( 1 );
}
