#	DHC-8-400 Audio and Radio Control Display Unit (ARCDU)
#		by Bea Wolf (D-ECHO)
#		based on A3XX Lower ECAM Canvas by Josh Davidson (Octal450)

#	References
#		Q400-Communications.pdf from https://www.smartcockpit.com/my-aircraft/bombardier-dash-8-400/
#		https://www.youtube.com/watch?v=cEqKBtiHj80
#		https://www.youtube.com/watch?v=p2pC81kyxTs


#	TODO-List
#
#		- Volume On/Off Toggle and Indication (green vs white)
#		- Expand/Particular Pages
#		- Memory Channels
#		- Optimize
#		- Special Functions e.g. DME Hold etc.

var arcdu_display = nil;
var instrument_path = "Aircraft/DHC-8/Models/Cockpit/Instruments/ARCDU/";
var base_property = props.globals.initNode("/instrumentation/arcdu[0]");

#	Radios/Modules Encoding
#
#		0	VHF1
#		1	VOR1
#		2	ADF1
#		3	VHF2
#		4	VOR2
#		5	ADF2
#		6	VHF3
#		7	ATC

#	Pages
#		-	Selftest
#		0	MAIN 1
#		1	MAIN 2
#		2-9	Particular Pages for each Radio (VHF1; VHF2; VHF3; NAV/ILS1; NAV/ILS2; ADF1; ADF2; ATC)
#		90	Channels Page

# This variable holds all canvas pages
var pages = {
	main: [ nil, nil ],
	part: [ nil, nil, nil, nil, nil, nil, nil, nil ],
	chan: nil,
	selftest: nil,
	common: nil,
};

var page = 0;	# active page (see labels above)

# 	- Selected Display Field -
# 	0 = none, matrix for 1-8:
#	
#	BTN		   DISPLAY			BTN
#		-	-	-	-	-
#	L1	|	1	|	2	|	R1
#		-	-	-	-	-
#	L2	|	3	|	4	|	R2
#		-	-	-	-	-
#	L3	|	5	|	6	|	R3
#		-	-	-	-	-
#	L4	|	7	|	8	|	R4
#		-	-	-	-	-

var arcdu_controls = {
	brt:		base_property.initNode("brightness", 0.9, "DOUBLE"),
	sel_field:	base_property.initNode("selected-field", 0, "INT"),
	dimbrt:	base_property.initNode("dimbrt-field", 0, "BOOL"), # activated by pressing DIM/BRT switch, de-activated by second press or timeout (5 sec)
};

var color = {
	white:	"#ffffff",
	green:	"#00ff00",
	cyan:		"#00ffff",
	black:	"#000000",
};

var comm_ptt = [
	props.globals.getNode("instrumentation/comm[0]/ptt"),
	props.globals.getNode("instrumentation/comm[1]/ptt"),
	props.globals.getNode("instrumentation/comm[2]/ptt"),
];

var frequencies = {
	comm: [
		{	active:	props.globals.getNode( "instrumentation/comm[0]/frequencies/selected-mhz" ),
			standby:	props.globals.getNode( "instrumentation/comm[0]/frequencies/standby-mhz"  ),
		},
		{	active:	props.globals.getNode( "instrumentation/comm[1]/frequencies/selected-mhz" ),
			standby:	props.globals.getNode( "instrumentation/comm[1]/frequencies/standby-mhz"  ),
		},
		{	active:	props.globals.initNode( "instrumentation/comm[2]/frequencies/selected-mhz", 123.45, "DOUBLE"  ),
			standby:	props.globals.initNode( "instrumentation/comm[2]/frequencies/standby-mhz",  123.45, "DOUBLE"  ),
		},
	],
	nav: [
		{	active:	props.globals.getNode( "instrumentation/nav[0]/frequencies/selected-mhz" ),
			standby:	props.globals.getNode( "instrumentation/nav[0]/frequencies/standby-mhz"  ),
		},
		{	active:	props.globals.getNode( "instrumentation/nav[1]/frequencies/selected-mhz" ),
			standby:	props.globals.getNode( "instrumentation/nav[1]/frequencies/standby-mhz"  ),
		},
	],
	adf: [
		{	active:	props.globals.getNode( "instrumentation/adf[0]/frequencies/selected-khz" ),
			standby:	props.globals.getNode( "instrumentation/adf[0]/frequencies/standby-khz"  ),
		},
		{	active:	props.globals.getNode( "instrumentation/adf[1]/frequencies/selected-khz" ),
			standby:	props.globals.getNode( "instrumentation/adf[1]/frequencies/standby-khz"  ),
		},
	],
};
var volumes = {
	comm: [
		props.globals.getNode( "instrumentation/comm[0]/volume" ),
		props.globals.getNode( "instrumentation/comm[1]/volume" ),
		props.globals.getNode( "instrumentation/comm[2]/volume" ),
	],
	nav: [
		props.globals.initNode( "instrumentation/nav[0]/volume", 0.0, "DOUBLE"),
		props.globals.initNode( "instrumentation/nav[1]/volume", 0.0, "DOUBLE"),
	],
	dme: [
		props.globals.initNode( "instrumentation/dme[0]/volume", 0.0, "DOUBLE"),
		props.globals.initNode( "instrumentation/dme[1]/volume", 0.0, "DOUBLE"),
	],
	adf: [
		props.globals.initNode( "instrumentation/adf[0]/volume", 0.0, "DOUBLE"),
		props.globals.initNode( "instrumentation/adf[1]/volume", 0.0, "DOUBLE"),
	],
	itsv: props.globals.initNode( "instrumentation/arcdu[0]/int-volume",  0.0, "DOUBLE"),
	mkr:	props.globals.initNode( "instrumentation/arcdu[0]/mkr-volume",  0.0, "DOUBLE"),
	spkr:	props.globals.initNode( "instrumentation/arcdu[0]/spkr-volume", 0.0, "DOUBLE"),
};
var memory = { 
	vhf: [ nil, nil, nil ], 
	vor: [ nil, nil ], 
	adf: [ nil, nil ],
};
foreach( var el; keys(memory) ){
	forindex( var x; memory[el] ){
		memory[el][x] = [  ];
		for( var i = 0; i <= 7; i += 1 ){
			append( memory[el][x], base_property.getNode("memory/"~ el ~"["~ x ~"]/channel["~ i ~"]") );
		}
	}
}

var adf = {
	tune_mode: [ 0, 0 ],	# 0 = 0.5 kHz, 1 = 1 kHz
	mode: [
		props.globals.getNode( "instrumentation/adf[0]/mode" ),
		props.globals.getNode( "instrumentation/adf[1]/mode" ),
	],
	adf_ant: [ 0, 0 ],	# 0 = ADF, 1 = ANT
	bfo: [ 0, 0 ],		# 0 = OFF, 1 = ON
};

var nav_loc = [
	props.globals.getNode("instrumentation/nav[0]/nav-loc"),
	props.globals.getNode("instrumentation/nav[1]/nav-loc"),
];

var atc = {
	active_code: [
		props.globals.getNode("instrumentation/transponder[0]/id-code"),
		props.globals.getNode("instrumentation/transponder[1]/id-code"),
	],
	standby_code: [
		props.globals.initNode("instrumentation/transponder[0]/inputs/stby-code", 0, "INT"),
		props.globals.initNode("instrumentation/transponder[1]/inputs/stby-code", 0, "INT"),
	],
	knob_mode: [
		props.globals.getNode("instrumentation/transponder[0]/inputs/knob-mode"),
		props.globals.getNode("instrumentation/transponder[1]/inputs/knob-mode"),
	],
	ident: [
		props.globals.getNode("instrumentation/transponder[0]/ident"),
		props.globals.getNode("instrumentation/transponder[1]/ident"),
	],
	controls: {
		rel_abs: 0,	# 0 = REL, 1 = ABS
		above_below: 0, # -1 = ABOVE, 0 = NORM, 1 = BELOW
		ta_ra: 2, # 0 = STBY, 1 = TA ONLY, 2 = TA/RA
		test: 0, # 0 = OFF, 1 = ON
		alt_src: -1, # Altitude Encoder Source: -1 = ALT1, 0 = OFF, 1 = ALT2
		atc: -1, # Active Transponder: -1 = ATC1, 0 = SBY, 1 = ATC2
	},
};

var volts = props.globals.getNode("/systems/electrical/DC/lessential-bus/volts");	# temp; TODO: find better solution for volt handling: ARCDUs are powered by both the Left Essential as well as the Right Main Bus
var volts_essential_bus_left = props.globals.getNode("/systems/electrical/DC/lessential-bus/volts");
var volts_main_bus_right = props.globals.getNode("/systems/electrical/DC/rmain-bus/volts");

var reset_selected_field_time = -1; # -1 = off, >0: time at which the selected field is reset
var reset_dimbrt_field_time   = -1; # -1 = off, >0: time at which the DIM/BRT field is reset
var reset_atc_ident_time   = -1; # -1 = off, >0: time at which the ATC ident mode is reset
var time = props.globals.getNode("/sim/time/elapsed-sec");

var powered = 0;

var canvas_arcdu_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			return "B612/B612Mono-Regular.ttf";
		};

		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});
		
		var svg_keys = me.getKeys();
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
			var clip_el = canvas_group.getElementById(key ~ "_clip");
			if (clip_el != nil) {
				clip_el.setVisible(0);
				var tran_rect = clip_el.getTransformedBounds();
				var clip_rect = sprintf("rect(%d,%d, %d,%d)", 
				tran_rect[1], # 0 ys
				tran_rect[2], # 1 xe
				tran_rect[3], # 2 ye
				tran_rect[0]); #3 xs
				#   coordinates are top,right,bottom,left (ys, xe, ye, xs) ref: l621 of simgear/canvas/CanvasElement.cxx
				me[key].set("clip", clip_rect);
				me[key].set("clip-frame", canvas.Element.PARENT);
			}
		}
		
		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		powered = ( volts_essential_bus_left.getDoubleValue() > 10 or volts_main_bus_right.getDoubleValue() > 10 );
		if ( powered and getprop("/instrumentation/arcdu/selftest")==0) {
			
			if( page == 0 ){
				pages.main[0].page.show();
				pages.main[0].update();
				
				pages.main[1].page.hide();
				pages.chan.page.hide();
				foreach( var el; pages.part ){
					el.page.hide();
				}
			} elsif( page == 1 ){
				pages.main[1].page.show();
				pages.main[1].update();
				
				pages.main[0].page.hide();
				pages.chan.page.hide();
				foreach( var el; pages.part ){
					el.page.hide();
				}
			} elsif( page >= 2 and page <= 9 ){
				
				forindex( var i; pages.part ){
					if( i != ( page - 2 ) ){
						pages.part[ i ].page.hide();
					} else {
						pages.part[ i ].page.show();
						pages.part[ i ].update();
					}
				}
				
				pages.main[0].page.hide();
				pages.main[1].page.hide();
				pages.chan.page.hide();
			}elsif( page == 90 ){
				pages.chan.page.show();
				pages.chan.update();
				
				pages.main[0].page.hide();
				pages.main[1].page.hide();
				
				foreach( var el; pages.part ){
					el.page.hide();
				}
			} else {
				pages.main[0].page.hide();
				pages.main[1].page.hide();
				pages.chan.page.hide();
				
				foreach( var el; pages.part ){
					el.page.hide();
				}
			}
				
			pages.common.page.show();
			pages.common.update();
			
			if( reset_selected_field_time > 0 and time.getDoubleValue() > reset_selected_field_time ){
				if( page == 0 or page == 1 ){
					arcdu_controls.sel_field.setIntValue( 0 );
					reset_selected_field_time = -1;
					update_selected_field();
				} elsif( page == 90 ){
					pages.chan.selected_channel = -1;
					reset_selected_field_time = -1;
					pages.chan.update_selected_channel();
				}
					
			}
			if( reset_dimbrt_field_time > 0 and time.getDoubleValue() > reset_dimbrt_field_time ){
				arcdu_controls.dimbrt.setIntValue( 0 );
				reset_dimbrt_field_time = -1;
				pages.common.update_dimbrt_field();
			}
			if( reset_atc_ident_time > 0 and time.getDoubleValue() > reset_atc_ident_time ){
				atc.ident[0].setBoolValue( 0 );
				reset_atc_ident_time = -1;
				pages.common.update_atc_id();
			}
			
			pages.selftest.page.hide();
		}else if( powered and getprop("/instrumentation/arcdu/selftest")==1){
			
			pages.selftest.page.show();
			
			pages.main[0].page.hide();
			pages.main[1].page.hide();
			pages.chan.page.hide();
			pages.common.page.hide();
			
			foreach( var el; pages.part ){
				el.page.hide();
			}
		} else {
			pages.main[0].page.hide();
			pages.main[1].page.hide();
			pages.chan.page.hide();
			pages.common.page.hide();
			pages.selftest.page.hide();
			
			foreach( var el; pages.part ){
				el.page.hide();
			}
		}
	},
};

var canvas_arcdu_selftest = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_selftest,canvas_arcdu_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	},
};

var selected_previous = -1;
var nav_loc_previous = [ -1, -1 ];

var canvas_arcdu_main1 = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_main1, canvas_arcdu_base] };
		m.init(canvas_group, file);
		
		m.module_identifier = [ "vhf1", "vhf2", "vor1", "vor2", "adf1", "adf2" ];
		
		m.tune_mode = [ 0, 0, 0, 0, 0, 0 ]; # 0 = Frequency, 1 = Channel
		
		m.act_channel = [ -1, -1, -1, -1, -1, -1 ]; # Tuned from which channel; -1 = none, 0 - 7: CH1 - CH8
		m.sby_channel = [ -1, -1, -1, -1, -1, -1 ]; # Tuned from which channel; -1 = none, 0 - 7: CH1 - CH8

		return m;
	},
	getKeys: func() {
		return [	"vhf1.act", "vhf1.prst", "vhf1.box", "vhf1.vol.bar", "vhf1.stored", "vhf1.tx",
				"vhf2.act", "vhf2.prst", "vhf2.box", "vhf2.vol.bar", "vhf2.stored", "vhf2.tx",
				"vor1.act", "vor1.prst", "vor1.box", "vor1.vol.bar", "vor1.flag", "ils1.flag", "vor1.stored", "vor1.mkr_hi",
				"vor2.act", "vor2.prst", "vor2.box", "vor2.vol.bar", "vor2.flag", "ils2.flag", "vor2.stored", "vor2.mkr_hi",
				"adf1.act", "adf1.prst", "adf1.box", "adf1.vol.bar", "adf1.stored", "adf1.mode",
				"adf2.act", "adf2.prst", "adf2.box", "adf2.vol.bar", "adf2.stored", "adf2.mode",
 			];
	},
	update_selected_field: func() {
		forindex( var i; me.module_identifier ){
			if( ( arcdu_controls.sel_field.getIntValue() - 1 ) != i ){
				me[ me.module_identifier[ i ] ~".box" ].hide();
				me[ me.module_identifier[ i ] ~".prst" ].setColor(color.cyan);
				if( me.tune_mode[ i ] ){
					me.tune_mode[ i ] = 0;
					me.update_tune_mode( i );
				}
			} else {
				me[ me.module_identifier[ i ] ~".box" ].show();
				me[ me.module_identifier[ i ] ~".prst" ].setColor(color.black);
			}
		}
	},
	update_frequency: func( module ){
		var new_freq = [ "", "" ];
		if( module == 0 or module == 1){
			new_freq[ 0 ] = sprintf("%3.3f", frequencies.comm[ module ].active.getDoubleValue() );
			new_freq[ 1 ] = sprintf("%3.3f", frequencies.comm[ module ].standby.getDoubleValue() );
		} elsif( module == 2 or module == 3 ){
			new_freq[ 0 ] = sprintf("%3.3f", frequencies.nav[ module - 2 ].active.getDoubleValue() );
			new_freq[ 1 ] = sprintf("%3.3f", frequencies.nav[ module - 2 ].standby.getDoubleValue() );
			
			if( nav_loc[ module - 2 ].getBoolValue() ){
				me["vor"~ ( module - 1 ) ~".flag"].hide();
				me["ils"~ ( module - 1 ) ~".flag"].show();
			}else{	
				me["vor"~ ( module - 1 ) ~".flag"].show();
				me["ils"~ ( module - 1 ) ~".flag"].hide();
			}
		} else {
			new_freq[ 0 ] = sprintf("%3d", frequencies.adf[ module - 4 ].active.getDoubleValue() );
			new_freq[ 1 ] = sprintf("%3d", frequencies.adf[ module - 4 ].standby.getDoubleValue() );
		}
			
		
		me[ me.module_identifier[ module ]~".act"].setText(  new_freq[0] );
		me[ me.module_identifier[ module ]~".prst"].setText( new_freq[1] );
		
		if( ( arcdu_controls.sel_field.getIntValue() - 1 ) == module and me.tune_mode[ module ] ){
			# Channel Tuning
		} else {
			# Active Channel Indication
			if( me.act_channel[ module ] != -1 ){
				me[ me.module_identifier[ module ]~".stored" ].setColor( color.green );
				me[ me.module_identifier[ module ]~".stored" ].setText("CH"~ ( me.act_channel[ module ] + 1 ) );
				me[ me.module_identifier[ module ]~".stored" ].show();
			} else {
				me[ me.module_identifier[ module ]~".stored" ].hide();
			}
		}
	},
	switch_tune_mode: func( module ){
		if( me.tune_mode[ module ] == 0 ) me.tune_mode[module] = 1;
		else me.tune_mode[ module ] = 0;
		
		me.update_tune_mode( module );
	},
	update_sby_channel: func( module ){
		me[ me.module_identifier[ module ]~".stored" ].setText("CH"~ ( me.sby_channel[ module ] + 1 ) );
	},
	update_tune_mode: func( module ){
		if( me.tune_mode[ module ] == 1 ){
			# Switch to Channel Mode
			me[ me.module_identifier[ module ]~".stored" ].show();
			me[ me.module_identifier[ module ]~".stored" ].setColor( color.cyan );
			if( me.sby_channel[ module ] == -1 ){
				me[ me.module_identifier[ module ]~".stored" ].setText("CH-");
			} else {
				me[ me.module_identifier[ module ]~".stored" ].setText("CH"~ ( me.sby_channel[ module ] + 1 ) );
			}
			
		} else {
			# Switch to Frequency Mode			
			# Check whether the Stored Annunciator needs to be kept for the active frequency
			if( me.act_channel[ module ] != -1 ){
				me[ me.module_identifier[ module ]~".stored" ].setColor( color.green );
				me[ me.module_identifier[ module ]~".stored" ].setText("CH"~ ( me.act_channel[ module ] + 1 ) );
				me[ me.module_identifier[ module ]~".stored" ].show();
			} else {
				me[ me.module_identifier[ module ]~".stored" ].hide();
			}
		}
	},
	update_all_freq: func{
		for( var i = 0; i <= 5; i += 1 ){
			me.update_frequency( i );
		}
	},
	update_mkr_sens: func( x ){
		
		
	},
	update: func() {
		
		# VHF1
		me["vhf1.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.comm[0].getDoubleValue() ) * 64 );
		if( comm_ptt[0].getBoolValue() ){
			me["vhf1.tx"].show();
		} else {
			me["vhf1.tx"].hide();
		}
		
		# VHF2
		me["vhf2.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.comm[1].getDoubleValue() ) * 64 );
		if( comm_ptt[1].getBoolValue() ){
			me["vhf2.tx"].show();
		} else {
			me["vhf2.tx"].hide();
		}
		
		# VOR/ILS 1
		me["vor1.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.nav[0].getDoubleValue() ) * 64 );
		
		# VOR/ILS 2
		me["vor2.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.nav[1].getDoubleValue() ) * 64 );
		
		# ADF 1
		me["adf1.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.adf[0].getDoubleValue() ) * 64 );
		me["adf1.mode"].setText( string.uc( adf.mode[0].getValue() ) );
		
		# ADF 2
		me["adf2.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.adf[1].getDoubleValue() ) * 64 );
		me["adf2.mode"].setText( string.uc( adf.mode[1].getValue() ) );
		
	},
};



var canvas_arcdu_main2 = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_main2, canvas_arcdu_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [	"vhf3.act", "vhf3.prst", "vhf3.box", "vhf3.vol.bar", "vhf3.tx",
				"vhf1.vol.bar", "vhf2.vol.bar",
				"vor1.vol.bar", "vor2.vol.bar",
				"dme1.vol.bar", "dme2.vol.bar",
				"adf1.vol.bar", "adf2.vol.bar",
				"mkr.vol.bar",  "spkr.vol.bar",
 			];
	},
	update_selected_field: func() {
		var field_selected = arcdu_controls.sel_field.getIntValue();
		if( field_selected == 1 ){	# VHF3
			me["vhf3.box"].show();
			me["vhf3.prst"].setColor(color.black);
		} else {	# INT, ATC or none
			me["vhf3.box"].hide();
			me["vhf3.prst"].setColor(color.cyan);
		}
	},
	update: func() {
		
		# VHF3
		me["vhf3.act"].setText(		sprintf("%3.3f", frequencies.comm[2].active.getDoubleValue() ));
		me["vhf3.prst"].setText(	sprintf("%3.3f", frequencies.comm[2].standby.getDoubleValue() ));
		me["vhf3.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.comm[0].getDoubleValue() ) * 64 );
		if( comm_ptt[2].getBoolValue() ){
			me["vhf3.tx"].show();
		} else {
			me["vhf3.tx"].hide();
		}
		
		# Volumes
		me["vhf1.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.comm[0].getDoubleValue() ) * 64 );
		me["vhf2.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.comm[1].getDoubleValue() ) * 64 );
		
		me["vor1.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.nav[0].getDoubleValue() ) * 64 );
		me["vor2.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.nav[1].getDoubleValue() ) * 64 );
		
		me["dme1.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.dme[0].getDoubleValue() ) * 64 );
		me["dme2.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.dme[1].getDoubleValue() ) * 64 );
		
		me["adf1.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.adf[0].getDoubleValue() ) * 64 );
		me["adf2.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.adf[1].getDoubleValue() ) * 64 );
		
		me["mkr.vol.bar"].set(  "coord[1]", 76 +  ( 1.0 - volumes.mkr.getDoubleValue() ) * 64 );
		me["spkr.vol.bar"].set( "coord[1]", 76 +  ( 1.0 - volumes.spkr.getDoubleValue() ) * 64 );
	},
};


var canvas_arcdu_chan = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_chan, canvas_arcdu_base] };
		m.init(canvas_group, file);
		
		m.modules = [ 	[ "vhf1", memory.vhf[0] ],
					[ "vor1", memory.vor[0] ],  
					[ "adf1", memory.adf[0] ], 
					[ "vhf2", memory.vhf[1] ], 
					[ "vor2", memory.vor[1] ], 
					[ "adf2", memory.adf[1] ], ];
					
		m.selected_module = 0;
		m.selected_channel = -1;
		
		return m;
	},
	getKeys: func() {
		return [	"vhf1.label", "vhf1.box", 
				"vhf2.label", "vhf2.box",
				"vor1.label", "vor1.box", 
				"vor2.label", "vor2.box",
				"adf1.label", "adf1.box", 
				"adf2.label", "adf2.box",
				"ch1.freq", "ch1.box",
				"ch2.freq", "ch2.box",
				"ch3.freq", "ch3.box",
				"ch4.freq", "ch4.box",
				"ch5.freq", "ch5.box",
				"ch6.freq", "ch6.box",
				"ch7.freq", "ch7.box",
				"ch8.freq", "ch8.box",
 			];
	},
	update_selected_module: func() {
		
		forindex( var i; me.modules ){
			if( i == me.selected_module ){
				me[ me.modules[i][0] ~".box" ].show();
				me[ me.modules[i][0] ~".label" ].setColor( color.black );
				
				# Load Memory Channels
				for( var x = 0; x <= 7; x += 1 ){
					if( me.modules[i][0] == "adf1" or me.modules[i][0] == "adf2" ){
						me["ch"~ ( x + 1 ) ~".freq"].setText( sprintf("%4.1f", me.modules[i][1][ x ].getValue() ) );
					} else {
						me["ch"~ ( x + 1 ) ~".freq"].setText( sprintf("%6.3f", me.modules[i][1][ x ].getValue() ) );
					}
				}
			} else {
				me[ me.modules[i][0] ~".box" ].hide();
				me[ me.modules[i][0] ~".label" ].setColor( color.cyan );
			}
			
		}
	},
	update_selected_channel: func() {
		for( var i = 0; i <= 7; i += 1 ){
			if( i == me.selected_channel ){
				me["ch"~ ( i + 1 ) ~".box"].show();
				me["ch"~ ( i + 1 ) ~".freq"].setColor( color.black );
			} else {
				me["ch"~ ( i + 1 ) ~".box"].hide();
				me["ch"~ ( i + 1 ) ~".freq"].setColor( color.cyan );
			}
		}
	},
	update_freq: func() {
		for( var x = 0; x <= 7; x += 1 ){
			if( me.modules[ me.selected_module ][0] == "adf1" or me.modules[ me.selected_module ][0] == "adf2" ){
				me["ch"~ ( x + 1 ) ~".freq"].setText( sprintf("%4.1f", me.modules[ me.selected_module ][1][ x ].getValue() ) );
			} else {
				me["ch"~ ( x + 1 ) ~".freq"].setText( sprintf("%6.3f", me.modules[ me.selected_module ][1][ x ].getValue() ) );
			}
		}
	},
	update: func() {
	},
};


var canvas_arcdu_part_vhf = {
	new: func(canvas_group, file, x) {
		var m = { parents: [canvas_arcdu_part_vhf, canvas_arcdu_base] };
		m.init(canvas_group, file);
		m.x = x;

		return m;
	},
	getKeys: func() {
		return [	"module.label", "module.act", "module.prst", "module.box", "module.vol.bar", "module.stored", "module.tx" ];
	},
	update_selected_field: func() {
		var field_selected = arcdu_controls.sel_field.getIntValue();
		if( field_selected == 1 ){
			me["module.box"].show();
			me["module.prst"].setColor(color.black);
		} else {
			me["module.box"].hide();
			me["module.prst"].setColor(color.cyan);
		}
	},
	update_frequency: func(){
		var new_active = frequencies.comm[ me.x ].active.getDoubleValue();
		me["module.act"].setText(		sprintf("%3.3f", new_active ));
		me["module.prst"].setText(	sprintf("%3.3f", frequencies.comm[ me.x ].standby.getDoubleValue() ));
	},
	update: func() {
		if( comm_ptt[ me.x ].getBoolValue() ){
			me["module.tx"].show();
		} else {
			me["module.tx"].hide();
		}
	},
};

var canvas_arcdu_part_vor = {
	new: func(canvas_group, file, x) {
		var m = { parents: [canvas_arcdu_part_vor, canvas_arcdu_base] };
		m.init(canvas_group, file);
		m.x = x;

		return m;
	},
	getKeys: func() {
		return [	"module.label", "module.act", "module.prst", "module.box", "module.vol.bar", "module.stored", "module.mkr_hi",
				"module.mkr_sens.hi", "module.mkr_sens.hi.box",
				"module.mkr_sens.lo", "module.mkr_sens.lo.box",
 			];
	},
	update_selected_field: func() {
		var field_selected = arcdu_controls.sel_field.getIntValue();
		if( field_selected == 1 ){
			me["module.box"].show();
			me["module.prst"].setColor(color.black);
		} else {
			me["module.box"].hide();
			me["module.prst"].setColor(color.cyan);
		}
	},
	update_frequency: func(){
		var new_active = frequencies.comm[ me.x ].active.getDoubleValue();
		me["module.act"].setText(		sprintf("%3.3f", new_active ));
		me["module.prst"].setText(	sprintf("%3.3f", frequencies.comm[ me.x ].standby.getDoubleValue() ));
	},
	update_mkr_sens: func(){
		
	},
	update: func() {
		if( nav_loc[ me.x ].getBoolValue() ){
			me["module.label"].setText( "I\nL\nS\n"~ me.x );
		} else {
			me["module.label"].setText( "V\nO\nR\n"~ me.x );
		}
	},
};

var canvas_arcdu_part_adf = {
	new: func(canvas_group, file, x) {
		var m = { parents: [canvas_arcdu_part_adf, canvas_arcdu_base] };
		m.init(canvas_group, file);
		m.x = x;

		return m;
	},
	getKeys: func() {
		return [	"module.label", "module.act", "module.prst", "module.box", "module.vol.bar", "module.stored", "module.mode",
				"adf", "adf.box",
				"ant", "ant.box",
				"bfo_on", "bfo_on.box",
				"bfo_off", "bfo_off.box",
 			];
	},
	update_selected_field: func() {
		var field_selected = arcdu_controls.sel_field.getIntValue();
		if( field_selected == 1 ){
			me["module.box"].show();
			me["module.prst"].setColor(color.black);
		} else {
			me["module.box"].hide();
			me["module.prst"].setColor(color.cyan);
		}
	},
	update_frequency: func(){
		var new_active = frequencies.comm[ me.x ].active.getDoubleValue();
		me["module.act"].setText(		sprintf("%3.3f", new_active ));
		me["module.prst"].setText(	sprintf("%3.3f", frequencies.comm[ me.x ].standby.getDoubleValue() ));
	},
	update_adf_ant: func(){
		if( adf.adf_ant[ me.x ] == 0 ){
			me["adf"].setColor( color.black );
			me["adf.box"].show();
			me["ant"].setColor( color.white );
			me["ant.box"].hide();
		} else {
			me["adf"].setColor( color.white );
			me["adf.box"].hide();
			me["ant"].setColor( color.black );
			me["ant.box"].show();
		}
	},
	update_bfo: func(){
		if( adf.bfo[ me.x ] ){
			me["bfo_on"].setColor( color.black );
			me["bfo_on.box"].show();
			me["bfo_off"].setColor( color.white );
			me["bfo_off.box"].hide();
		} else {
			me["bfo_on"].setColor( color.white );
			me["bfo_on.box"].hide();
			me["bfo_off"].setColor( color.black );
			me["bfo_off.box"].show();
		}
	},
	update: func() {
		me["module.mode"].setText( string.uc( adf.mode[ me.x ].getValue() ) );
	},
};

var canvas_arcdu_part_atc = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_part_atc, canvas_arcdu_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [	"rel", "rel.box",
				"abs", "abs.box",
				"above", "above.box",
				"norm", "norm.box",
				"below", "below.box",
				"ta_ra", "ta_ra.box",
				"ta_only", "ta_only.box",
				"tcas_stby", "tcas_stby.box",
				"test", "test.box",
				"alt1", "alt1.box",
				"alt_off", "alt_off.box",
				"alt2", "alt2.box",
				"atc1", "atc1.box",
				"atc_sby", "atc_sby.box",
				"atc2", "atc2.box",
 			];
	},
	update: func() {
		# REL/ABS
		if( atc.controls.rel_abs == 0 ){
			me["rel.box"].show();
			me["rel"].setColor( color.black );
			me["abs.box"].hide();
			me["abs"].setColor( color.white );
		} else {
			me["abs.box"].show();
			me["abs"].setColor( color.black );
			me["rel.box"].hide();
			me["rel"].setColor( color.white );
		}
		
		# ABOVE/NORM/BELOW
		if( atc.controls.above_below == -1 ){
			me["above.box"].show();
			me["above"].setColor( color.black );
			me["norm.box"].hide();
			me["norm"].setColor( color.white );
			me["below.box"].hide();
			me["below"].setColor( color.white );
		} elsif( atc.controls.above_below == 0 ){
			me["above.box"].hide();
			me["above"].setColor( color.white );
			me["norm.box"].show();
			me["norm"].setColor( color.black );
			me["below.box"].hide();
			me["below"].setColor( color.white );
		} else {
			me["above.box"].hide();
			me["above"].setColor( color.white );
			me["norm.box"].hide();
			me["norm"].setColor( color.white );
			me["below.box"].show();
			me["below"].setColor( color.black );
		}
		
		# TA/RA / TA ONLY / STBY
		if( atc.controls.ta_ra == 0 ){
			me["tcas_stby.box"].show();
			me["tcas_stby"].setColor( color.black );
			me["ta_only.box"].hide();
			me["ta_only"].setColor( color.white );
			me["ta_ra.box"].hide();
			me["ta_ra"].setColor( color.white );
		} elsif( atc.controls.ta_ra == 0 ){
			me["tcas_stby.box"].hide();
			me["tcas_stby"].setColor( color.white );
			me["ta_only.box"].show();
			me["ta_only"].setColor( color.black );
			me["ta_ra.box"].hide();
			me["ta_ra"].setColor( color.white );
		} else {
			me["tcas_stby.box"].hide();
			me["tcas_stby"].setColor( color.white );
			me["ta_only.box"].hide();
			me["ta_only"].setColor( color.white );
			me["ta_ra.box"].show();
			me["ta_ra"].setColor( color.black );
		}
		
		# TEST
		if( atc.controls.test ){
			me["test.box"].show();
			me["test"].setColor( color.black );
		} else {
			me["test.box"].hide();
			me["test"].setColor( color.white );
		}
		
		# ALT1/OFF/ALT2
		if( atc.controls.alt_src == -1 ){
			me["alt1.box"].show();
			me["alt1"].setColor( color.black );
			me["alt_off.box"].hide();
			me["alt_off"].setColor( color.white );
			me["alt2.box"].hide();
			me["alt2"].setColor( color.white );
		} elsif( atc.controls.alt_src == 0 ){
			me["alt1.box"].hide();
			me["alt1"].setColor( color.white );
			me["alt_off.box"].show();
			me["alt_off"].setColor( color.black );
			me["alt2.box"].hide();
			me["alt2"].setColor( color.white );
		} else {
			me["alt1.box"].hide();
			me["alt1"].setColor( color.white );
			me["alt_off.box"].hide();
			me["alt_off"].setColor( color.white );
			me["alt2.box"].show();
			me["alt2"].setColor( color.black );
		}
		
		# ATC1/SBY/ATC2
		if( atc.controls.atc == -1 ){
			me["atc1.box"].show();
			me["atc1"].setColor( color.black );
			me["atc_sby.box"].hide();
			me["atc_sby"].setColor( color.white );
			me["atc2.box"].hide();
			me["atc2"].setColor( color.white );
		} elsif( atc.controls.atc == 0 ){
			me["atc1.box"].hide();
			me["atc1"].setColor( color.white );
			me["atc_sby.box"].show();
			me["atc_sby"].setColor( color.black );
			me["atc2.box"].hide();
			me["atc2"].setColor( color.white );
		} else {
			me["atc1.box"].hide();
			me["atc1"].setColor( color.white );
			me["atc_sby.box"].hide();
			me["atc_sby"].setColor( color.white );
			me["atc2.box"].show();
			me["atc2"].setColor( color.black );
		}
	},
};

var canvas_arcdu_common = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_arcdu_common, canvas_arcdu_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return ["atc.act", "atc.text", "atc.box", "atc.prst", "atc.id", "int.vol.bar", "int.dimbrt"];
	},
	update_selected_field: func() {
		var field_selected = arcdu_controls.sel_field.getIntValue();
		if( field_selected == 8 ){ # ATC
			me["atc.box"].show();
			me["atc.prst"].setColor(color.black);
		} else {
			me["atc.box"].hide();
			me["atc.prst"].setColor(0,1,1);
		}
	},
	update_dimbrt_field: func() {
		if( arcdu_controls.dimbrt.getBoolValue() ){
			me["int.dimbrt"].show();
		} else {
			me["int.dimbrt"].hide();
		}
	},
	update_atc_id: func() {
		if( atc.ident[0].getBoolValue() ){
			me["atc.id"].show();
		} else {
			me["atc.id"].hide();
		}
	},
	update: func() {
		# INT (Intercom)
		me["int.vol.bar"].set( "coord[1]", 84 +  ( 1.0 - volumes.itsv.getDoubleValue() ) * 64 );
		
		# ATC1 (TRANSPONDER)
		me["atc.act"].setText(sprintf("%04d", atc.active_code[0].getIntValue() ) );
		var transponder_mode = atc.knob_mode[0].getIntValue();
		if(transponder_mode==0){
			me["atc.text"].setText("");
		}else if(transponder_mode==1){
			me["atc.text"].setText("SBY");
		}else if(transponder_mode==4){
			me["atc.text"].setText("ON");
		}else if(transponder_mode==5){
			me["atc.text"].setText("ON ALT");
		}else{
			me["atc.text"].setText("XXXX");
		}
		me["atc.prst"].setText(sprintf("%04d", atc.standby_code[0].getIntValue() ) );
	},
};

var arcdu_update = maketimer( 0.1, func() { canvas_arcdu_base.update() } );
arcdu_update.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	arcdu_display = canvas.new({
		"name": "arcdu",
		"size": [512, 512],
		"view": [512, 512],
		"mipmapping": 1
	});
	arcdu_display.addPlacement({"node": "arcdu1.display"});
	arcdu_display.addPlacement({"node": "arcdu2.display"});
	
	pages.main[0]  = canvas_arcdu_main1.new(    arcdu_display.createGroup(), instrument_path~"main1.svg"    );
	pages.main[1]  = canvas_arcdu_main2.new(    arcdu_display.createGroup(), instrument_path~"main2.svg"    );
	pages.chan     = canvas_arcdu_chan.new(     arcdu_display.createGroup(), instrument_path~"chan.svg"     );
	
	# Particular Pages: VHF1/2/3
	pages.part[0] = canvas_arcdu_part_vhf.new( arcdu_display.createGroup(), instrument_path~"vhf_part.svg", 0 );
	pages.part[3] = canvas_arcdu_part_vhf.new( arcdu_display.createGroup(), instrument_path~"vhf_part.svg", 1 );
	pages.part[6] = canvas_arcdu_part_vhf.new( arcdu_display.createGroup(), instrument_path~"vhf_part.svg", 2 );
	
	# Particular Pages: VOR/ILS 1/2
	pages.part[1] = canvas_arcdu_part_vor.new( arcdu_display.createGroup(), instrument_path~"vor_part.svg", 0 );
	pages.part[4] = canvas_arcdu_part_vor.new( arcdu_display.createGroup(), instrument_path~"vor_part.svg", 1 );
	
	# Particular Pages: ADF 1/2
	pages.part[2] = canvas_arcdu_part_adf.new( arcdu_display.createGroup(), instrument_path~"adf_part.svg", 0 );
	pages.part[5] = canvas_arcdu_part_adf.new( arcdu_display.createGroup(), instrument_path~"adf_part.svg", 1 );
	
	# Particular Page: ATC
	pages.part[7] = canvas_arcdu_part_atc.new( arcdu_display.createGroup(), instrument_path~"atc_part.svg" );
	
	pages.common   = canvas_arcdu_common.new(   arcdu_display.createGroup(), instrument_path~"common.svg"   );
	pages.selftest = canvas_arcdu_selftest.new( arcdu_display.createGroup(), instrument_path~"selftest.svg" );
	
	pages.main[0].update_selected_field();
	pages.main[0].update_all_freq();
	pages.main[1].update_selected_field();
	pages.common.update_dimbrt_field();
	pages.common.update_atc_id();
	pages.common.update_selected_field();
	
	setlistener( frequencies.comm[0].active, func{
		pages.main[0].update_frequency( 0 );
	});
	setlistener( frequencies.comm[0].standby, func{
		pages.main[0].update_frequency( 0 );
	});
	
	setlistener( frequencies.comm[1].active, func{
		pages.main[0].update_frequency( 1 );
	});
	setlistener( frequencies.comm[1].standby, func{
		pages.main[0].update_frequency( 1 );
	});
	
	setlistener( frequencies.nav[0].active, func{
		pages.main[0].update_frequency( 2 );
	});
	setlistener( frequencies.nav[0].standby, func{
		pages.main[0].update_frequency( 2 );
	});
	
	setlistener( frequencies.nav[1].active, func{
		pages.main[0].update_frequency( 3 );
	});
	setlistener( frequencies.nav[1].standby, func{
		pages.main[0].update_frequency( 3 );
	});
	
	setlistener( frequencies.adf[0].active, func{
		pages.main[0].update_frequency( 4 );
	});
	setlistener( frequencies.adf[0].standby, func{
		pages.main[0].update_frequency( 4 );
	});
	
	setlistener( frequencies.adf[1].active, func{
		pages.main[0].update_frequency( 5 );
	});
	setlistener( frequencies.adf[1].standby, func{
		pages.main[0].update_frequency( 5 );
	});

	arcdu_update.start();
});

var update_selected_field = func {
	# Figure out which pages to update
	if( page == 0 ){
		pages.main[0].update_selected_field();
	} elsif( page == 1 ){
		pages.main[1].update_selected_field();
	}
	pages.common.update_selected_field();
}

var set_reset_selected_field_timer = func{
	reset_selected_field_time = time.getDoubleValue() + 5.0;
}

var set_reset_dimbrt_field_timer = func{
	reset_dimbrt_field_time = time.getDoubleValue() + 5.0;
}

var set_reset_atc_ident_timer = func{
	reset_atc_ident_time = time.getDoubleValue() + 17.0;
}

var showarcdu = func {
	var dlg = canvas.Window.new([512, 512], "dialog").set("resize", 1);
	dlg.setCanvas(arcdu_display);
}

setlistener(volts, func (i){
	if( volts.getDoubleValue() == 0 and getprop("/instrumentation/arcdu/was_0")!=1){
		setprop("/instrumentation/arcdu/was_0", 1);
	}
	if( volts.getDoubleValue() != 0 and getprop("/instrumentation/arcdu/was_0")==1){
		setprop("/instrumentation/arcdu/was_0", 0);
		setprop("/instrumentation/arcdu/selftest", 1);
		settimer( func(){ setprop("/instrumentation/arcdu/selftest", 0); }, 5);
	}
});

var transfer_sby_to_act_channel = func( x ){
	if( pages.main[0].tune_mode[x] and pages.main[0].sby_channel[x] != -1 ){
		pages.main[0].act_channel[x] = pages.main[0].sby_channel[x];
	} else {
		pages.main[0].act_channel[x] = -1;
	}
}

# I/O

var side_button_pressed_time = [ -1, -1, -1, -1, -1, -1, -1, -1 ];
var side_button = {
	l1:	func{
		if( page == 90 ){
			var prev = pages.chan.selected_module;
			
			if( prev >= 2 ){
				pages.chan.selected_module = 0;
			} elsif( prev == 0 ){
				pages.chan.selected_module = 1;
			} else {
				pages.chan.selected_module = 2;
			}
			
			pages.chan.update_selected_module();
			
		} elsif( page == 9 ) {
			if( atc.controls.rel_abs == 0 ) atc.controls.rel_abs = 1;
			else atc.controls.rel_abs = 0;
			pages.part[7].update();
		} else {
			if( arcdu_controls.sel_field.getIntValue() != 1 ){ 
				arcdu_controls.sel_field.setIntValue( 1 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 0 );
				props.swapValues( frequencies.comm[0].active, frequencies.comm[0].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
	l2:	func{
		if( page == 90 ){
			if( pages.chan.selected_channel == 0 ){
				pages.chan.selected_channel = 1;
			} else {
				pages.chan.selected_channel = 0;
			}
			
			pages.chan.update_selected_channel();
		} elsif( page == 9 ) {
			atc.controls.above_below += 1;
			if( atc.controls.above_below > 1 ) atc.controls.above_below = -1;
			pages.part[7].update();
		} else {
			if( arcdu_controls.sel_field.getIntValue() != 3 ){ 
				arcdu_controls.sel_field.setIntValue( 3 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 2 );
				props.swapValues( frequencies.nav[0].active, frequencies.nav[0].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
	l3:	func{
		if( page >= 2 and page <= 8 ){
			pages.chan.selected_module = page - 2;
			page = 90; # CHANNELS page
			pages.chan.update_selected_module();
			pages.chan.update_selected_channel();
		} elsif( page == 90 ){
			if( pages.chan.selected_channel == 2 ){
				pages.chan.selected_channel = 3;
			} else {
				pages.chan.selected_channel = 2;
			}
			
			pages.chan.update_selected_channel();
		} elsif( page == 9 ) {
			atc.controls.ta_ra += 1;
			if( atc.controls.ta_ra > 2 ) atc.controls.ta_ra = 0;
			pages.part[7].update();
		} else {
		
			if( arcdu_controls.sel_field.getIntValue() != 5 ){ 
				arcdu_controls.sel_field.setIntValue( 5 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 4 );
				props.swapValues( frequencies.adf[0].active, frequencies.adf[0].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
	l4:	func{
		#update_selected_field();
		#set_reset_selected_field_timer();
	},
	r1:	func{	
		if( page == 90 ){
			var prev = pages.chan.selected_module;
			
			if( prev < 3 or prev == 5 ){
				pages.chan.selected_module = 3;
			} elsif( prev == 3 ){
				pages.chan.selected_module = 4;
			} else {
				pages.chan.selected_module = 5;
			}
			
			pages.chan.update_selected_module();
			
		} elsif( page == 9 ) {
			if( atc.controls.test == 0 ) atc.controls.test = 1;
			else atc.controls.test = 0;
			pages.part[7].update();
		} elsif( page == 4 or page == 7 ){
			var adf_mod = 0;
			if( page == 7 ) adf_mod = 1;
			if( !adf.bfo[ adf_mod ] ){ # Function disabled when BFO Mode is ON
				# Toggle ADF/ANT
				if( adf.adf_ant[ adf_mod ] == 0 ){
					adf.adf_ant[ adf_mod ] = 1;
					adf.mode[ adf_mod ].setValue( "ANT" );
				} else {
					adf.adf_ant[ adf_mod ] = 0;
					adf.mode[ adf_mod ].setValue( "ADF" );
				}
				pages.part[ page - 2 ].update_adf_ant();
			}
		} else {
			if( arcdu_controls.sel_field.getIntValue() != 2 ){ 
				arcdu_controls.sel_field.setIntValue( 2 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 1 );
				props.swapValues( frequencies.comm[1].active, frequencies.comm[1].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
		
	},
	r2:	func{
		if( page == 90 ){
			if( pages.chan.selected_channel == 4 ){
				pages.chan.selected_channel = 5;
			} else {
				pages.chan.selected_channel = 4;
			}
			
			pages.chan.update_selected_channel();
		} elsif( page == 9 ) {
			atc.controls.alt_src += 1;
			if( atc.controls.alt_src > 1 ) atc.controls.alt_src = -1;
			pages.part[7].update();
		} else {
			if( arcdu_controls.sel_field.getIntValue() != 4 ){ 
				arcdu_controls.sel_field.setIntValue( 4 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 3 );
				props.swapValues( frequencies.nav[1].active, frequencies.nav[1].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
	r3:	func{
		if( page == 90 ){
			if( pages.chan.selected_channel == 6 ){
				pages.chan.selected_channel = 7;
			} else {
				pages.chan.selected_channel = 6;
			}
			
			pages.chan.update_selected_channel();
		} elsif( page == 9 ) {
			atc.controls.atc += 1;
			if( atc.controls.atc > 1 ) atc.controls.atc = -1;
			pages.part[7].update();
		} elsif( page == 4 or page == 7 ){
			var adf_mod = 0;
			if( page == 7 ) adf_mod = 1;
			# Toggle BFO OFF/ON
			if( adf.bfo[ adf_mod ] == 0 ){
				adf.bfo[ adf_mod ] = 1;
				adf.adf_ant[ adf_mod ] = 0; # BFO automatically sets ADF mode
				adf.mode[ adf_mod ].setValue( "BFO" );
			} else {
				adf.bfo[ adf_mod ] = 0;
				adf.adf_ant[ adf_mod ] = 0;
				adf.mode[ adf_mod ].setValue( "ADF" );
			}
			pages.part[ page - 2 ].update_bfo();
			pages.part[ page - 2 ].update_adf_ant();
		} else {
			if( arcdu_controls.sel_field.getIntValue() != 6 ){ 
				arcdu_controls.sel_field.setIntValue( 6 );
				set_reset_selected_field_timer();
			} else {
				arcdu_controls.sel_field.setIntValue( 0 );
				transfer_sby_to_act_channel( 5 );
				props.swapValues( frequencies.adf[1].active, frequencies.adf[1].standby );
			}
			
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
	r4:	func( pressed ){
		
		if( pressed ){
			side_button_pressed_time[7] = time.getDoubleValue();
		} else {
			if( ( time.getDoubleValue() - side_button_pressed_time[7] ) > 5 ){
				# Switch Transponder Mode
				if( atc.knob_mode[0].getIntValue() != 5 ){
					atc.knob_mode[0].setIntValue( 5 );	# ON/ALT
				} else {
					atc.knob_mode[0].setIntValue( 1 );	# SBY
				}
			} else {
				# Normal Select Field Action
		
				if( arcdu_controls.sel_field.getIntValue() != 8 ){ 
					arcdu_controls.sel_field.setIntValue( 8 );
					set_reset_selected_field_timer();
				} else {
					arcdu_controls.sel_field.setIntValue( 0 );
					props.swapValues( atc.active_code[0], atc.standby_code[0] );
				}
			}
			update_selected_field();
			set_reset_selected_field_timer();
		}
	},
};
var dimrtn_button = func{
	if( !arcdu_controls.dimbrt.getBoolValue() ){
		arcdu_controls.dimbrt.setBoolValue( 1 );
		pages.common.update_dimbrt_field();
		set_reset_dimbrt_field_timer();
	} else {
		arcdu_controls.dimbrt.setBoolValue( 0 );
		pages.common.update_dimbrt_field();
	}
}
var pg12_button = func{
	if( page == 0 ){
		page = 1;
	} else { # when on MAIN2, CHAN or particular pages, return to MAIN1
		page = 0;
	}
}
var exp_button = func{
	if( page == 0 ){
		if( arcdu_controls.sel_field.getIntValue() == 1 ){
			page = 2;
			pages.part[0].update_frequency();
		} elsif( arcdu_controls.sel_field.getIntValue() == 2 ){
			page = 5;
			pages.part[3].update_frequency();
		} elsif( arcdu_controls.sel_field.getIntValue() == 3 ){
			page = 3;
			pages.part[1].update_frequency();
		} elsif( arcdu_controls.sel_field.getIntValue() == 4 ){
			page = 6;
			pages.part[4].update_frequency();
		} elsif( arcdu_controls.sel_field.getIntValue() == 5 ){
			page = 4;
			pages.part[2].update_frequency();
			pages.part[2].update_adf_ant();
			pages.part[2].update_bfo();
		} elsif( arcdu_controls.sel_field.getIntValue() == 6 ){
			page = 7;
			pages.part[5].update_frequency();
			pages.part[5].update_adf_ant();
			pages.part[5].update_bfo();
		} elsif( arcdu_controls.sel_field.getIntValue() == 8 ){
			page = 9;
			pages.part[7].update();
		}
	} elsif( page == 1 ){
		if( arcdu_controls.sel_field.getIntValue() == 1 ){
			page = 8;
			pages.part[6].update_frequency();
		}
	} elsif( page == 90 ){
		# CHAN Page: Return to the selected radio's particular page
		page = pages.chan.selected_module - 2;
		pages.part[ pages.chan.selected_module ].update_frequency();
	}
}
var ch_button = func{
	# Switch Freq vs Channel Tuning Mode
	if( page == 0 ){
		# MAIN1 Page
		var sel_field = arcdu_controls.sel_field.getIntValue();
		if( sel_field != 0 and sel_field <= 6 ){
			pages.main[0].switch_tune_mode( sel_field - 1 );
		}
	}
	
	set_reset_selected_field_timer();
}
var id_button = func{
	if( powered ){
		atc.ident[0].setBoolValue( 1 );
		pages.common.update_atc_id();
		set_reset_atc_ident_timer();
	}
}
var tune_knob = func( a, x ){
	
	if( powered ){
		# a: 0 = small knob, 1 = big knob
		# x: direction
		
		if( arcdu_controls.dimbrt.getBoolValue() ){
			arcdu_controls.brt.setValue( math.clamp( arcdu_controls.brt.getValue() + x * ( ( a + 1 ) * 0.05 ), 0.0, 1.0) );
			set_reset_dimbrt_field_timer();
		} else {
			if( page == 0 ){
				var sel_field = arcdu_controls.sel_field.getIntValue();
				
				if( sel_field == 8 ){
						# ATC / Transponder
						# Tune Standby Squawk
						# 0000 - 7777
						
						var sq_100 = math.round( math.floor( atc.standby_code[0].getIntValue(), 100.0 ) / 100 );
						var sq_1  = math.round( 100 * math.mod( atc.standby_code[0].getIntValue() / 100, 1.0 ) );
						
						if( a == 0 ){
							# Tune 1
							sq_1 = math.clamp( sq_1 + x, 0, 77 );
						} elsif( a == 1 ){
							# Tune 100
							sq_100 = math.clamp( sq_100 + x, 0, 77 );
						}
						atc.standby_code[0].setIntValue( 100 * sq_100 + sq_1 );
				} else {
				
					if( pages.main[0].tune_mode[ sel_field - 1 ] == 0 ){
						# MAIN1 Page Functions
						if( sel_field == 1 or sel_field == 2 ){
							# VHF1/2
							# Tune Standby Radio
							# min 118.0 max 129.0
							
							var mhz = math.floor( frequencies.comm[ sel_field - 1 ].standby.getValue(), 1.0 );
							var khz = math.mod( frequencies.comm[ sel_field - 1 ].standby.getValue(), 1.0 );
							
							if( a == 0 ){
								# Tune Khz
								khz += x * .025;
								if( khz > 1.0 ){ khz -= 1.0 }
								if( khz < 0.0 ){ khz += 1.0 }
							} elsif( a == 1 ){
								# Tune Mhz
								mhz = math.clamp( mhz + x, 118.0, 129.0 );
							}
							frequencies.comm[ sel_field - 1 ].standby.setValue( mhz + khz );
						}elsif( sel_field == 3 or sel_field == 4 ){
							# VOR/ILS 1/2
							# Tune Standby Radio
							# min 108.0 max 117.0
							
							var mhz = math.floor( frequencies.nav[ sel_field - 3 ].standby.getValue(), 1.0 );
							var khz = math.mod( frequencies.nav[ sel_field - 3 ].standby.getValue(), 1.0 );
							
							if( a == 0 ){
								# Tune Khz
								khz += x * .025;
								if( khz > 1.0 ){ khz -= 1.0 }
								if( khz < 0.0 ){ khz += 1.0 }
							} elsif( a == 1 ){
								# Tune Mhz
								mhz = math.clamp( mhz + x, 108.0, 117.0 );
							}
							frequencies.nav[ sel_field - 3 ].standby.setValue( mhz + khz );
						}elsif( sel_field == 5 or sel_field == 6 ){
							# ADF 1/2
							# Tune Standby Radio
							# min 190 max 1750
							
							var khz_100 = math.round( math.floor( frequencies.adf[ sel_field - 5 ].standby.getValue(), 100.0 ) / 100 );
							var khz_1  = math.round( 100 * math.mod( frequencies.adf[ sel_field - 5 ].standby.getValue() / 100, 1.0 ) );
							
							if( a == 0 ){
								# Tune 1 Khz
								khz_1 += x;
								if( ( 100 * khz_100 + khz_1 ) < 190 ) khz_1 = 90;
								if( ( 100 * khz_100 + khz_1 ) > 1750 ) khz_1 = 50;
							} elsif( a == 1 ){
								# Tune 10 Khz
								khz_100 += x;
								if( ( 100 * khz_100 + khz_1 ) < 190 ){
									khz_100 = 1;
									khz_1 = 90;
								}
								if( ( 100 * khz_100 + khz_1 ) > 1750 ){
									khz_100 = 17;
									khz_1 = 50;
								}
							}
							frequencies.adf[ sel_field - 5 ].standby.setValue( 100 * khz_100 + khz_1 );
						}
					} else {
						# Channel Tune
						
						if( pages.main[0].sby_channel[ sel_field - 1 ] == -1 ){
							if( x == 1 ){
								pages.main[0].sby_channel[ sel_field - 1 ] = 0;
							} else {
								pages.main[0].sby_channel[ sel_field - 1 ] = 7;
							}
						} else {
							var n = pages.main[0].sby_channel[ sel_field - 1 ] + x;
							if( n > 7 ) n = 0;
							if( n < 0 ) n = 7;
							pages.main[0].sby_channel[ sel_field - 1 ] = n;
						}
						
						if( sel_field == 1 or sel_field == 2 ){
							# VHF 1/2
							frequencies.comm[ sel_field - 1 ].standby.setValue( memory.vhf[ sel_field - 1 ][ pages.main[0].sby_channel[ sel_field - 1 ] ].getDoubleValue() );
						} elsif( sel_field == 3 or sel_field == 4 ){
							# VOR/ILS 1/2
							frequencies.nav[ sel_field - 3 ].standby.setValue( memory.vor[ sel_field - 3 ][ pages.main[0].sby_channel[ sel_field - 3 ] ].getDoubleValue() );
						} else {
							# ADF 1/2
							frequencies.adf[ sel_field - 5 ].standby.setValue( memory.adf[ sel_field - 5 ][ pages.main[0].sby_channel[ sel_field - 5 ] ].getDoubleValue() );
						}
						
						pages.main[0].update_sby_channel( sel_field - 1 );
						
					}
				}
			} elsif( page == 1 ){
				# MAIN2 Page Functions
				var sel_field = arcdu_controls.sel_field.getIntValue();
				if( sel_field == 1){
					# VHF3
					# Tune Standby Radio
					# min 118.0 max 129.0
					
					var mhz = math.floor( frequencies.comm[ 2 ].standby.getValue(), 1.0 );
					var khz = math.mod( frequencies.comm[ 2 ].standby.getValue(), 1.0 );
					
					if( a == 0 ){
						# Tune Khz
						khz += x * .025;
						if( khz > 1.0 ){ khz -= 1.0 }
						if( khz < 0.0 ){ khz += 1.0 }
					} elsif( a == 1 ){
						# Tune Mhz
						mhz = math.clamp( mhz + x, 118.0, 129.0 );
					}
					frequencies.comm[ 2 ].standby.setValue( mhz + khz );
				}
			} elsif( page >= 2 and page <= 4 ){
				# VHF1/2/3 Particular Page Functions
				
			} elsif( page == 90 ){
				# Channel Page Functions
				if( pages.chan.selected_channel != -1 ){
					if( pages.chan.selected_module == 0 or pages.chan.selected_module == 3 or pages.chan.selected_module == 6 ){ # VHF 1/2/3
						var vhf_mod = 0;
						if( pages.chan.selected_module == 3 ){
							vhf_mod = 1;
						} elsif( pages.chan.selected_module == 6 ){
							vhf_mod = 2;
						}
						
						var mhz = math.floor( memory.vhf[ vhf_mod ][ pages.chan.selected_channel ].getValue(), 1.0 );
						var khz = math.mod( memory.vhf[ vhf_mod ][ pages.chan.selected_channel ].getValue(), 1.0 );
						if( a == 0 ){
							# Tune Khz
							khz += x * .025;
							if( khz > 1.0 ){ khz -= 1.0 }
							if( khz < 0.0 ){ khz += 1.0 }
						} elsif( a == 1 ){
							# Tune Mhz
							mhz = math.clamp( mhz + x, 118.0, 129.0 );
						}
						memory.vhf[ vhf_mod ][ pages.chan.selected_channel ].setValue( mhz + khz );
						pages.chan.update_freq();
					} elsif( pages.chan.selected_module == 1 or pages.chan.selected_module == 4 ){ # VOR/ILS 1/2
						var vor_mod = 0;
						if( pages.chan.selected_module == 4 ) vor_mod = 1;
						
						var mhz = math.floor( memory.vor[ vor_mod ][ pages.chan.selected_channel ].getValue(), 1.0 );
						var khz = math.mod( memory.vor[ vor_mod ][ pages.chan.selected_channel ].getValue(), 1.0 );
					
						if( a == 0 ){
							# Tune Khz
							khz += x * .025;
							if( khz > 1.0 ){ khz -= 1.0 }
							if( khz < 0.0 ){ khz += 1.0 }
						} elsif( a == 1 ){
							# Tune Mhz
							mhz = math.clamp( mhz + x, 108.0, 117.0 );
						}
						memory.vor[ vor_mod ][ pages.chan.selected_channel ].setValue( mhz + khz );
						pages.chan.update_freq();
					} elsif( pages.chan.selected_module == 2 or pages.chan.selected_module == 5 ){ # ADF 1/2
						var adf_mod = 0;
						if( pages.chan.selected_module == 5 ) adf_mod = 1;
						
						var khz_100 = math.round( math.floor( memory.adf[ adf_mod ][ pages.chan.selected_channel ].getValue(), 100.0 ) / 100 );
						var khz_1  = math.round( 100 * math.mod( memory.adf[ adf_mod ][ pages.chan.selected_channel ].getValue() / 100, 1.0 ) );
						
						if( a == 0 ){
							# Tune 1 Khz
							khz_1 += x;
							if( ( 100 * khz_100 + khz_1 ) < 190 ) khz_1 = 90;
							if( ( 100 * khz_100 + khz_1 ) > 1750 ) khz_1 = 50;
						} elsif( a == 1 ){
							# Tune 10 Khz
							khz_100 += x;
							if( ( 100 * khz_100 + khz_1 ) < 190 ){
								khz_100 = 1;
								khz_1 = 90;
							}
							if( ( 100 * khz_100 + khz_1 ) > 1750 ){
								khz_100 = 17;
								khz_1 = 50;
							}
						}
						memory.adf[ adf_mod ][ pages.chan.selected_channel ].setValue( 100 * khz_100 + khz_1 );
						pages.chan.update_freq();
					}
				}
			}
			
			set_reset_selected_field_timer();
		}
	}
}
