# LX Vario S80 by Bea Wolf (D-ECHO) based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (it0uchpods)

# THANKS TO Colin Geniet (SoundPitchController), original developers of the ilec-sc7 (WooT)

# Information based on manual http://www.lx-avionik.de/wp/download/manuals/LXS80ManualGermanVer0100.pdf
#######################################

## REQUIRES:
##	* additional altimeter ("s80-altimeter") and airspeed indiator ("s80-airspeed-indicator") modules enabled from <sim><instrumentation>
##	* power supply from systems/electrical/outputs/S80
##	(opt) * vario sound set up in the aircraft's sound file (see S80-sound.txt as an example)

#####################
## Version 05/2020 ##
#####################
## Features of this version:
##	* use maketimer, only update visible pages
##	* use props.nas, store in local variables where sensible
##	* store instrument directory in variable
##	* clarity through indentation
##	* use functions for common functionality
##	* clean up listeners

var state = 0;		#	0 = off;
					#	1 = init 1: bootloader and application version (ca. 0.7s)
					#	2 = init 2: "loading" (ca. 0.1s)
					#	3 = init 3: init SD card (ca. 1.0s)
					#	4 = init 4: Initializing, Indicator version, dots (ca. 2.0s)
					#	5 = init 5: Show Serial Number and Sensor Box (ca. 1.0s)
					#	6 = Set ALT/QNH (until center button is pressed)
					#	7 = normal on

var S80_start = nil;
var S80_main = [ nil, nil, nil, nil, nil, nil ];
var S80_qnhalt = nil;
var S80_vario = nil;
var S80_display = nil;

var s80		=	props.globals.initNode("/instrumentation/s80");

var te_rdg	=	s80.initNode("te-reading-mps", 0.0, "DOUBLE");
var te_avg	=	s80.initNode("te-average-mps", 0.0, "DOUBLE");
var volume	=	s80.initNode("volume", 0.5, "DOUBLE");

setprop("instrumentation/s80-altimeter/serviceable", 1 );
setprop("instrumentation/s80-airspeed-indicator/serviceable", 1 );

var alt		=	props.globals.getNode("instrumentation/s80-altimeter/indicated-altitude-ft", 1);
var press_alt	=	props.globals.getNode("instrumentation/s80-altimeter/pressure-alt-ft", 1);
var qnh		=	props.globals.getNode("instrumentation/s80-altimeter/setting-hpa", 1);
var ias		=	props.globals.getNode("instrumentation/s80-airspeed-indicator/indicated-speed-kt", 1);
var tas		=	props.globals.getNode("instrumentation/s80-airspeed-indicator/true-speed-kt", 1);

var g_accel =	props.globals.getNode("accelerations/pilot-g", 1);
var g_max = [ 1.0, 1.0 ];

var oat		=	props.globals.getNode("environment/temperature-degc", 1);

var gps = {
	lat:	props.globals.getNode("position/latitude-string"),
	lon:	props.globals.getNode("position/longitude-string"),
};



var volt_prop	=	props.globals.initNode("/systems/electrical/outputs/S80", 0.0, "DOUBLE");

var mc		=	s80.initNode("mc", 1.5, "DOUBLE");

var needle	=	s80.initNode( "needle-deg", 0.0, "DOUBLE" );

var polar_def = [ -0.000997169190256748, 0.19882060566162, -13.4837656352864 ]; # see src/performance_calc.ods

var mode	=	0;	# 0 = info mode, 1 = flarm mode, 2 = thermal mode, 3 = waypoint mode, 4 = task mode, 5 = setup mode
var mode_flag = [ "Inf", "Flm", "Thm", "Wpt", "Tsk", "" ];



var instrument_dir	=	"Aircraft/ASW28/Models/Instruments/S80/";

var canvas_S80_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			return "LiberationFonts/LiberationSans-Bold.ttf";
		};

		
		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});

		var svg_keys = me.getKeys();
		 
		foreach(var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
		}

		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		var volts = volt_prop.getDoubleValue();
		if( volts > 9 ){
			if( state >= 7 ){
				forindex( var i; S80_main ){
					if( i != mode ){
						S80_main[ i ].page.hide();
					} else {
						S80_main[ i ].page.show();
						S80_main[ i ].update();
					}
				}
				if( mode < 5 ){
					S80_vario.page.show();
					S80_vario.update();
				} else {
					S80_vario.page.hide();
				}
				S80_start.page.hide();
				S80_qnhalt.page.hide();

				check_g_force();
			} elsif( state == 6 ){
				S80_qnhalt.page.show();
				S80_qnhalt.update();
				S80_start.page.hide();
				foreach( var el; S80_main ){
					el.page.hide();
				}
				S80_vario.page.hide();

				check_g_force();
			} elsif( state >= 1 ) {
				S80_start.page.show();
				foreach( var el; S80_main ){
					el.page.hide();
				}
				S80_qnhalt.page.hide();
				S80_vario.page.hide();
			} else {
				foreach( var el; S80_main ){
					el.page.hide();
				}
				S80_start.page.hide();
				S80_qnhalt.page.hide();
				S80_vario.page.hide();
			}
		} else {
			foreach( var el; S80_main ){
				el.page.hide();
			}
			S80_start.page.hide();
			S80_qnhalt.page.hide();
			S80_vario.page.hide();
		}
	},
};

var check_g_force = func{
	var current_val = g_accel.getDoubleValue();

	if( current_val < g_max[0] ) {
		g_max[0] = current_val;
	} elsif( current_val > g_max[1] ) {
		g_max[1] = current_val;
	}
}
	
var canvas_S80_info = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_info , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "gps.text.1", "gps.text.2", "gps.text.3", "gps.text.4", "gps.text.5", "info.text.1", "info.text.2", "info.text.3", "info.text.4", "info.text.5" ];
	},
	update: func() {
		me["gps.text.2"].setText( gps.lat.getValue() );
		me["gps.text.3"].setText( gps.lon.getValue() );

		var pressure_alt = press_alt.getDoubleValue();
		me["info.text.1"].setText( "FLevel: "~ sprintf("%4d", math.round( pressure_alt * FT2M ) )~"m  FL"~ sprintf("%03d", math.round( pressure_alt / 100 ) ) );
		var ind_alt = alt.getDoubleValue();
		me["info.text.2"].setText( "Alt.: "~ sprintf("%4d", math.round( ind_alt * FT2M ) ) ~"m  "~ sprintf("%4d", math.round( ind_alt ) ) ~"ft" );
		me["info.text.3"].setText( "B:"~ sprintf( "%2.1f", volt_prop.getDoubleValue() ) );
		me["info.text.4"].setText( "OAT: "~ sprintf("%2d", math.round( oat.getDoubleValue() ) ) ~"°C" );
		me["info.text.5"].setText( "GFL: "~ sprintf("%3.2f", g_max[0] ) ~" "~ sprintf("%3.2f", g_max[1] ) );
	}

};

var canvas_S80_vario = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_vario , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "vario_needle", "mccready_needle", "red_diamond_needle", "mode_flag" ];
	},
	update: func() {
		#Average climbrate
		var av = te_avg.getDoubleValue();
		me["red_diamond_needle"].setRotation( math.clamp( av, -5, 5 ) * D2R * 24);

		var te = te_rdg.getDoubleValue();
		me["vario_needle"].setRotation( math.clamp( te, -5, 5 ) * D2R * 24);

		#McCready
		me["mccready_needle"].setRotation( math.clamp( mc.getDoubleValue(), -5, 5 ) * D2R * 24 );

		me["mode_flag"].setText( mode_flag[ mode ] );
	}

};
	
var canvas_S80_thermal = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_thermal , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "avg_vario.digits", "netto_vario.digits" ];
	},
	update: func() {
		
		#Altimeter
	#	me["altitude"].setText( sprintf( "%4d", math.round( alt.getDoubleValue() * FT2M ) ) );
		
		
		#Average climbrate
		var av = te_avg.getDoubleValue();
		me["avg_vario.digits"].setText(sprintf("%2.1f", av));

		var te = te_rdg.getDoubleValue();

		# Calculate netto vario from total energy vario
		var x = tas.getDoubleValue() * 1.852; # convert kts to kph
		var netto = te + ( ( polar_def[0] * math.pow( x, 2 ) + polar_def[1] * x + polar_def[0] ) / 3.6 );
		me["netto_vario.digits"].setText( sprintf("%2.1f", netto ) );
		
	}
	
};


var canvas_S80_start = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_start , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "label", "logo"];
	},
	on_state_change: func {
		if( state == 1 ){
			me["logo"].show();
			me["label"].setText("S80\nBootloader v1.6\nApplication v5.1");
		} elsif( state == 2 ){
			me["logo"].hide();
			me["label"].setText("loading");
		} elsif( state == 3 ){
			me["logo"].show();
			me["label"].setText("S80\nInitializing SD Card...");
		} elsif( state == 4 ){
			me["label"].setText("S80\nInitializing...\n\nIndicator: A1.01,B8.00");
		} elsif( state == 5 ){
			me["label"].setText("S80\nInitialized...\nSN: 81A (2022)\nIndicator: A1.01,B8.00\nSensor Box: A1.02,B7.92");
		}
	},
};

var canvas_S80_flarm = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_flarm , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};

var canvas_S80_qnhalt = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_S80_qnhalt , canvas_S80_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return ["elevation_m", "qnh_mbar"];
	},
	update: func() {
		me["elevation_m"].setText( sprintf("%3d", math.round( alt.getDoubleValue() * FT2M ) )~ "m" );
		me["qnh_mbar"].setText( sprintf("%4d",math.round( qnh.getDoubleValue() ) )~ "mbar" );
	}
};

var s80_update = maketimer(0.05, func() { canvas_S80_base.update() } );
s80_update.simulatedTime = 1;


var switch_state = func {
	if( state == 0 ){
		state = 1;
		state_switch_timer.restart( 0.7 );
		S80_start.on_state_change();
	} elsif( state == 1 ){
		state = 2;
		state_switch_timer.restart( 0.2 );
		S80_start.on_state_change();
	} elsif( state == 2 ){
		state = 3;
		state_switch_timer.restart( 1.0 );
		S80_start.on_state_change();
	} elsif( state == 3 ){
		state = 4;
		state_switch_timer.restart( 2.0 );
		S80_start.on_state_change();
	} elsif( state == 4 ){
		state = 5;
		state_switch_timer.restart( 1.0 );
		S80_start.on_state_change();
	} elsif( state == 5 ){
		state = 6;
	}
}
var state_switch_timer = maketimer( 1.0, switch_state );
state_switch_timer.simulatedTime = 1;

var ls = setlistener("sim/signals/fdm-initialized", func {
	S80_display = canvas.new({
		"name": "S80",
		"size": [240, 320],
		"view": [240, 320],
		"mipmapping": 1
	});
	S80_display.addPlacement({"node": "S80.display"});


	S80_main[0] = canvas_S80_info.new(S80_display.createGroup(), instrument_dir~"S80_info.svg");
	S80_main[1] = canvas_S80_flarm.new(S80_display.createGroup(), instrument_dir~"S80_flarm.svg");
	S80_main[2] = canvas_S80_thermal.new(S80_display.createGroup(), instrument_dir~"S80_thermal.svg");
	S80_main[3] = canvas_S80_thermal.new(S80_display.createGroup(), instrument_dir~"S80_thermal.svg");
	S80_main[4] = canvas_S80_thermal.new(S80_display.createGroup(), instrument_dir~"S80_thermal.svg");
	S80_main[5] = canvas_S80_thermal.new(S80_display.createGroup(), instrument_dir~"S80_thermal.svg");
	S80_vario = canvas_S80_vario.new( S80_display.createGroup(), instrument_dir~"S80_vario.svg");
	S80_start = canvas_S80_start.new(S80_display.createGroup(), instrument_dir~"S80_start.svg");
	S80_qnhalt = canvas_S80_qnhalt.new(S80_display.createGroup(), instrument_dir~"S80_set_qnh.svg");
	
	s80_update.start();
	
	removelistener(ls);
});

var check_electric_off = func () {
	if( volt_prop.getDoubleValue() < 9.0 and state != 0 ){
		state = 0;
	}
}


var power_btn = func {
	if( volt_prop.getDoubleValue() >= 9.0 and state == 0 ){
		switch_state();
	} else {
		check_electric_off();
	}
}

var btn = [
	func{
		if( state == 0 ) power_btn();
	},
	func{
		if( state == 0 ){
			power_btn();
		} elsif( state == 6 ){
			state = 7;
		} elsif( state == 7 ){
			mode += 1;
			if( mode > 5 ) mode = 0;
		}
	},
	func{
		if( state == 0 ) power_btn();
	},
];

var upper_knob = [
	func( i ){
	},
	func( i ){
		if( state == 0 ){
			power_btn();
		}
	}
];
var lower_knob = [
	func( i ){
		if( state == 6 ){
			qnh.setDoubleValue( qnh.getDoubleValue() + 0.5 * i );
		}
	},
	func( i ){
		if( state == 0 ){
			power_btn();
		}
	}
];


setlistener(volt_prop, func {
	check_electric_off();
});



#The following code is based on the ILEC SC7 e-vario and computes the different values shown by the display
io.include("Aircraft/Generic/soaring-instrumentation-sdk.nas");

####################################
####	INSTRUMENT SETUP	####
####################################

# Vario sound pitch controller by Colin Geniet (for ASK21), thanks!
#
# var vario_sound = SoundPitchController.new(
#   input: Object connected to the pitch controller input, e.g. a variometer reading.
#   max_pitch: (optional) Maximum sound frequency factor, the output will be
#              in the range [1/max_pitch, max_pitch], default 2.
#   max_input: Value of input for which max_pitch is reached.
#	on_update: (optional) function to call whenever a new output is available

var SoundPitchController = {
	parents: [InstrumentComponent],
	
	new: func(input, max_input, max_pitch = 2, on_update = nil) {
		return {
			parents: [me],
			input: input,
			max_pitch: max_pitch,
			max_input: max_input,
			on_update: on_update,
		};
	},
	
	update: func {
		var input = math.clamp(me.input.output, -me.max_input, me.max_input);
		me.output = math.pow(me.max_pitch, input / me.max_input);
		
		if (me.on_update != nil) me.on_update(me.output);
	},
};

var probe = TotalEnergyProbe.new();

var s80_needle = Dampener.new(
	input: probe,
	dampening: 1.5, 
	on_update: update_prop("/instrumentation/s80/te-reading-mps"));#1.5 is default dampening value according to POH

var averager = Averager.new(
	input: probe,
	buffer_size: 20,
	on_update: update_prop("/instrumentation/s80/te-average-mps")); #20s is default time according to POH
	
var s80_sound = SoundPitchController.new(
	input: s80_needle,
	max_input: 5,
	on_update: update_prop("/instrumentation/s80/sound-pitch"));

# Wrap everything together into an instrument
var fast_instruments = UpdateLoop.new(
	update_period: 0,
	components: [probe, s80_needle, s80_sound],
	enable: 1);

var slow_instruments = UpdateLoop.new(
	update_period: 1,
	components: [averager],
	enable: 1);
	
