##
# Procedural model of an ASK13 electrical system.
#
# Based on C172P electrical system.

# Power Users:
#  Instruments:
#   AR-6201 COMM (front panel)
#   FLARM main unit (rear panel)
#   FLARM display FLARMv3 (front panel)
#   ILEC SC7 e-vario (front panel)

# System:
#  -usually no master switch
#	-> radio uses volume knob
#	-> 1 FLARM on/off switch
#	-> 1 Vario on/off switch

## Initialize properties
# Instruments
var com_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);
var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);
# Switches
var flarm_switch = props.globals.getNode("/instrumentation/FLARM/master-switch", 1);
var vario_switch = props.globals.getNode("/instrumentation/ilec-sc7/master-switch", 1);


var electrical = props.globals.getNode("/systems/electrical", 1);
var outputs = {
    vario:  electrical.initNode("outputs/ilec-sc7", 0.0, "DOUBLE"),
    radio:  electrical.initNode("outputs/comm[0]", 0.0, "DOUBLE"),
    flarm:  electrical.initNode("outputs/flarm", 0.0, "DOUBLE"),
};
var volts = electrical.initNode("volts", 0.0, "DOUBLE");
var amps = electrical.initNode("amps", 0.0, "DOUBLE");
var serviceable = electrical.initNode("serviceable", 1, "BOOL");

var batt_connector = props.globals.getNode("/controls/electric/battery-connector");
var batt_breaker = props.globals.initNode("/controls/circuit-breakers/battery", 1, "BOOL");

var batt_prop	=	electrical.initNode("battery");

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

var ammeter_ave = 0.0;


##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;

		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));

		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

#       Battery used:
#           12V 10Ah Pb
#           Ref.: https://data.accu-24.de/Airbatt/Energiepower/AIRPB/8000204/Datenblatt_8000204_AIRBATT_Energiepower_AIR-PB_12-10_eng.pdf

var battery = BatteryClass.new( 12.0, 0.95, 9.5, 2.5, 0);

var recharge_battery = func {
    # Charge battery to 100 %
    battery.reset_to_full_charge();
}
##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
    new : func {
        var m = {
            parents: [ElectricalSystemUpdater]
        };
        # Request that the update function be called each frame
        m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
        return m;
    },

    enable: func {
        me.loop.reset();
        me.loop.enable();
    },

    disable: func {
        me.loop.disable();
    },

    reset: func {
        # Do nothing
    },

    update: func (dt) {
        update_virtual_bus(dt);
    }
};

##
# Model the system of relays and connections that join the battery,master/alt switches.
#



var update_virtual_bus = func (dt) {
    var load = 0.0;
    var bus_volts = 0.0;
    var power_source = "";

    if ( serviceable.getBoolValue() and batt_breaker.getBoolValue() and batt_connector.getBoolValue() ){
        bus_volts = battery.get_output_volts();
        power_source = battery;
    }

    load += electrical_bus( bus_volts );
    
    # swtich the master breaker off if load is out of limits
    if ( load > 18 ) {
        batt_breaker.setBoolValue( 0 );
    }

    # charge/discharge the battery
    battery.apply_load( load, dt );

    # filter ammeter needle pos
    ammeter_ave = 0.8 * ammeter_ave + 0.2 * (-load);

    # outputs
    amps.setDoubleValue( ammeter_ave );
    volts.setDoubleValue( bus_volts );
}

#Load sources:
#	com:		https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var electrical_bus = func( bv ) {
	var bus_volts = bv;
	var load = 0.0;
        
    # Vario
    if( vario_switch.getBoolValue() and bus_volts > 9.2 ) {
        outputs.vario.setDoubleValue( bus_volts );
        #Energy consumption:	25mA (medium volume) 60mA (max volume) -> guess: at 12V
        #			guess: base consumption 5mA (no volume)
        load += 0.06 / bus_volts;
        if(vario_aud.getIntValue() == 2 or (vario_aud.getIntValue() == 1 and vario_read.getDoubleValue() > 0)){
            load += (vario_vol.getValue()*0.66) / bus_volts;
        }
    }else{
        outputs.vario.setDoubleValue( 0.0 );
    }

    # Radio
    if( bus_volts > 9.0 ){
        outputs.radio.setDoubleValue( bus_volts );
        if( com_ptt.getBoolValue() and com_start.getDoubleValue() >= 0.99 ){
            load += 19.2 / bus_volts;
        }else{
            load += 1.02 * com_start.getDoubleValue() / bus_volts;
        }
    } else {
        outputs.radio.setDoubleValue( 0.0 );
    }

    # FLARM
    if( flarm_switch.getBoolValue() and bus_volts > 9.0 ){
        outputs.flarm.setDoubleValue( bus_volts );
        load += 0.66 / bus_volts; #FLARM
        load += 0.12 / bus_volts; #FLARM display
    } else {
        outputs.flarm.setDoubleValue( 0.0 );
    }

	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
    # checking if battery should be automatically recharged
    if (!getprop("/systems/electrical/save-battery-charge")) {
        battery.reset_to_full_charge();
    };

    system_updater.enable();
    print("Electrical system initialized");
});
