##
# Electric aircraft electrical system model
#

#	Internals
var eng_volts = electrical.initNode( "engine-volts", 0.0, "DOUBLE" );
var eng_limit = electrical.initNode( "engine-limit", 0.0, "DOUBLE" );
var aux_batt_serv = electrical.getNode("components/aux-battery-serviceable", 1);
var batt_fan_serv = props.globals.getNode("/systems/cooling/battery-fan-serviceable", 1);
var lv = {
	serv: electrical.initNode( "lv-serviceable", 1, "BOOL" ),
	volts: electrical.initNode( "lv-volts", 0.0, "DOUBLE" ),
	amps: electrical.initNode(  "lv-amps", 0.0, "DOUBLE" ),
};
var charger = props.globals.initNode( "/controls/electric/charger", 0, "BOOL" );
var hv = {
	serv: electrical.initNode(  "hv-serviceable", 1, "BOOL" ),
	volts: electrical.initNode( "hv-volts", 0.0, "DOUBLE" ),
	amps: electrical.initNode(  "hv-amps", 0.0, "DOUBLE" ),
};
var dcdc_avail = 0;
var dcdc_load = 0;
var freeze = props.globals.getNode( "/sim/freeze/replay-state", 1 );

var power_lever_comm_serv = props.globals.getNode("/engines/engine[0]/components/power-lever-communication-serviceable", 1);
var epsi570_throttle = props.globals.getNode("/controls/engines/engine[0]/epsi570-throttle", 1);

setlistener( engine.thr, func {
	if( power_lever_comm_serv.getBoolValue() ){
		epsi570_throttle.setDoubleValue( engine.thr.getDoubleValue() );
	}
});

# Switches
var switches = {
	master:		props.globals.getNode("/controls/switches/master-bat", 1),
	avionics:	props.globals.getNode("/controls/switches/master-avionics",   1),
	batt_en:	props.globals.getNode("/controls/switches/batt-en",    1),
	pwr_en:		props.globals.getNode("/controls/switches/pwr-en",     1),
	#		Only Alpha Electro
	socket:		props.globals.getNode("/controls/switches/socket",     1),
	light:	{
		landing:	props.globals.getNode("/controls/lighting/landing-lights",	1),
		navac:		props.globals.getNode("/controls/lighting/nav-ac-lights",	1),
		cockpit:	props.globals.getNode("/controls/lighting/cockpit-light",	1),
	},
};

#	Outputs
var otpts = electrical.initNode( "outputs" );
var outputs = {
	trim: otpts.initNode( "trim", 0.0, "DOUBLE"),
	eis:  otpts.initNode( "eis",  0.0, "DOUBLE"),
	socket: otpts.initNode( "socket", 0.0, "DOUBLE"),
	cockpit_light: otpts.initNode( "cockpit-light", 0.0, "DOUBLE"),
	landing_lights: otpts.initNode( "landing-lights", 0.0, "DOUBLE"),
	nav_ac_lights: otpts.initNode( "nav-ac-lights", 0.0, "DOUBLE"),
	nav_lights_norm: otpts.initNode( "nav-lights-norm", 0.0, "DOUBLE"),
	ac_lights_bool: otpts.initNode( "ac-lights-bool", 0.0, "BOOL"),
	comm:		otpts.initNode( "comm[0]",     0.0, "DOUBLE"),
	gps:		otpts.initNode( "gps",         0.0, "DOUBLE"),
	transponder:	otpts.initNode( "transponder", 0.0, "DOUBLE"),
	instruments:	otpts.initNode( "instruments", 0.0, "DOUBLE"),
	batt_fan:	otpts.initNode( "batt-fan",    0.0, "DOUBLE"),
	coolant_pump:	otpts.initNode( "coolant-pump",0.0, "DOUBLE"),
	batt_pump_f:	otpts.initNode( "batt-pump-f", 0.0, "DOUBLE"),
	batt_pump_r:	otpts.initNode( "batt-pump-r", 0.0, "DOUBLE"),
};


setlistener( batt.present[0], func {
	if( batt.present[0].getBoolValue() ){
		batt.mass[0].setDoubleValue( batt_mass );
		batt_present[0] = 1;
	} else {
		batt.mass[0].setDoubleValue( 0.0 );
		batt_present[0] = 0;
	}
});
setlistener( batt.present[1], func {
	if( batt.present[1].getBoolValue() ){
		batt.mass[1].setDoubleValue( batt_mass );
		batt_present[1] = 1;
	} else {
		batt.mass[1].setDoubleValue( 0.0 );
		batt_present[1] = 0;
	}
});

var comm_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);

var cb_path = props.globals.initNode("/controls/circuit-breakers");

var cb = {
	com:		cb_path.initNode("com", 	1, "BOOL"),
	gps:		cb_path.initNode("gps", 	1, "BOOL"),
	xpdr:		cb_path.initNode("xpdr", 	1, "BOOL"),
	instr:		cb_path.initNode("instr", 	1, "BOOL"),
	eis:		cb_path.initNode("eis", 	1, "BOOL"),	# "EPSI" on Velis Electro
	trim:		cb_path.initNode("trim",	1, "BOOL"),
	pump:		cb_path.initNode("pump",	1, "BOOL"),	# "INV PUMP" on Velis Electro
	pwrctrl:	cb_path.initNode("pwrctrl",	1, "BOOL"),
	mainsys:	cb_path.initNode("mainsys",	1, "BOOL"),	# "SYS CTRL" on Velis Electro
	chgr:		cb_path.initNode("chgr",	1, "BOOL"),	# "CHG" on Velis Electro
	batt_f:		cb_path.initNode("batt-f",	1, "BOOL"),	# "BATT FRONT" on Velis Electro
	batt_r:		cb_path.initNode("batt-r",	1, "BOOL"),	# "BATT REAR" on Velis Electro
	#		Only Velis Electro
	batt_fan:	cb_path.initNode("batt-fan",	1, "BOOL"),
	batt_pump_f:	cb_path.initNode("batt-pump-f",	1, "BOOL"),
	batt_pump_r:	cb_path.initNode("batt-pump-r",	1, "BOOL"),
	dcdc_conv:	cb_path.initNode("dcdc-conv",	1, "BOOL"),
	aux_batt:	cb_path.initNode("aux-batt",	1, "BOOL"),
};

# Lights
var anti_coll_lights = aircraft.light.new( "/systems/electrical/outputs/ac-lights-norm", [0.05, 0.8], "/systems/electrical/outputs/ac-lights-bool" );

##
# Battery model class.
#

var BatteryClass = {
	new: func (volts, amps, amp_hrs, name, cb = nil) {
		var obj = { parents : [BatteryClass],
			ideal_volts : volts,
			ideal_amps : amps,
			amp_hours : amp_hrs,
			name : name,
			cb: cb,
			charge_percent : props.globals.initNode("/systems/electrical/battery-charge-percent-"~name, 0.0, "DOUBLE"),
			amps: props.globals.initNode("/systems/electrical/battery-amps-"~name, 0.0, "DOUBLE"),
			volts: props.globals.initNode("/systems/electrical/battery-volts-"~name, 0.0, "DOUBLE"),
			charge_amps : 7.0,
			state: 1,
		};
		return obj;
	},
	apply_load: func (amps, dt) {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		
		me.amps.setDoubleValue( amps );
		
		var old_charge_percent = me.charge_percent.getDoubleValue();
		
		if ( freeze.getBoolValue() ){	return me.amp_hours * old_charge_percent;	}
		
		var amphrs_used = amps * dt / 3600.0;
		var percent_used = amphrs_used / me.amp_hours;
		
		var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));
		
		if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Apply external power to recharge battery!", 10);
		me.charge_percent.setDoubleValue( new_charge_percent );
		return me.amp_hours * new_charge_percent;
	},
	get_output_volts: func {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		var x = 1.0 - me.charge_percent.getDoubleValue();
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		var output_volts = me.ideal_volts * factor;
		me.volts.setDoubleValue( output_volts);
		return output_volts;
	},
	get_output_amps: func {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		var x = 1.0 - me.charge_percent.getDoubleValue();
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},
	reset_to_full_charge: func {
		me.apply_load(-(1.0 - me.charge_percent.getDoubleValue() ) * me.amp_hours, 3600);
	},
};

var battery_aux = BatteryClass.new(13.2, 15, 12.4,"aux",cb.aux_batt); 	# AMM Velis p.24-04: EarthX 680C-12V
									# https://earthxbatteries.com/wp-content/uploads/2012/10/ETX680C-Product-Spec.pdf
									# TODO: ideal amps? (CB rated at 35A, max. continuous for battery 100A, graph shows 6.2, 12.4 and 18.6 A discharge
var battery_front = BatteryClass.new(345, 87.5, 35,"front", cb.batt_f);	# POH Velis p.7-14: PB345V124E-L
									# capacity 32 - 35 Ah, depending on temp
var battery_rear  = BatteryClass.new(345, 87.5, 35,"back",  cb.batt_r);	# POH Velis p.7-14: PB345V124E-L
									# capacity 32 - 35 Ah, depending on temp

var reset_battery_and_circuit_breakers = func {
	# Charge battery to 100 %
	battery_aux.reset_to_full_charge();
	battery_front.reset_to_full_charge();
	battery_rear.reset_to_full_charge();
	
	reset_circuit_breakers();
}

var reset_circuit_breakers = func {
	foreach( var i; keys(cb) ){
		cb[i].setBoolValue( 1 );
	}
}


##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_lv_bus(dt);
		update_hv_bus(dt);
	}
};

var update_lv_bus = func (dt) {
	var load = 0.0;
	var battery_aux_volts = 0.0;
	if ( lv.serv.getBoolValue() and aux_batt_serv.getBoolValue() ) {
		battery_aux_volts = battery_aux.get_output_volts();
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( switches.master.getBoolValue() ) {	# Master switch, connects aux battery to main bus
		bus_volts = battery_aux_volts;
		power_source = "battery";
	}
	if ( switches.batt_en.getBoolValue() and dcdc_avail ){
		bus_volts = 14.0;
		power_source = "dcdc";
	}
	
	load += avionics_bus( bus_volts );
	load += engine_bus( bus_volts );
	
	
	# Trim Power
	if ( cb.trim.getBoolValue() ) {
		outputs.trim.setDoubleValue( bus_volts );
		load += bus_volts / 28;
	} else {
		outputs.trim.setDoubleValue( 0.0 );
	}
	
	# Variant-specific consumers
	if( variant == 0 ){	# Alpha Electro
		# Socket Power
		if ( switches.socket.getBoolValue() ) {
			outputs.socket.setDoubleValue( bus_volts );
			load += bus_volts / 57;
		} else {
			outputs.socket.setDoubleValue( 0.0 );
		}
		
		# Cockpit lights
		if ( switches.light.cockpit.getBoolValue() ) {
			outputs.cockpit_light.setDoubleValue( bus_volts );
			load += bus_volts / 57;
		} else {
			outputs.cockpit_light.setDoubleValue( 0.0 );
		}
		
		# Landing Light Power
		if ( switches.light.landing.getBoolValue() ) {
			outputs.landing_lights.setDoubleValue( bus_volts );
			load += bus_volts / 5;
		} else {
			outputs.landing_lights.setDoubleValue( 0.0 );
		}
		
		
		# Beacon Power
		if ( switches.light.navac.getBoolValue() ) {
			outputs.nav_ac_lights.setDoubleValue( bus_volts );
			if( bus_volts > 10 ) {
				outputs.nav_lights_norm.setDoubleValue( 1.0 );
				outputs.ac_lights_bool.setBoolValue( 1 );
			} else {
				outputs.nav_lights_norm.setDoubleValue( 0.0 );
				outputs.ac_lights_bool.setBoolValue( 0 );
			}
			load += bus_volts / 28;
		} else {
			outputs.nav_ac_lights.setDoubleValue( 0.0 );
			outputs.nav_lights_norm.setDoubleValue( 0.0 );
			outputs.ac_lights_bool.setBoolValue( 0 );
		}
	}
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		bus_volts = 0;
	}
	
	# charge/discharge the battery
	if ( power_source == "battery" ) {
		battery_aux.apply_load( load, dt );
	} elsif ( bus_volts > battery_aux_volts ) {
		battery_aux.apply_load( -battery_aux.charge_amps, dt );
		dcdc_load = load + battery_aux.charge_amps;
	}
	
	# Output properties
	lv.amps.setDoubleValue( load );
	lv.volts.setDoubleValue( bus_volts );
}

var batt_error = [ 0, 0 ];

var update_hv_bus = func (dt) {
	var load = 0.0;
	var battery_front_volts = 0.0;
	var battery_rear_volts = 0.0;
	var external_volts = 0.0;
	
	if( batt.temp[0].getDoubleValue() >= annunciator.limits.battery[1] and !batt_error[0] ){
		batt_error[0] = 1;
		annunciator.triggers[0][0].trigger("OVERTEMPERATURE");
	} elsif( batt.temp[0].getDoubleValue() < annunciator.limits.battery[1] and batt_error[0] ){
		batt_error[0] = 0;
		annunciator.triggers[0][0].reset();
	}
	
	if( batt.temp[1].getDoubleValue() >= annunciator.limits.battery[1] and !batt_error[1] ){
		batt_error[1] = 1;
		annunciator.triggers[0][1].trigger("OVERTEMPERATURE");
	} elsif( batt.temp[1].getDoubleValue() < annunciator.limits.battery[1] and batt_error[1]  ){
		batt_error[1] = 1;
		annunciator.triggers[0][1].reset();
	}
	
	if ( hv.serv.getBoolValue() ) {
		if( !batt_error[0] and batt_present[0] ){
			battery_front_volts = battery_front.get_output_volts();
			annunciator.batt_front_offline = 0;
		} else {
			annunciator.batt_front_offline = 1;
		}
		if( !batt_error[1] > 0 and batt_present[1] ){
			battery_rear_volts = battery_rear.get_output_volts();
			annunciator.batt_rear_offline = 0;
		} else {
			annunciator.batt_rear_offline = 1;
		}
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	
	if( charger.getBoolValue() and batt.temp[0].getDoubleValue() < 45 and batt.temp[1].getDoubleValue() < 45 ){
		external_volts = 230;
	}
	
	if ( switches.batt_en.getBoolValue() ) {
		if(battery_front_volts > 0){
			bus_volts = battery_front_volts;
			power_source = "battery_front";
		}
		if(battery_rear_volts > 0){
			if(bus_volts > 0){
				bus_volts += battery_rear_volts;
				power_source = "both_battery_packs";
				batt.status[1].setIntValue( 2 );
			}else{
				bus_volts = battery_rear_volts;
				power_source = "battery_rear";
				batt.status[1].setIntValue( 2 );
			}
		} else {
			batt.status[1].setIntValue( 1 );
		}
	}
	if ( external_volts > bus_volts ) {
		bus_volts = external_volts;
		power_source = "external";
	};
		
	if ( power_source == "both_battery_packs" ) {
		eng_limit.setDoubleValue( 1.0 );
	} else {
		eng_limit.setDoubleValue( 0.5833 );
	}
	
	#	The Power Controller H300C converts HV DC power to AC power for the engine
	if( cb.pwrctrl.getDoubleValue() and switches.pwr_en.getBoolValue() and bus_volts > 0 ){
		
		eng_volts.setDoubleValue( bus_volts );
		
		var eng_power = engine.power_hp.getDoubleValue() * 745.7 * 1.1236 * 0.3; #hp to watts * ( 1 / motor efficiency) * calibration factor to achieve 1h endurance at 20kW (useful capacity 20kWh)
		
		if( engine.rpm.getDoubleValue() > 50 ){
			load += eng_power / bus_volts;   #P=U*I -> I=P/U
		}
	} else {
		eng_volts.setDoubleValue( 0.0 );
	}
	
	#	The DC/DC converter reduces voltage to 14V to power the 14V buses and recharge the auxiliary battery
	if( ( cb.dcdc_conv.getBoolValue() or variant == 0 ) and bus_volts > 250 ){
		dcdc_avail = 1;
		load += dcdc_load * 14 / bus_volts;
	} else {
		dcdc_avail = 0;
	}
	
	# swtich the master breaker off if load is out of limits
	if ( load > 150 ) {
		bus_volts = 0;
	}
	
	if ( power_source == "both_battery_packs" ){
		batt.status[0].setIntValue( 2 );
		batt.status[1].setIntValue( 2 );
	} elsif( power_source == "battery_front" ){
		batt.status[0].setIntValue( 2 );
		if( batt_error[1] ){
			batt.status[1].setIntValue( 0 );
		} else {
			batt.status[1].setIntValue( 1 );
		}
	} elsif( power_source == "battery_rear" ){
		if( batt_error[0] ){
			batt.status[0].setIntValue( 0 );
		} else {
			batt.status[0].setIntValue( 1 );
		}
		batt.status[1].setIntValue( 2 );
	} else {
		if( batt_error[0] ){
			batt.status[0].setIntValue( 0 );
		} else {
			batt.status[0].setIntValue( 1 );
		}
		if( batt_error[1] ){
			batt.status[1].setIntValue( 0 );
		} else {
			batt.status[1].setIntValue( 1 );
		}
	}
		
	
	
	# charge/discharge the battery
	if ( power_source == "battery_front" ) {
		battery_front.apply_load( load, dt );
	}else if ( power_source == "battery_rear" ) {
		battery_rear.apply_load( load, dt );
	}else if (power_source == "both_battery_packs" ) {
		battery_front.apply_load( 0.5*load, dt );
		battery_rear.apply_load( 0.5*load, dt );
	}
	
	# outputs
	hv.amps.setDoubleValue( load );
	hv.volts.setDoubleValue( bus_volts );
}

#	Engine Bus: part of the 14V low voltage systems
#
#	Consumers:
#		BATT FRONT	?? does the management system consume power or does this mean that the batts are connected to the LV system?
#		BATT REAR	?? does the management system consume power or does this mean that the batts are connected to the LV system?
#		BATT FAN	implemented
#		BATT PUMP F	implemented
#		BATT PUMP R	implemented
#		PWR CTRL
#		INV PUMP	implemented
#		CHG	?? does the management system consume power or does this mean that the charger port is connected to the LV system?
#		EPSI		implemented
var engine_bus = func( bv ) {
	var bus_volts = bv;
	var load = 0.0;	
	
	if( variant == 1 and cb.batt_fan.getBoolValue() and batt_fan_serv.getBoolValue() and charger.getBoolValue() and ( batt.temp[0].getDoubleValue() > 35 or batt.temp[1].getDoubleValue() > 35 )  ){
		outputs.batt_fan.setDoubleValue( bus_volts );
	} else {
		outputs.batt_fan.setDoubleValue( 0.0 );
	}
	
	
	# Engine Information System Power
	if ( cb.eis.getBoolValue() ) {
		outputs.eis.setDoubleValue( bus_volts );
	} else {
		outputs.eis.setDoubleValue( 0.0);
	}
	
	# Engine Coolant Pump (INV PUMP)
	if( cb.pump.getBoolValue() ){
		outputs.coolant_pump.setDoubleValue( bus_volts );
	} else {
		outputs.coolant_pump.setDoubleValue( 0.0 );
	}
	
	# Battery Coolant Pumps (Velis Electro only) (BATT PUMP F/R)
	if( variant and cb.batt_pump_f.getBoolValue() ){
		outputs.batt_pump_f.setDoubleValue( bus_volts );
	} else {
		outputs.batt_pump_f.setDoubleValue( 0.0 );
	}
	if( variant and cb.batt_pump_r.getBoolValue() ){
		outputs.batt_pump_r.setDoubleValue( bus_volts );
	} else {
		outputs.batt_pump_r.setDoubleValue( 0.0 );
	}
	
	
	return load;
}

var avionics_bus = func( bv ) {
	var bus_volts = 0.0;
	var load = 0.0;
	
	# we are fed from the electrical bus 1
	if ( switches.avionics.getBoolValue() ) {
		bus_volts = bv;
	}
	
	#	COM: ATR833-II-OLED
	#		Ref.: https://www.funkeavionics.de/produkt/atr833-ii-oled/
	#			receiving: 220/120 mA (assume: first number for 11V, second for 30V)
	#			transmitting: 1,3/0,65 A (assume: first number for 11V, second for 30V)
	if ( cb.com.getBoolValue() ) {
		outputs.comm.setDoubleValue( bus_volts );
		if( atr833_ii.state > 0 ){
			if( comm_ptt.getBoolValue() ){
				load += 14.3 / bus_volts;	# 1.3A * 11V
			} else {
				load += 2.42 / bus_volts	# 0.22A * 11V
			}
		}
	} else {
		outputs.comm.setDoubleValue( 0.0 );
	}
	
	#	GPS: Garmin Aera 660
	#		Ref.: https://static.garmin.com/pumac/190-02017-20_h.pdf
	#		TODO: implement its own electrical system (has an integrated battery)
	#			no power usage data can be found, assume: 0.3A at 11V
	if ( cb.gps.getBoolValue() ) {
		outputs.gps.setDoubleValue( bus_volts );
		if( canvas_aera660.state > 0 ){
			load += 3.3 / bus_volts;
		}
	} else {
		outputs.gps.setDoubleValue( 0.0 );
	}
	
	#	XPDR: TRT800H OLED
	#		Ref.: https://www.funkeavionics.de/produkt/trt800h-oled/
	#			mean amperage: < 300 mA at 13.8V, assume: 250 mA
	if ( cb.xpdr.getBoolValue() ) {
		outputs.transponder.setDoubleValue( bus_volts );
		if( trt800h.state > 0 ){
			load += 3.45 / bus_volts;
		}
	} else {
		outputs.transponder.setDoubleValue( 0.0 );
	}
	
	# Other Instruments Power
	if ( cb.instr.getBoolValue() ) {
		outputs.instruments.setDoubleValue( bus_volts );
	} else {
		outputs.instruments.setDoubleValue( 0.0 );
	}
	
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

var el_ls = setlistener("/sim/signals/fdm-initialized", func {
	# Check variant
	if( getprop("/sim/aero") == "alphaelectro-jsb" ){
		variant = 0;
	} else if ( getprop("/sim/aero") == "veliselectro-jsb" ){
		variant = 1;
	} else {
		die( "Electrical system: FATAL: Unknown variant!: "~ getprop("/sim/aero") );
	}
	
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery_aux.reset_to_full_charge();
		battery_front.reset_to_full_charge();
		battery_rear.reset_to_full_charge();
	};
	reset_circuit_breakers();
	
	system_updater.enable();
	
	removelistener( el_ls );
	print("Electrical system initialized");
});

##
# Extra functions for the "seen zero" safety feature
#
var throttle_listener = nil;
var throttle_en = props.globals.getNode("fdm/jsbsim/fcs/epsi570/enable-throttle", 1);
var throttle_zero = props.globals.getNode("/fdm/jsbsim/fcs/epsi570/throttle-at-zero");

var enable_throttle = func () {
	if(throttle_zero.getValue() == 1){
		throttle_en.setValue( 1 * switches.pwr_en.getBoolValue() );
		if( throttle_listener != nil ){
			removelistener(throttle_listener);
			throttle_listener = nil;
		}
	}
}
var disable_throttle = func () {
	throttle_en.setValue( 0 );
}

var power_enable_button = func( i )  {
	if( i == 1 ){
		throttle_listener = setlistener(throttle_zero, func{ if( throttle_zero.getValue() == 1 ) { enable_throttle(); } } );
		startup_selftest();
	} else {
		disable_throttle();
	}
}

setlistener(switches.pwr_en, func (i) {
	power_enable_button(i.getValue());
});
