#!BPY

"""
Name: 'Cloud Generator'
Blender: 248
Group: 'Wizards'
Tooltip: 'Generate billboard clouds'
"""

__author__ = 'Alan Dennis'
__version__ = '0.11'
__email__ = "alan@curiousexistence.com"
__bpydoc__ = """\
This script generates interactive volumetric clouds.
"""

# ***** BEGIN GPL LICENSE BLOCK *****
#
# Script copyright (C): Alan Dennis
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# ***** END GPL LICENCE BLOCK *****
# --------------------------------------------------------------------------


import Blender, time
from Blender import Constraint, Material, Mathutils, Mesh, Noise, Object, Scene, Texture, Window
from Blender.Noise import random
from Blender.Mathutils import Vector
from Blender.BGL import *
from Blender.Draw import *
from Blender.Noise import *

def draw():
	global imgHeader, tglBounds, tglShape, tglBillboards	#Generic
	global btnAssignBounds, txtBounds						#Bounds
	global texPreview, mnuBasis, sldThreshold, sldContrast,	sldBrightness, sldSize, sldJitter, sldResolution, btnPoints, txtCloudName	#Shape
	global sldBillboardSize, txtCamera, btnBillboards		#Billboards
	global strLine1, strLine2, strLine3, strLine4, strBounds1, strBounds2, strBounds3, strBounds4, strPoints1, strPoints2, strPoints3, strPoints4, strBillboard1, strBillboard2, strBillboard3, strBillboard4 #Helptext	
	
	##################
	#Generic
	##################
	glClearColor(0.753, 0.753, 0.753, 0.0) #Background color
	glClear(GL_COLOR_BUFFER_BIT)
	glColor3f(0.000, 0.000, 0.000) #Seperator line
	glRecti(0, 238, 279, 239)
	glColor3f(0.502, 0.502, 0.502) #Helptext Background
	glRecti(0, 238, 279, 293)

	#Header
	DrawImg(imgHeaderBuf, imgHeaderWidth, imgHeaderHeight, imgHeaderLeft, imgHeaderTop)
	#Borders
	glColor3f(0.000,0.000,0.000)
	glRecti(280,0,281,347)
	glRecti(0,0,280,1)
	glRecti(0,347,281,348)
	
	#Toggles
	tglBounds = Toggle('Bounds', 1, 0, 294, 92, 23, tglBounds.val, '')
	tglShape = Toggle('Shape', 2, 94, 294, 92, 23, tglShape.val, '')
	tglBillboards = Toggle('Billboards', 3, 188, 294, 92, 23, tglBillboards.val, '')
	#Helptext
	glColor3f(0.000, 0.000, 0.000)
	glRasterPos2i(2, 283)
	Text(strLine1)
	glRasterPos2i(2, 270)
	Text(strLine2)
	glRasterPos2i(2, 257)
	Text(strLine3)
	glRasterPos2i(2, 244)
	Text(strLine4)
	
	##################
	#Bounds
	##################
	if tglBounds.val:
		glRasterPos2i(0, 220)
		Text('Bounding Object:')
		txtBounds = String('', 101, 136, 214, 143, 23, txtBounds.val, 64, '')
		Button('Assign Bounds', 102, 88, 174, 111, 31, '')
	
	##################
	#Shape
	##################
	elif tglShape.val:
		glRasterPos2i(0, 222)
		Text('Cloud\'s Name:')
		txtCloudName = String('', 201, 120, 214, 159, 23, txtCloudName.val, 64, '')
		sldResolution = Slider('Points / Blender Unit', 202, 0, 190, 279, 23, sldResolution.val, 0.1, 10, 0, '')
		sldJitter = Slider('Jitter', 203, 0, 166, 279, 23, sldJitter.val, 0.0, 1.0, 0, '')
		sldSize = Slider('Noise Size', 204, 0, 142, 279, 23, sldSize.val, 0.01, 10.0, 0, '')
		sldBrightness = Slider('Brightness', 205, 0, 118, 167, 23, sldBrightness.val, 0.0, 5.0, 1, '')
		sldContrast = Slider('Contrast', 206, 0, 94, 167, 23, sldContrast.val, 0.0, 5.0, 1, '')
		sldThreshold = Slider('Threshold', 207, 0, 70, 167, 23, sldThreshold.val, 0.0, 1.0, 1, '')
		mnuBasis = Menu('Noise Basis%t|Blender %x1|Perlin %x2|Improved Perlin %x3|Voronoi F1 %x4|Voronoi F2 %x5|Voronoi F3 %x6|Voronoi F4 %x7|Voronoi F2-F1 %x8|Voronoi Crackle %x9|Cellnoise %x10', 208, 0, 46, 167, 23, mnuBasis.val, '')	
		#Noise Preview
		for x in range(168, 279):
			for y in range(46, 141):
				sample = Vector(x/10.0, y/ 10.0, 0)
				value = texPreview.evaluate(sample)[3]
				if value > sldThreshold.val:					
					glColor3f(value, value, value)
				else:
					glColor3f(0.0,0.0,0.0)
				glRecti(x, y, x + 1, y + 1)
				
		Button('Create Points', 209, 88, 6, 111, 31, '')
	
	##################
	#Billboards
	##################
	else:
		glRasterPos2i(0, 222)
		Text('Camera Object to track:')
		txtCamera = String('', 301, 136, 214, 143, 23, txtCamera.val, 64, '')
		sldBillboardSize = Slider('Billboard Size', 302, 0, 190, 279, 23, sldBillboardSize.val, 0.1, 5.0, 0, '')
		Button('Create Billboards', 303, 88, 150, 111, 31, '')
	

def imgldr(data, width, height):
	imgbuf = Buffer(GL_BYTE, [width, height, 4])
	i = 0
	for row in imgbuf.list:
		for pix in row:
			pix[0] = data[i][0]
			pix[1] = data[i][1]
			pix[2] = data[i][2]
			i += 1
	return imgbuf

def DrawImg(buf, imgw, imgh, px, py):
	glRasterPos2f(px, py)
	glDrawPixels(imgw, imgh, GL_RGBA, GL_UNSIGNED_BYTE, buf)

def event(evt, val):
	if (evt== QKEY and not val): Exit()
def bevent(evt):
	global txtBounds
	global strLine1, strLine2, strLine3, strLine4
	
	##################
	#Toggles
	##################
	if evt == 1: #tglBounds
		tglShape.val = 0
		tglBillboards.val = 0
		strLine1 = strBounds1
		strLine2 = strBounds2
		strLine3 = strBounds3
		strLine4 = strBounds4
		
	elif evt == 2: #tglShape
		tglBounds.val = 0
		tglBillboards.val = 0
		strLine1 = strShape1
		strLine2 = strShape2
		strLine3 = strShape3
		strLine4 = strShape4
		
	elif evt == 3: #tglBillboards
		tglBounds.val = 0
		tglShape.val = 0
		strLine1 = strBillboard1
		strLine2 = strBillboard2
		strLine3 = strBillboard3
		strLine4 = strBillboard4
		
	##################
	#Bounds
	##################
	elif evt == 101: #txtBounds
		if not objectExists(txtBounds.val): txtBounds.val = ''
				
	elif evt == 102: #btnAssignBounds
		checkBounds()
	
	##################
	#Shape
	##################
	elif evt == 205:#sldBrightness
		texPreview.brightness = sldBrightness.val
		
	elif evt == 206: # sldContrast
		texPreview.contrast = sldContrast.val	
		
	elif evt == 208: #mnuBasis		
		texPreview.noiseBasis = getNoiseBasis(mnuBasis.val)
		
	elif evt == 209: #btnPoints
		checkBounds()
		createPoints(Object.Get(txtBounds.val), 1.0 / sldResolution.val, texPreview, txtCloudName.val, sldJitter.val, sldSize.val, sldThreshold.val)
	
	##################
	#Billboards
	##################
	elif evt == 301: #txtCamera
		if not objectExists(txtCamera.val):
			#Could not find the Camera Object
			txtCamera.val = ''
	elif evt == 303: #btnBillboards
		if  txtCamera.val == '':
			#Create a new Camera Object
			txtCamera.val = createCamera()
			
		elif not objectExists(txtCamera.val):
			#Could not find the Camera Object (Shouldn't happen unless the user deletes the object after specifying it)
			txtCamera.val = ''
			print 'Cannot find the object (or camera) "' + txtCamera.val + '".\nTracking will not work without an object to track.'
		
		checkBounds()		
		createEmpties(txtCloudName.val, txtBounds.val)
		mat = createMaterials(sldBrightness.val, sldContrast.val, sldSize.val, getNoiseBasis(mnuBasis.val), txtCloudName.val, mtTranslate, mtDarkening)
		createPlanes(Object.Get(txtCloudName.val + 'Preview'), sldBillboardSize.val, Object.Get(txtCamera.val), mat, txtCloudName.val, txtBounds.val)

	Blender.Redraw()
	
def objectExists(objName):
	try:
		obj = Object.Get(objName)
		return True
	except:
		return False

def getNoiseBasis(selValue):
	if selValue == 1:
		return Texture.Noise.BLENDER
	elif selValue == 2:
		return Texture.Noise.PERLIN
	elif selValue == 3:
	    return Texture.Noise.IMPROVEDPERLIN
	elif selValue == 4:
	    return Texture.Noise.VORONOIF1
	elif selValue == 5:
		return Texture.Noise.VORONOIF2
	elif selValue == 6:
		return Texture.Noise.VORONOIF3
	elif selValue == 7:
		return Texture.Noise.VORONOIF4
	elif selValue == 8:
		return Texture.Noise.VORONOIF2F1
	elif selValue == 9:
		return Texture.Noise.VORONOICRACKLE
	else:
		return Texture.Noise.CELLNOISE

def checkBounds():
	global txtBounds
	if  txtBounds.val == '':
		#Create a new Bounding Object
		txtBounds.val = createBoundingObject()
		
	elif not objectExists(txtBounds.val):
		#Could not find the Bounding Object (Shouldn't happen unless the user deletes the object after specifying it)
		txtBounds.val = ''
		
def createBoundingObject():
	scn = Scene.GetCurrent()
	me = Mesh.Primitives.Cube(2.0)	
	ob = scn.objects.new(me,'Bounds')        # add a new mesh-type object to the scene
	ob.setDrawType(1)
	return ob.getName()
	
def getMaxBounds(boundsName):
	boundsObj = Object.Get(boundsName).boundingBox
	bounds = [[boundsObj[0].x,boundsObj[0].y,boundsObj[0].z],[boundsObj[0].x,boundsObj[0].y,boundsObj[0].z]]
		
	for i in range(1,8):
		for n in range(3):
			if bounds[0][n] > boundsObj[i][n]: bounds[0][n] = boundsObj[i][n]		
			if bounds[1][n] < boundsObj[i][n]: bounds[1][n] = boundsObj[i][n]
		
	return bounds

def createCamera():
	Camera = Blender.Camera.New('persp','Camera')
	Camera = scn.objects.new(Camera)
	Camera.setLocation(0,-7,0)
	Camera.setEuler(1.57,0,0)	
	Blender.Redraw()
	return Camera.getName()
	
def createPoints(boundingObject, resolution, tex, baseName, jitter, noiseSize, threshold):
	scn = Scene.GetCurrent()
	scale = 1000.0
	point = [0.0,0.0,0.0]
	coords = []
	
	meshExists = False
	meshName = baseName + 'Preview'
	
	editmode = Window.EditMode()
	if editmode: Window.EditMode(0)
	
	center = boundingObject.getLocation('worldspace')
	bounds = getMaxBounds(boundingObject.getName())
	for i in range(2):
		for n in range(3):
			bounds[i][n] += (resolution / 2.0) - center[n]
			
	bounds = [tuple([x*scale for x in bounds[0]]),tuple([x*scale for x in bounds[1]])]
	step = int(resolution * scale)
	timer = time.time()
	
	for x in range(int(bounds[0][0]), int(bounds[1][0] - step / 2), step):
		for y in range(int(bounds[0][1]), int(bounds[1][1] - step / 2), step):
			for z in range(int(bounds[0][2]), int(bounds[1][2] - step / 2), step):
				point = [x/scale,y/scale,z/scale]
				sample = Vector([(point[0]) / noiseSize, (point[1]) / noiseSize, (point[2]) / noiseSize])
				value = tex.evaluate(sample)[3]
				if value > threshold:					
					jitterpoint = [
						point[0] + ((random()-.5) * resolution * jitter),
						point[1] + ((random()-.5) * resolution * jitter),
						point[2] + ((random()-.5) * resolution * jitter)]
					coords.extend([jitterpoint])		
		
	try:
		CloudPreview = Mesh.Get(meshName)
		CloudPreview.verts.delete(CloudPreview.verts)
		meshExists = True
	except:
		CloudPreview = Mesh.New(meshName)

	CloudPreview.verts.extend(coords)

	if meshExists:
		CloudPreview.update()
		try:
			ob = Object.Get(meshName)
			try:
				scn.link(ob)
			except:
				meshExists = True
		except:
			ob = scn.objects.new(CloudPreview, meshName)	
	else:
		ob = scn.objects.new(CloudPreview, meshName)	

	ob.setLocation(center[0],center[1],center[2])
	print "cloudgen:", round(time.time() - timer,3), "seconds"

	Blender.Redraw()
	
def createEmpties(baseName, boundingObject):
	global mtTranslate, mtDarkening
	bounds = getMaxBounds(boundingObject)
	boundsObj = Object.Get(boundingObject)
	#Translation handle
	try:
		mtTranslate = Object.Get(baseName + 'Translate')		
	except:
		mtTranslate = Object.New('Empty',baseName + 'Translate')
		scn.link(mtTranslate)

	mtTranslate.setLocation(boundsObj.getLocation('worldspace'))
	mtTranslate.setSize(1,1,1)
	
	#Darkening handle
	try:
		mtDarkening = Object.Get(baseName + 'Darkening')		
	except:
		mtDarkening = Object.New('Empty',baseName + 'Darkening')
		scn.link(mtDarkening)

	mtDarkening.setLocation(((bounds[0][0] + bounds[1][0])/2.0, (bounds[0][1] + bounds[1][1])/2.0, bounds[0][2]))
	mtDarkening.setSize(1,1,bounds[1][2] - bounds[0][2])
	
def createMaterials(brightness, contrast, noiseSize, noiseBasis, baseName, objTranslate, objDarkening):
	###################
	#Material
	###################
	mat = Material.New('Clouds')
	mat.setRGBCol([0.5,0.5,0.5])
	mat.setRef(1.0)
	mat.setSpec(0.0)
	mat.setAlpha(0.0)
	mat.setTranslucency(1.0)
	mat.setSpecTransp(0.0)
	mat.mode = 68 #ztransparent and shadeless

	###################
	#Texture - Cloud Shape
	###################
	texShape = Texture.New(baseName + '_Shape')
	texShape.setType('Clouds')
	texShape.noiseSize = 1.0
	texShape.colorband = [[0,0,0,0,0],[1,1,1,1,1]]
	texShape.flags = Texture.Flags.COLORBAND
	texShape.brightness = brightness
	texShape.contrast = contrast
	texShape.noiseDepth = 6
	texShape.noiseBasis = noiseBasis

	mat.setTexture(0, texShape)
	texOptions = mat.getTextures()[0]
	texOptions.mapto = 129 #color and alpha
	texOptions.colfac = 0.1
	texOptions.object = objTranslate
	texOptions.texco = Texture.TexCo.OBJECT
	texOptions.size = (1.0 / noiseSize, 1.0 / noiseSize, 1.0 / noiseSize)

	###################
	#Texture - Cloud Darkening
	###################
	texColor = Texture.New(baseName + '_Darkening')
	texColor.setType('Blend')
	texColor.setImageFlags('Rot90')
	texColor.colorband = [[.3,.3,.3,1,0],[1,1,1,1,1]]
	texColor.flags = Texture.Flags.COLORBAND

	mat.setTexture(1, texColor)
	texOptions = mat.getTextures()[1]
	texOptions.blendmode = Texture.BlendModes.MULTIPLY
	texOptions.object = objDarkening
	texOptions.texco = Texture.TexCo.OBJECT
	texOptions.xproj = Texture.Proj.Z
	texOptions.yproj = Texture.Proj.Z
	texOptions.zproj = Texture.Proj.Z

	###################
	#Texture - Ease
	###################
	texEase = Texture.New(baseName + '_Ease')
	texEase.setType('Blend')
	texEase.colorband = [[0,0,0,0,0],[1,1,1,1,1]]
	texEase.flags = Texture.Flags.COLORBAND
	texEase.setSType('BlendSphere')

	mat.setTexture(2, texEase)
	texOptions = mat.getTextures()[2]
	texOptions.blendmode = Texture.BlendModes.MULTIPLY
	texOptions.mapto = 128 #alpha
	
	return mat
	
def createPlanes(points, scale, camera, material, baseName, boundingName):
	editmode = Window.EditMode()
	if editmode: Window.EditMode(0)

	scn = Scene.GetCurrent()   # link object to current scene
	pointcloudobj = points
	pointcloud = pointcloudobj.getData(False, True).verts
	
	boundingobj = Object.Get(boundingName)
	center = boundingobj.getLocation('worldspace')

	#Generic Cloud Plane
	coords=[[-scale,scale,0], [scale,scale,0], [scale,-scale,0], [-scale,-scale,0]]  
	faces= [[0,1,2,3]]

	cloudplane = Mesh.New(baseName + 'plane')   # create a new mesh

	cloudplane.verts.extend(coords)   # add vertices to mesh
	cloudplane.faces.extend(faces)    # add faces to the mesh (also adds edges)
	cloudplane.materials += [material]

	#Loop through current object's vertices and add planes
	for v in range(len(pointcloud)):		
		ob = scn.objects.new(cloudplane, baseName + 'plane')
		ob.setLocation (pointcloud[v].co[0] + center[0],pointcloud[v].co[1] + center[1] ,pointcloud[v].co[2] + center[2])
		#Camera Tracking
		trackto = ob.constraints.append(Constraint.Type.TRACKTO)
		trackto[Constraint.Settings.TARGET] = camera
		trackto[Constraint.Settings.TRACK] = Constraint.Settings.TRACKNEGZ
		trackto[Constraint.Settings.UP] = Constraint.Settings.UPY
		
	
	scn.objects.unlink(pointcloudobj)
	Window.Redraw()
	scn.update(1)

##################
#Bounds
##################
try:
	bounds = Object.GetSelected()[0]
	if bounds.getType() == 'Mesh':
		boundsName = Object.GetSelected()[0].getName()
	else:
		boundsName = ''	
except:
	boundsName = ''
txtBounds = Create(boundsName)	

strBounds1 = 'The Bounding Object is the name of a Blender'
strBounds2 = 'Object that defines a cube-shaped region where'
strBounds3 = 'clouds will be generated.  If you leave this blank'
strBounds4 = 'and press [Assign], one will be created for you.'

##################
#Shape
##################
mnuBasis = Create(1)
sldBrightness = Create(0.1)
sldContrast = Create(5.0)
sldThreshold = Create(0.0)
sldSize = Create(2.0)
sldJitter = Create(1.0)
sldResolution = Create(3.0)
txtCloudName = Create('MyClouds')

try:
	texPreview = Texture.Get('Preview')
	texPreview.brightness = brightness
	texPreview.contrast = contrast
	
except:
	texPreview = Texture.New('Preview')
	texPreview.setType('Clouds')
	texPreview.noiseSize = 1.0	
	texPreview.noiseDepth = 6
	texPreview.brightness = sldBrightness.val
	texPreview.contrast = sldContrast.val
	texPreview.noiseBasis = getNoiseBasis(mnuBasis.val)


strShape1 = 'The Shape of the clouds are represented as a'
strShape2 = 'grid of points within the Bounding Object.  The'
strShape3 = 'points are drawn in localspace. Edit the Bounding'
strShape4 = 'Object\'s vertices to change the region of interest.'

##################
#Billboards
##################

sldBillboardSize = Create(0.5)
try:
	scn = Scene.GetCurrent()
	cameraName = scn.objects.camera().getName()
except:
	cameraName = ''

txtCamera = Create(cameraName)
strBillboard1 = 'Billboards are textured planes that always face'
strBillboard2 = 'the camera.  These replace the points from the'
strBillboard3 = 'previous step.  You can later adjust the size of'
strBillboard4 = 'all planes by resizing any plane from edit mode.'

##################
#General
##################
mtTranslate = ''
mtDarkening = ''
tglBounds = Create(1)
tglShape = Create(0)
tglBillboards = Create(0)

strLine1 = strBounds1
strLine2 = strBounds2
strLine3 = strBounds3
strLine4 = strBounds4

imgHeaderLeft = 0
imgHeaderTop = 317
imgHeaderWidth = 280
imgHeaderHeight = 30
imgHeader = [[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[172, 172, 172],[172, 172, 172],[171, 171, 171],[171, 171, 171],[170, 170, 170],[170, 170, 170],[170, 170, 170],[170, 170, 170],[171, 171, 171],[171, 171, 171],[172, 172, 172],[174, 174, 174],[175, 175, 175],[176, 176, 176],[177, 177, 177],[177, 177, 177],[178, 178, 178],[179, 179, 179],[180, 180, 180],[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[191, 191, 191],[190, 190, 190],[190, 190, 190],[189, 189, 189],[188, 188, 188],[187, 187, 187],[186, 186, 186],[185, 185, 185],[184, 184, 184],[183, 183, 183],[183, 183, 183],[184, 184, 184],[186, 186, 186],[187, 187, 187],[189, 189, 189],[188, 188, 188],[187, 187, 187],[186, 186, 186],[186, 186, 186],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[186, 186, 186],[187, 187, 187],[188, 188, 188],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[191, 191, 191],[192, 192, 192],[192, 192, 192],[192, 192, 192],[193, 193, 193],[193, 193, 193],[193, 193, 193],[194, 194, 194],[193, 193, 193],[193, 193, 193],[193, 193, 193],[191, 191, 191],[191, 191, 191],[191, 191, 191],[190, 190, 190],[189, 189, 189],[189, 189, 189],[188, 188, 188],[188, 188, 188],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 189],[191, 191, 191],[192, 192, 192],[194, 194, 194],[197, 197, 197],[199, 199, 199],[200, 200, 200],[201, 201, 201],[201, 201, 201],[201, 201, 201],[200, 200, 200],[199, 199, 199],[198, 198, 198],[198, 198, 198],[197, 197, 197],[196, 196, 196],[195, 195, 195],[192, 194, 193],[191, 193, 192],[190, 192, 191],[189, 191, 190],[188, 189, 191],[188, 189, 191],[187, 188, 190],[187, 188, 190],[188, 189, 191],[187, 188, 190],[187, 188, 190],[187, 188, 190],[187, 188, 192],[187, 188, 192],[186, 187, 191],[184, 185, 189],[181, 186, 192],[176, 185, 194],[173, 183, 192],[163, 176, 192],[149, 167, 187],[134, 155, 182],[107, 134, 163],[78, 108, 144],[53, 89, 125],[42, 77, 117],[34, 71, 113],[35, 72, 114],[33, 72, 111],[32, 71, 110],[41, 80, 119],[57, 92, 132],[75, 103, 127],[91, 111, 135],[107, 128, 149],[125, 142, 160],[141, 153, 167],[153, 161, 172],[162, 167, 173],[169, 170, 174],[174, 176, 175],[177, 175, 176],[179, 178, 176],[179, 178, 176],[176, 178, 177],[176, 178, 177],[173, 177, 178],[173, 177, 178],[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[176, 176, 176],[176, 176, 176],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[179, 179, 179],[179, 179, 179],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[177, 177, 177],
	[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[172, 172, 172],[172, 172, 172],[171, 171, 171],[171, 171, 171],[171, 171, 171],[171, 171, 171],[171, 171, 171],[171, 171, 171],[172, 172, 172],[173, 173, 173],[175, 175, 175],[176, 176, 176],[177, 177, 177],[178, 178, 178],[178, 178, 178],[180, 180, 179],[180, 180, 179],[181, 181, 180],[181, 181, 181],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[184, 184, 184],[184, 184, 184],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 187],[188, 188, 187],[188, 188, 187],[188, 188, 187],[189, 189, 188],[189, 189, 188],[189, 189, 188],[190, 190, 189],[190, 190, 189],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 190],[191, 191, 190],[191, 191, 191],[191, 191, 191],[191, 191, 191],[190, 190, 190],[189, 189, 189],[188, 188, 188],[187, 187, 187],[186, 186, 186],[185, 185, 185],[184, 184, 184],[183, 183, 183],[183, 183, 183],[185, 185, 184],[187, 187, 186],[188, 188, 187],[189, 189, 188],[189, 189, 188],[188, 188, 187],[188, 188, 187],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[189, 189, 189],[190, 190, 190],[190, 190, 190],[191, 191, 191],[190, 190, 190],[190, 190, 190],[190, 190, 190],[189, 189, 189],[189, 189, 189],[189, 189, 188],[189, 189, 188],[189, 189, 188],[190, 190, 189],[191, 191, 190],[191, 191, 191],[191, 191, 191],[192, 192, 192],[193, 193, 193],[194, 194, 194],[194, 194, 194],[194, 194, 194],[194, 194, 194],[194, 194, 194],[193, 193, 193],[192, 192, 192],[192, 192, 192],[191, 191, 191],[191, 191, 191],[190, 190, 190],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 189],[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[191, 191, 191],[192, 192, 191],[192, 192, 191],[190, 190, 190],[189, 189, 189],[189, 189, 189],[189, 189, 189],[190, 190, 190],[191, 191, 191],[193, 193, 193],[195, 195, 195],[197, 197, 197],[200, 200, 200],[202, 202, 202],[203, 203, 203],[204, 204, 204],[203, 203, 203],[202, 202, 202],[201, 201, 201],[201, 201, 201],[200, 200, 200],[198, 198, 198],[197, 197, 197],[195, 195, 195],[191, 193, 192],[191, 193, 192],[190, 192, 191],[189, 191, 190],[189, 190, 191],[189, 190, 191],[189, 190, 191],[189, 190, 192],[188, 189, 191],[187, 188, 190],[187, 188, 190],[188, 189, 191],[188, 189, 193],[186, 187, 191],[184, 185, 189],[181, 182, 186],[175, 180, 186],[169, 178, 187],[164, 174, 183],[154, 166, 182],[136, 154, 173],[115, 136, 162],[88, 114, 142],[61, 91, 126],[44, 79, 114],[38, 72, 111],[33, 69, 110],[34, 70, 111],[32, 70, 108],[33, 71, 109],[44, 82, 120],[60, 95, 134],[79, 107, 130],[96, 116, 139],[112, 132, 153],[129, 145, 163],[144, 156, 170],[156, 164, 175],[166, 171, 177],[173, 174, 178],[175, 177, 176],[178, 176, 177],[179, 178, 176],[179, 178, 176],[177, 179, 177],[175, 177, 175],[173, 177, 177],[173, 177, 178],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[176, 176, 176],[176, 176, 176],[176, 176, 175],[176, 176, 175],[176, 176, 175],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[178, 178, 177],[178, 178, 177],[178, 178, 177],[178, 178, 177],[178, 178, 177],[178, 178, 177],[179, 179, 178],[179, 179, 178],[179, 179, 178],[180, 180, 179],[180, 180, 179],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 179],[181, 181, 180],[181, 181, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],
	[175, 175, 175],[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[173, 173, 173],[174, 174, 174],[176, 176, 176],[178, 178, 177],[179, 179, 179],[180, 180, 180],[180, 180, 180],[181, 181, 181],[182, 182, 182],[182, 182, 182],[184, 184, 183],[184, 184, 183],[185, 185, 184],[184, 184, 183],[184, 184, 183],[184, 184, 183],[183, 183, 182],[183, 183, 182],[183, 183, 182],[183, 183, 182],[184, 184, 183],[184, 184, 183],[184, 184, 183],[184, 184, 183],[185, 185, 184],[185, 185, 184],[185, 185, 185],[185, 185, 185],[185, 185, 185],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[188, 188, 187],[188, 188, 187],[188, 188, 187],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[190, 190, 189],[190, 190, 189],[190, 190, 189],[191, 191, 190],[191, 191, 190],[191, 191, 190],[191, 191, 190],[191, 191, 190],[190, 190, 190],[190, 190, 189],[190, 190, 189],[190, 190, 189],[191, 191, 190],[191, 191, 190],[191, 191, 190],[191, 191, 190],[192, 192, 192],[192, 192, 192],[192, 192, 192],[192, 192, 192],[192, 192, 192],[192, 192, 192],[192, 192, 191],[193, 193, 192],[192, 192, 192],[193, 193, 193],[192, 192, 192],[191, 191, 190],[189, 189, 188],[188, 188, 187],[186, 186, 186],[185, 185, 185],[184, 184, 184],[184, 184, 183],[185, 185, 184],[186, 186, 185],[188, 188, 188],[189, 189, 189],[189, 189, 189],[190, 190, 189],[190, 190, 189],[190, 190, 189],[191, 191, 190],[191, 191, 190],[192, 192, 191],[192, 192, 191],[193, 193, 192],[193, 193, 192],[194, 194, 193],[194, 194, 193],[194, 194, 193],[193, 193, 192],[192, 192, 191],[192, 192, 191],[190, 190, 189],[190, 190, 189],[188, 188, 188],[188, 188, 188],[189, 189, 189],[190, 190, 189],[191, 191, 190],[192, 192, 191],[193, 193, 192],[194, 194, 193],[195, 195, 194],[196, 196, 195],[196, 196, 195],[196, 196, 196],[196, 196, 196],[195, 195, 195],[195, 195, 195],[194, 194, 194],[194, 194, 194],[194, 194, 193],[194, 194, 193],[193, 193, 192],[191, 191, 191],[191, 191, 191],[191, 191, 191],[193, 193, 192],[193, 193, 192],[194, 194, 193],[194, 194, 193],[194, 194, 194],[194, 194, 194],[194, 194, 194],[194, 194, 194],[194, 194, 193],[194, 194, 193],[193, 193, 192],[191, 191, 191],[190, 190, 190],[191, 191, 191],[191, 191, 191],[193, 193, 193],[194, 194, 194],[196, 196, 196],[199, 199, 199],[202, 202, 202],[204, 204, 204],[206, 206, 206],[207, 207, 206],[208, 208, 207],[207, 207, 206],[206, 206, 205],[204, 204, 204],[203, 203, 203],[202, 202, 202],[199, 199, 199],[198, 198, 198],[193, 195, 194],[193, 195, 194],[193, 194, 193],[192, 194, 192],[191, 192, 193],[190, 191, 193],[190, 191, 193],[191, 192, 193],[190, 191, 192],[190, 191, 192],[189, 190, 193],[189, 190, 193],[188, 189, 192],[185, 186, 189],[180, 181, 184],[177, 178, 181],[166, 173, 180],[158, 167, 176],[147, 161, 172],[134, 150, 168],[112, 135, 157],[91, 117, 145],[67, 100, 132],[49, 85, 123],[39, 78, 119],[36, 75, 116],[33, 72, 113],[33, 72, 113],[35, 72, 110],[40, 77, 115],[51, 89, 125],[66, 99, 137],[88, 114, 133],[105, 122, 140],[121, 140, 155],[136, 151, 165],[151, 161, 170],[163, 169, 176],[171, 175, 177],[177, 177, 176],[180, 179, 174],[181, 180, 175],[182, 181, 176],[181, 180, 175],[178, 179, 176],[177, 178, 179],[175, 178, 179],[175, 179, 179],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[177, 177, 176],[178, 178, 177],[178, 178, 177],[178, 178, 177],[179, 179, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[179, 179, 179],[180, 180, 179],[180, 180, 179],[180, 180, 179],[180, 180, 179],[180, 180, 179],[180, 180, 179],[181, 181, 180],[181, 181, 180],[181, 181, 180],[181, 181, 180],[181, 181, 180],[182, 182, 181],[182, 182, 181],[182, 182, 181],[182, 182, 181],[182, 182, 181],[182, 182, 181],[183, 183, 182],[183, 183, 182],[183, 183, 182],[183, 183, 182],[182, 182, 181],[183, 183, 182],[182, 182, 182],[182, 182, 182],[183, 183, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[178, 178, 178],[178, 178, 178],[178, 178, 178],
	[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[172, 172, 172],[173, 173, 173],[175, 175, 175],[177, 177, 177],[179, 179, 179],[181, 181, 181],[183, 183, 182],[183, 183, 182],[184, 184, 183],[185, 185, 184],[185, 185, 184],[187, 187, 185],[187, 187, 185],[188, 188, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[186, 186, 185],[187, 187, 185],[187, 187, 185],[187, 187, 185],[187, 187, 185],[187, 187, 185],[187, 187, 185],[188, 188, 186],[188, 188, 186],[189, 189, 187],[189, 189, 187],[190, 190, 188],[191, 191, 189],[191, 191, 189],[191, 191, 189],[191, 191, 189],[192, 192, 190],[192, 192, 190],[192, 192, 190],[192, 192, 190],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[194, 194, 192],[194, 194, 192],[194, 194, 192],[194, 194, 192],[195, 195, 193],[195, 195, 193],[194, 194, 193],[194, 194, 193],[194, 194, 193],[194, 194, 193],[193, 193, 192],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[193, 193, 192],[194, 194, 193],[194, 194, 193],[196, 196, 194],[196, 196, 195],[196, 196, 195],[196, 196, 195],[195, 195, 194],[194, 194, 193],[194, 194, 193],[194, 194, 193],[194, 194, 193],[193, 193, 192],[192, 192, 191],[190, 190, 190],[190, 190, 189],[187, 187, 186],[187, 187, 186],[186, 186, 185],[186, 186, 185],[187, 187, 185],[189, 189, 188],[191, 191, 190],[192, 192, 191],[191, 191, 190],[192, 192, 190],[193, 193, 191],[193, 193, 191],[194, 194, 192],[194, 194, 193],[196, 196, 194],[197, 197, 195],[197, 197, 195],[197, 197, 195],[197, 197, 196],[197, 197, 196],[197, 197, 196],[197, 197, 195],[196, 196, 194],[195, 195, 193],[192, 192, 190],[192, 192, 190],[190, 190, 189],[190, 190, 189],[191, 191, 190],[192, 192, 190],[192, 192, 191],[193, 193, 192],[195, 195, 194],[196, 196, 195],[197, 197, 196],[198, 198, 197],[199, 199, 198],[199, 199, 197],[199, 199, 198],[199, 199, 198],[198, 198, 197],[198, 198, 197],[198, 198, 197],[197, 197, 196],[197, 197, 195],[196, 196, 195],[195, 195, 194],[195, 195, 194],[196, 196, 195],[196, 196, 195],[197, 197, 196],[197, 197, 196],[198, 198, 197],[198, 198, 197],[198, 198, 197],[198, 198, 197],[198, 198, 197],[198, 198, 196],[196, 196, 195],[194, 194, 193],[193, 193, 192],[192, 192, 192],[192, 192, 192],[192, 192, 191],[194, 194, 193],[195, 195, 194],[197, 197, 197],[200, 200, 200],[204, 204, 203],[207, 207, 206],[208, 208, 207],[209, 209, 209],[212, 212, 211],[211, 211, 210],[209, 209, 209],[209, 209, 208],[207, 207, 206],[205, 205, 204],[203, 203, 202],[201, 201, 200],[196, 198, 196],[196, 198, 196],[195, 197, 196],[195, 197, 195],[194, 195, 196],[194, 195, 195],[193, 194, 195],[194, 194, 195],[195, 196, 196],[194, 195, 195],[192, 193, 195],[191, 192, 194],[189, 190, 193],[185, 186, 188],[179, 180, 182],[175, 176, 177],[162, 168, 173],[149, 158, 165],[135, 148, 157],[121, 136, 151],[98, 120, 138],[81, 106, 130],[64, 95, 123],[53, 87, 122],[46, 83, 120],[44, 81, 118],[41, 79, 116],[40, 78, 115],[44, 80, 114],[50, 86, 120],[62, 98, 131],[75, 107, 141],[98, 123, 139],[115, 132, 147],[131, 148, 161],[145, 159, 171],[159, 168, 175],[168, 175, 180],[176, 179, 180],[180, 180, 178],[184, 183, 178],[184, 183, 178],[185, 184, 178],[185, 184, 178],[181, 183, 178],[180, 181, 181],[178, 182, 181],[178, 182, 181],[180, 180, 179],[180, 180, 179],[181, 181, 179],[181, 181, 179],[181, 181, 179],[182, 182, 180],[182, 182, 180],[182, 182, 181],[182, 182, 180],[182, 182, 180],[183, 183, 181],[183, 183, 181],[183, 183, 181],[184, 184, 182],[184, 184, 182],[184, 184, 182],[183, 183, 182],[184, 184, 182],[184, 184, 182],[184, 184, 182],[185, 185, 183],[185, 185, 183],[185, 185, 183],[185, 185, 183],[185, 185, 183],[185, 185, 183],[185, 185, 183],[186, 186, 184],[186, 186, 184],[186, 186, 185],[186, 186, 185],[186, 186, 185],[186, 186, 185],[186, 186, 185],[186, 186, 184],[186, 186, 184],[185, 185, 184],[185, 185, 184],[185, 185, 184],[184, 184, 183],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[178, 178, 178],[178, 178, 178],
	[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[173, 173, 173],[172, 172, 172],[173, 173, 173],[175, 175, 175],[176, 176, 176],[179, 179, 179],[180, 180, 180],[182, 182, 182],[184, 184, 183],[185, 185, 184],[187, 187, 185],[187, 187, 186],[189, 189, 187],[189, 189, 187],[190, 190, 188],[191, 191, 188],[191, 191, 189],[191, 191, 188],[191, 191, 188],[191, 191, 188],[191, 191, 188],[191, 191, 189],[192, 192, 189],[192, 192, 189],[192, 192, 189],[193, 193, 190],[193, 193, 190],[193, 193, 190],[194, 194, 191],[194, 194, 191],[195, 195, 192],[196, 196, 193],[196, 196, 193],[198, 198, 195],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[197, 197, 194],[198, 198, 195],[198, 198, 195],[198, 198, 195],[198, 198, 195],[198, 198, 195],[198, 198, 195],[198, 198, 196],[198, 198, 196],[198, 198, 196],[199, 199, 196],[200, 200, 197],[199, 199, 197],[200, 200, 197],[199, 199, 197],[198, 198, 196],[198, 198, 196],[196, 196, 194],[195, 195, 194],[195, 195, 193],[195, 195, 193],[195, 195, 193],[196, 196, 195],[197, 197, 195],[198, 198, 196],[199, 199, 197],[199, 199, 197],[198, 198, 196],[198, 198, 196],[198, 198, 196],[196, 196, 194],[196, 196, 194],[196, 196, 195],[195, 195, 194],[194, 194, 193],[193, 193, 192],[192, 192, 191],[191, 191, 190],[190, 190, 189],[188, 188, 187],[189, 189, 187],[189, 189, 187],[190, 190, 188],[192, 192, 190],[193, 193, 191],[195, 195, 192],[195, 195, 192],[195, 195, 193],[196, 196, 193],[197, 197, 194],[198, 198, 195],[199, 199, 197],[200, 200, 198],[201, 201, 199],[202, 202, 200],[202, 202, 200],[203, 203, 201],[203, 203, 201],[202, 202, 200],[201, 201, 198],[200, 200, 197],[199, 199, 196],[197, 197, 195],[195, 195, 193],[194, 194, 191],[194, 194, 192],[195, 195, 193],[194, 194, 192],[195, 195, 193],[197, 197, 195],[196, 196, 194],[197, 197, 195],[199, 199, 197],[200, 200, 198],[202, 202, 200],[203, 203, 201],[203, 203, 201],[205, 205, 202],[204, 204, 202],[203, 203, 201],[202, 202, 200],[202, 202, 199],[201, 201, 198],[200, 200, 198],[200, 200, 198],[200, 200, 198],[199, 201, 198],[200, 202, 199],[202, 202, 200],[203, 203, 201],[204, 204, 202],[204, 204, 202],[204, 204, 202],[204, 204, 202],[203, 203, 201],[201, 201, 199],[200, 200, 198],[197, 197, 196],[195, 195, 192],[194, 194, 191],[194, 194, 191],[193, 193, 191],[193, 195, 193],[196, 198, 196],[199, 200, 201],[202, 203, 204],[204, 208, 207],[207, 210, 210],[209, 213, 214],[211, 215, 216],[212, 216, 217],[211, 215, 216],[211, 212, 214],[210, 211, 213],[209, 210, 210],[207, 208, 209],[206, 206, 207],[205, 205, 206],[204, 202, 202],[201, 199, 198],[198, 198, 200],[198, 199, 201],[195, 200, 200],[194, 198, 199],[194, 200, 197],[196, 201, 199],[198, 199, 194],[199, 201, 195],[196, 199, 196],[188, 195, 194],[181, 193, 197],[174, 189, 199],[161, 183, 193],[152, 172, 189],[139, 161, 180],[129, 149, 175],[115, 137, 161],[100, 126, 152],[87, 115, 144],[75, 106, 137],[65, 100, 133],[63, 97, 130],[58, 95, 127],[55, 92, 124],[53, 88, 120],[55, 87, 116],[59, 89, 115],[70, 98, 123],[84, 112, 135],[98, 122, 143],[120, 134, 145],[133, 144, 153],[145, 158, 167],[159, 169, 176],[168, 177, 182],[176, 182, 186],[183, 187, 189],[187, 187, 188],[189, 189, 186],[188, 188, 186],[187, 187, 183],[186, 186, 182],[186, 186, 181],[186, 186, 181],[185, 185, 181],[186, 186, 181],[187, 187, 184],[187, 187, 184],[187, 187, 185],[187, 187, 185],[187, 187, 185],[188, 188, 185],[188, 188, 185],[188, 188, 185],[188, 188, 185],[187, 187, 185],[188, 188, 186],[188, 188, 186],[188, 188, 186],[188, 188, 186],[189, 189, 186],[189, 189, 187],[189, 189, 187],[190, 190, 187],[190, 190, 187],[190, 190, 187],[190, 190, 187],[190, 190, 187],[190, 190, 187],[190, 190, 187],[191, 191, 188],[191, 191, 188],[191, 191, 188],[191, 191, 188],[191, 191, 188],[191, 191, 188],[192, 192, 189],[192, 192, 189],[191, 191, 188],[191, 191, 188],[190, 190, 187],[189, 189, 187],[189, 189, 187],[189, 189, 187],[188, 188, 187],[187, 187, 186],[187, 187, 186],[186, 186, 185],[184, 184, 183],[183, 183, 183],[183, 183, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],
	[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[173, 173, 173],[173, 173, 173],[173, 173, 173],[175, 175, 175],[178, 178, 177],[179, 179, 179],[182, 182, 182],[184, 184, 183],[186, 186, 184],[187, 187, 186],[189, 189, 187],[191, 191, 188],[193, 193, 190],[194, 194, 190],[195, 195, 192],[196, 196, 192],[197, 197, 193],[198, 198, 194],[198, 198, 194],[199, 199, 194],[199, 199, 194],[199, 199, 195],[199, 199, 195],[199, 199, 195],[200, 200, 195],[200, 200, 195],[200, 200, 195],[201, 201, 196],[201, 201, 196],[201, 201, 197],[201, 201, 197],[202, 202, 198],[202, 202, 198],[203, 203, 199],[203, 203, 199],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[203, 203, 199],[203, 203, 199],[203, 203, 199],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 199],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[205, 205, 201],[205, 205, 201],[205, 205, 201],[205, 205, 201],[205, 205, 201],[204, 204, 200],[202, 202, 199],[202, 202, 199],[200, 200, 198],[199, 199, 197],[199, 199, 197],[199, 199, 196],[199, 199, 197],[199, 199, 197],[201, 201, 199],[202, 202, 199],[203, 203, 200],[203, 203, 200],[203, 203, 199],[202, 202, 199],[202, 202, 199],[200, 200, 197],[199, 199, 197],[198, 198, 196],[197, 197, 196],[197, 197, 195],[195, 195, 194],[195, 195, 194],[194, 194, 193],[193, 193, 192],[193, 193, 191],[193, 193, 190],[193, 193, 190],[195, 195, 191],[197, 197, 193],[198, 198, 195],[199, 199, 195],[200, 200, 196],[200, 200, 196],[201, 201, 197],[202, 202, 198],[202, 202, 199],[204, 204, 200],[205, 205, 201],[206, 206, 202],[206, 206, 203],[207, 207, 204],[208, 208, 204],[208, 208, 205],[208, 208, 204],[207, 207, 203],[205, 205, 201],[204, 204, 200],[202, 202, 199],[201, 201, 197],[199, 199, 195],[200, 200, 196],[199, 199, 196],[199, 199, 195],[200, 200, 197],[201, 201, 198],[201, 201, 198],[202, 202, 199],[203, 203, 200],[205, 205, 202],[207, 207, 204],[208, 208, 205],[209, 209, 206],[210, 210, 207],[209, 209, 206],[210, 210, 206],[208, 208, 205],[208, 208, 204],[207, 207, 203],[206, 206, 203],[206, 206, 203],[206, 206, 202],[206, 208, 203],[206, 208, 203],[208, 208, 204],[210, 210, 206],[211, 211, 207],[210, 210, 207],[210, 210, 207],[210, 210, 206],[209, 209, 206],[207, 207, 204],[204, 204, 202],[201, 201, 199],[199, 199, 195],[198, 198, 194],[197, 197, 194],[197, 197, 193],[197, 198, 196],[199, 201, 198],[202, 203, 204],[206, 207, 207],[208, 212, 211],[211, 214, 213],[213, 216, 216],[214, 217, 218],[216, 219, 219],[215, 218, 219],[215, 216, 217],[214, 215, 216],[212, 213, 213],[211, 212, 211],[211, 211, 210],[209, 209, 209],[208, 206, 205],[205, 204, 202],[202, 203, 203],[202, 203, 204],[200, 204, 203],[199, 203, 202],[200, 204, 201],[200, 205, 201],[201, 203, 197],[202, 203, 197],[198, 201, 196],[191, 197, 194],[183, 194, 195],[172, 186, 192],[155, 175, 181],[144, 162, 174],[130, 149, 162],[122, 140, 160],[112, 131, 149],[102, 125, 144],[94, 119, 140],[86, 114, 138],[79, 110, 136],[78, 109, 135],[74, 107, 132],[71, 104, 130],[70, 101, 126],[73, 101, 125],[78, 105, 125],[89, 114, 133],[103, 128, 146],[117, 139, 155],[137, 150, 157],[148, 158, 164],[159, 171, 177],[171, 180, 184],[179, 187, 190],[185, 191, 193],[190, 194, 194],[193, 194, 193],[195, 195, 191],[194, 194, 190],[193, 193, 187],[192, 192, 187],[191, 191, 185],[191, 191, 185],[191, 191, 186],[192, 192, 186],[192, 192, 188],[193, 193, 189],[193, 193, 189],[194, 194, 189],[195, 195, 190],[195, 195, 190],[195, 195, 190],[194, 194, 190],[195, 195, 190],[194, 194, 190],[194, 194, 190],[194, 194, 190],[195, 195, 191],[195, 195, 191],[195, 195, 191],[195, 195, 191],[195, 195, 191],[195, 195, 191],[195, 195, 191],[196, 196, 192],[196, 196, 192],[196, 196, 192],[196, 196, 192],[196, 196, 192],[197, 197, 193],[197, 197, 193],[196, 196, 192],[196, 196, 192],[196, 196, 192],[198, 198, 193],[198, 198, 193],[198, 198, 193],[196, 196, 192],[196, 196, 192],[195, 195, 191],[194, 194, 191],[194, 194, 190],[192, 192, 189],[191, 191, 189],[189, 189, 187],[188, 188, 187],[187, 187, 186],[185, 185, 184],[184, 184, 183],[183, 183, 183],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],
	[175, 175, 175],[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[173, 173, 173],[173, 173, 173],[173, 173, 173],[174, 174, 174],[175, 175, 175],[178, 178, 178],[181, 181, 180],[183, 183, 182],[185, 185, 184],[187, 187, 186],[189, 189, 187],[192, 192, 189],[194, 194, 191],[197, 197, 193],[199, 199, 195],[201, 201, 196],[203, 203, 197],[204, 204, 198],[205, 205, 200],[206, 206, 200],[206, 206, 200],[207, 207, 201],[207, 207, 201],[207, 207, 201],[208, 208, 201],[208, 208, 202],[207, 207, 201],[208, 208, 201],[208, 208, 201],[208, 208, 201],[209, 209, 202],[209, 209, 202],[209, 209, 203],[209, 209, 203],[211, 211, 204],[211, 211, 205],[211, 211, 205],[211, 211, 205],[212, 212, 206],[212, 212, 206],[211, 211, 205],[211, 211, 205],[211, 211, 205],[211, 211, 205],[211, 211, 205],[211, 211, 205],[211, 211, 205],[210, 210, 205],[210, 210, 205],[211, 211, 205],[212, 212, 206],[211, 211, 205],[211, 211, 205],[212, 212, 206],[212, 212, 206],[212, 212, 206],[211, 211, 206],[210, 210, 205],[209, 209, 204],[208, 208, 203],[206, 206, 202],[205, 205, 201],[203, 203, 200],[202, 202, 199],[202, 202, 199],[203, 203, 200],[204, 204, 201],[206, 206, 202],[207, 207, 203],[208, 208, 204],[209, 209, 204],[209, 209, 204],[208, 208, 203],[207, 207, 203],[205, 205, 201],[204, 204, 200],[202, 202, 199],[200, 200, 197],[199, 199, 197],[199, 199, 197],[199, 199, 197],[199, 199, 197],[198, 198, 196],[198, 198, 195],[198, 198, 195],[199, 199, 195],[200, 200, 196],[202, 202, 197],[204, 204, 199],[205, 205, 200],[205, 205, 199],[206, 206, 200],[207, 207, 201],[207, 207, 202],[208, 208, 202],[209, 209, 204],[210, 210, 205],[211, 211, 205],[213, 213, 207],[213, 213, 207],[214, 214, 208],[214, 214, 208],[214, 214, 208],[213, 213, 208],[212, 212, 206],[212, 212, 206],[209, 209, 204],[208, 208, 202],[206, 206, 201],[206, 206, 200],[206, 206, 200],[204, 204, 199],[205, 205, 200],[206, 206, 202],[206, 206, 202],[208, 208, 204],[210, 210, 205],[212, 212, 208],[213, 213, 209],[215, 215, 210],[216, 216, 211],[217, 217, 213],[217, 217, 213],[217, 217, 212],[217, 217, 212],[215, 215, 210],[214, 214, 209],[213, 213, 208],[213, 213, 208],[213, 213, 208],[214, 214, 208],[215, 215, 210],[216, 216, 211],[217, 217, 212],[218, 218, 213],[218, 218, 214],[218, 218, 213],[217, 217, 213],[215, 215, 211],[213, 213, 209],[210, 210, 207],[207, 207, 203],[204, 204, 201],[202, 202, 199],[201, 201, 198],[200, 200, 198],[201, 203, 200],[203, 205, 202],[207, 208, 207],[211, 211, 210],[212, 215, 215],[215, 218, 218],[217, 220, 220],[217, 221, 221],[218, 222, 222],[218, 221, 221],[217, 220, 220],[215, 219, 218],[215, 217, 217],[213, 215, 216],[211, 213, 214],[210, 212, 213],[211, 211, 212],[209, 209, 210],[205, 209, 210],[205, 208, 212],[203, 209, 212],[203, 209, 212],[202, 209, 209],[203, 209, 210],[202, 207, 205],[201, 205, 204],[196, 202, 201],[189, 197, 200],[180, 191, 195],[168, 181, 189],[153, 170, 176],[140, 159, 168],[131, 151, 164],[124, 146, 160],[117, 139, 153],[112, 135, 151],[107, 133, 149],[104, 130, 149],[100, 128, 147],[99, 127, 146],[94, 123, 140],[93, 122, 139],[94, 121, 137],[99, 122, 136],[103, 126, 137],[115, 136, 145],[129, 150, 159],[142, 160, 169],[159, 170, 170],[168, 176, 177],[178, 186, 187],[187, 193, 195],[192, 198, 198],[197, 201, 199],[199, 201, 199],[202, 202, 197],[202, 202, 196],[201, 201, 195],[199, 199, 192],[198, 198, 191],[197, 197, 191],[197, 197, 190],[197, 198, 190],[198, 199, 191],[200, 200, 194],[201, 201, 195],[201, 201, 195],[202, 202, 195],[202, 202, 196],[203, 203, 197],[203, 203, 196],[202, 202, 196],[203, 203, 197],[202, 202, 196],[202, 202, 196],[202, 202, 196],[202, 202, 196],[203, 203, 197],[203, 203, 197],[203, 203, 197],[203, 203, 197],[203, 203, 197],[203, 203, 197],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[204, 204, 198],[205, 205, 198],[205, 205, 199],[205, 205, 199],[205, 205, 199],[204, 204, 198],[203, 203, 197],[201, 201, 196],[200, 200, 195],[199, 199, 194],[197, 197, 193],[195, 195, 192],[192, 192, 189],[190, 190, 188],[187, 187, 186],[186, 186, 185],[184, 184, 184],[183, 183, 183],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],[179, 179, 179],
	[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[174, 174, 174],[175, 175, 175],[176, 176, 176],[179, 179, 179],[181, 181, 181],[184, 184, 184],[187, 187, 185],[190, 190, 188],[193, 193, 190],[196, 196, 192],[199, 199, 195],[202, 202, 197],[205, 205, 199],[207, 207, 201],[210, 210, 203],[211, 211, 204],[214, 214, 206],[214, 214, 207],[215, 215, 207],[215, 215, 207],[216, 216, 208],[216, 216, 208],[216, 216, 208],[217, 217, 209],[216, 216, 208],[217, 217, 208],[217, 217, 208],[217, 217, 208],[217, 217, 208],[217, 217, 209],[218, 218, 209],[218, 218, 210],[219, 219, 211],[219, 219, 211],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[219, 219, 211],[219, 219, 211],[219, 219, 211],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[220, 220, 212],[219, 219, 211],[217, 217, 210],[216, 216, 210],[215, 215, 209],[212, 212, 207],[210, 210, 205],[209, 209, 204],[208, 208, 204],[207, 207, 203],[208, 208, 204],[209, 209, 205],[211, 211, 206],[212, 212, 207],[214, 214, 208],[215, 215, 209],[215, 215, 209],[214, 214, 208],[213, 213, 207],[211, 211, 205],[209, 209, 204],[206, 206, 202],[204, 204, 201],[203, 203, 200],[203, 203, 201],[204, 204, 202],[206, 206, 203],[206, 206, 202],[205, 205, 201],[205, 205, 200],[206, 206, 200],[208, 208, 201],[209, 209, 202],[210, 210, 203],[211, 211, 203],[211, 211, 204],[212, 212, 204],[212, 212, 204],[214, 214, 206],[214, 214, 207],[215, 215, 208],[217, 217, 210],[217, 217, 210],[218, 218, 211],[219, 219, 212],[220, 220, 212],[220, 220, 213],[220, 220, 213],[220, 220, 213],[219, 219, 212],[219, 219, 212],[218, 218, 210],[216, 216, 208],[214, 214, 207],[214, 214, 206],[213, 213, 205],[212, 212, 205],[212, 212, 205],[213, 213, 207],[214, 214, 208],[214, 214, 209],[216, 216, 211],[219, 219, 213],[220, 220, 214],[222, 222, 216],[224, 224, 218],[225, 225, 219],[225, 225, 219],[225, 225, 218],[224, 224, 218],[223, 223, 216],[223, 223, 216],[222, 222, 215],[221, 221, 214],[220, 220, 213],[222, 222, 215],[223, 223, 216],[224, 224, 217],[225, 225, 218],[225, 225, 219],[225, 225, 219],[225, 225, 219],[225, 225, 219],[222, 222, 217],[220, 220, 215],[216, 216, 212],[213, 213, 208],[209, 209, 205],[206, 206, 203],[205, 205, 202],[204, 204, 201],[206, 207, 204],[207, 209, 205],[211, 211, 210],[215, 216, 214],[217, 220, 218],[220, 222, 220],[221, 224, 223],[222, 225, 222],[222, 225, 222],[221, 224, 222],[220, 223, 221],[219, 222, 219],[218, 220, 218],[216, 218, 216],[213, 215, 214],[212, 214, 212],[212, 213, 211],[211, 211, 209],[207, 211, 209],[208, 211, 212],[207, 212, 212],[207, 212, 212],[206, 211, 209],[206, 211, 208],[205, 209, 205],[202, 206, 201],[196, 201, 195],[188, 194, 192],[180, 189, 186],[171, 182, 182],[160, 174, 171],[151, 166, 166],[146, 162, 165],[140, 158, 162],[137, 154, 158],[133, 153, 158],[131, 153, 158],[129, 150, 158],[126, 148, 156],[124, 147, 155],[123, 146, 153],[121, 145, 152],[123, 145, 151],[128, 147, 153],[135, 154, 157],[147, 164, 167],[157, 175, 179],[167, 182, 186],[181, 190, 186],[189, 196, 193],[196, 202, 200],[202, 206, 205],[205, 210, 207],[208, 211, 207],[209, 211, 206],[210, 210, 204],[209, 209, 202],[209, 209, 201],[207, 207, 198],[206, 206, 198],[205, 205, 196],[205, 205, 196],[205, 207, 197],[207, 208, 198],[209, 209, 200],[210, 210, 202],[211, 211, 202],[211, 211, 203],[212, 212, 203],[212, 212, 204],[212, 212, 204],[212, 212, 204],[212, 212, 204],[211, 211, 203],[211, 211, 203],[211, 211, 203],[211, 211, 203],[211, 211, 204],[211, 211, 204],[212, 212, 204],[212, 212, 204],[212, 212, 204],[212, 212, 204],[212, 212, 204],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[213, 213, 205],[211, 211, 204],[209, 209, 202],[207, 207, 200],[205, 205, 199],[202, 202, 197],[198, 198, 194],[195, 195, 192],[192, 192, 190],[189, 189, 188],[187, 187, 186],[185, 185, 184],[184, 184, 183],[182, 182, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],[179, 179, 179],
	[176, 176, 176],[176, 176, 176],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[174, 174, 174],[174, 174, 174],[175, 175, 175],[176, 176, 176],[177, 177, 177],[180, 180, 180],[182, 182, 182],[185, 185, 184],[189, 189, 187],[192, 192, 189],[195, 195, 192],[199, 199, 195],[203, 203, 198],[208, 208, 202],[212, 212, 205],[215, 215, 207],[218, 218, 209],[220, 220, 211],[222, 222, 212],[223, 223, 213],[224, 224, 214],[225, 225, 214],[225, 225, 215],[225, 225, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[227, 227, 216],[227, 227, 216],[227, 227, 216],[228, 228, 218],[228, 228, 218],[228, 228, 218],[229, 229, 219],[230, 230, 219],[230, 230, 219],[229, 229, 219],[229, 229, 219],[228, 228, 218],[228, 228, 218],[229, 229, 218],[229, 229, 218],[228, 228, 218],[228, 228, 218],[228, 228, 218],[228, 228, 218],[229, 229, 219],[229, 229, 219],[229, 229, 219],[229, 229, 219],[229, 229, 218],[228, 228, 218],[227, 227, 217],[225, 225, 216],[224, 224, 216],[221, 221, 214],[219, 219, 212],[216, 216, 210],[214, 214, 209],[214, 214, 208],[213, 213, 208],[214, 214, 208],[215, 215, 209],[217, 217, 211],[220, 220, 213],[221, 221, 214],[222, 222, 214],[222, 222, 215],[221, 221, 213],[220, 220, 212],[217, 217, 210],[214, 214, 208],[211, 211, 205],[209, 209, 204],[208, 208, 204],[209, 209, 206],[211, 211, 208],[213, 213, 210],[214, 214, 210],[213, 213, 208],[213, 213, 206],[213, 213, 206],[215, 215, 207],[217, 217, 208],[218, 218, 208],[218, 218, 208],[219, 219, 209],[219, 219, 209],[219, 219, 209],[220, 220, 211],[221, 221, 212],[222, 222, 213],[223, 223, 214],[225, 225, 215],[225, 225, 216],[227, 227, 217],[227, 227, 217],[228, 228, 218],[228, 228, 218],[228, 228, 218],[228, 228, 218],[228, 228, 218],[227, 227, 217],[225, 225, 215],[224, 224, 214],[223, 223, 213],[222, 222, 212],[221, 221, 212],[221, 221, 212],[221, 221, 213],[222, 222, 213],[223, 223, 215],[224, 224, 217],[227, 227, 219],[228, 228, 221],[230, 230, 222],[232, 232, 224],[233, 233, 225],[233, 233, 225],[233, 233, 225],[233, 233, 225],[232, 232, 224],[231, 231, 223],[230, 230, 222],[230, 230, 221],[230, 230, 221],[230, 230, 221],[231, 231, 222],[232, 232, 224],[233, 233, 225],[233, 233, 226],[234, 234, 226],[233, 233, 226],[232, 232, 226],[230, 230, 223],[227, 227, 221],[223, 223, 218],[219, 219, 213],[215, 215, 210],[212, 212, 208],[210, 210, 206],[209, 209, 205],[210, 211, 208],[212, 213, 210],[215, 215, 214],[219, 220, 218],[221, 224, 221],[224, 226, 224],[225, 229, 226],[226, 229, 228],[226, 230, 228],[225, 229, 227],[224, 228, 226],[222, 227, 225],[220, 225, 223],[217, 222, 220],[214, 219, 217],[213, 218, 216],[211, 216, 214],[210, 215, 214],[208, 215, 214],[209, 215, 215],[208, 216, 216],[209, 216, 216],[208, 216, 214],[208, 215, 212],[206, 213, 209],[204, 211, 208],[198, 205, 201],[191, 200, 195],[185, 196, 192],[180, 192, 187],[174, 186, 180],[169, 183, 175],[166, 182, 176],[165, 180, 176],[162, 179, 173],[162, 179, 172],[162, 179, 174],[160, 177, 173],[159, 177, 172],[156, 174, 171],[156, 173, 167],[155, 171, 164],[158, 173, 165],[164, 178, 170],[173, 184, 177],[181, 192, 185],[189, 199, 194],[195, 204, 200],[205, 209, 199],[209, 212, 204],[213, 217, 208],[217, 219, 212],[219, 221, 212],[221, 221, 214],[220, 221, 211],[220, 219, 210],[220, 219, 208],[218, 218, 207],[216, 216, 205],[215, 215, 204],[214, 214, 204],[214, 214, 204],[215, 216, 206],[216, 217, 207],[218, 218, 208],[220, 220, 210],[221, 221, 211],[222, 222, 211],[222, 222, 211],[223, 223, 212],[223, 223, 212],[222, 222, 211],[221, 221, 211],[220, 220, 210],[220, 220, 210],[220, 220, 210],[220, 220, 210],[220, 220, 210],[221, 221, 211],[221, 221, 211],[221, 221, 211],[221, 221, 211],[222, 222, 211],[222, 222, 212],[222, 222, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[222, 222, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[223, 223, 212],[222, 222, 212],[220, 220, 210],[219, 219, 209],[215, 215, 206],[213, 213, 205],[208, 208, 201],[204, 204, 198],[199, 199, 195],[196, 196, 192],[192, 192, 189],[189, 189, 187],[187, 187, 185],[185, 185, 184],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],[179, 179, 179],
	[176, 176, 176],[176, 176, 176],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[176, 176, 176],[176, 176, 176],[178, 178, 178],[180, 180, 180],[184, 184, 183],[187, 187, 186],[190, 190, 188],[194, 194, 191],[199, 199, 195],[203, 203, 198],[208, 208, 202],[213, 213, 206],[218, 218, 209],[223, 223, 213],[226, 226, 215],[229, 229, 218],[232, 232, 220],[233, 233, 221],[234, 234, 222],[235, 235, 222],[236, 236, 222],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[237, 237, 223],[237, 237, 223],[237, 237, 224],[238, 238, 225],[238, 238, 225],[238, 238, 225],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 226],[238, 238, 225],[238, 238, 225],[238, 238, 225],[238, 238, 225],[238, 238, 225],[238, 238, 226],[237, 237, 225],[235, 235, 223],[233, 233, 222],[230, 230, 220],[227, 227, 218],[224, 224, 215],[221, 221, 213],[220, 220, 213],[219, 219, 212],[220, 220, 213],[222, 222, 214],[224, 224, 216],[227, 227, 218],[229, 229, 220],[231, 231, 221],[230, 230, 221],[229, 229, 219],[227, 227, 218],[225, 225, 217],[221, 221, 213],[217, 217, 211],[214, 214, 209],[214, 214, 209],[216, 216, 212],[218, 218, 215],[222, 222, 218],[222, 222, 218],[221, 221, 215],[220, 220, 213],[221, 221, 212],[223, 223, 213],[225, 225, 214],[226, 226, 214],[226, 226, 214],[227, 227, 215],[227, 227, 215],[227, 227, 215],[227, 227, 216],[229, 229, 217],[229, 229, 218],[231, 231, 219],[233, 233, 220],[234, 234, 221],[235, 235, 222],[236, 236, 224],[237, 237, 224],[237, 237, 225],[238, 238, 225],[238, 238, 225],[238, 238, 225],[237, 237, 225],[236, 236, 223],[234, 234, 222],[233, 233, 221],[232, 232, 220],[231, 231, 219],[231, 231, 219],[231, 231, 220],[231, 231, 221],[231, 231, 222],[233, 233, 223],[235, 235, 225],[237, 237, 227],[239, 239, 229],[240, 240, 231],[241, 241, 232],[241, 241, 232],[242, 242, 232],[242, 242, 232],[241, 241, 231],[241, 241, 230],[240, 240, 230],[239, 239, 229],[239, 239, 229],[240, 240, 230],[240, 240, 230],[241, 241, 232],[242, 242, 232],[242, 242, 234],[242, 242, 233],[241, 241, 233],[240, 240, 232],[238, 238, 230],[234, 234, 227],[231, 231, 224],[226, 226, 220],[222, 222, 216],[218, 218, 213],[216, 216, 211],[216, 216, 211],[216, 216, 213],[218, 218, 214],[220, 221, 217],[224, 225, 221],[227, 229, 224],[230, 232, 227],[231, 233, 227],[232, 235, 229],[233, 235, 230],[232, 235, 229],[231, 234, 228],[229, 232, 226],[227, 230, 224],[224, 228, 222],[221, 225, 218],[219, 223, 217],[218, 222, 215],[219, 222, 217],[217, 222, 216],[218, 223, 217],[218, 224, 218],[219, 225, 217],[219, 224, 215],[219, 223, 213],[218, 222, 210],[217, 221, 209],[214, 219, 206],[210, 216, 202],[208, 214, 200],[204, 212, 195],[202, 209, 190],[198, 206, 186],[197, 207, 189],[196, 206, 189],[195, 206, 188],[195, 206, 188],[195, 206, 189],[194, 205, 189],[195, 206, 189],[195, 206, 189],[194, 204, 187],[194, 204, 187],[197, 206, 189],[201, 210, 195],[207, 215, 202],[213, 220, 209],[217, 223, 214],[220, 226, 217],[225, 228, 215],[228, 230, 218],[230, 232, 220],[232, 233, 222],[232, 234, 222],[233, 233, 222],[232, 232, 220],[231, 230, 218],[230, 229, 216],[228, 228, 214],[227, 226, 213],[225, 225, 212],[224, 224, 212],[225, 225, 212],[226, 226, 214],[228, 228, 216],[230, 230, 217],[231, 231, 218],[232, 232, 219],[233, 233, 220],[234, 234, 220],[234, 234, 220],[234, 234, 221],[234, 234, 220],[233, 233, 219],[231, 231, 219],[231, 231, 218],[231, 231, 218],[231, 231, 218],[231, 231, 218],[231, 231, 218],[231, 231, 218],[231, 231, 218],[231, 231, 218],[232, 232, 219],[232, 232, 219],[233, 233, 220],[233, 233, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 220],[234, 234, 221],[234, 234, 220],[233, 233, 220],[231, 231, 218],[229, 229, 216],[225, 225, 214],[221, 221, 211],[216, 216, 207],[211, 211, 203],[205, 205, 199],[200, 200, 196],[195, 195, 191],[191, 191, 188],[187, 187, 186],[185, 185, 184],[183, 183, 183],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[176, 176, 176],[176, 176, 176],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[175, 175, 175],[176, 176, 176],[177, 177, 177],[178, 178, 178],[181, 181, 181],[184, 184, 183],[187, 187, 186],[192, 192, 190],[196, 196, 193],[201, 201, 196],[206, 206, 200],[184, 184, 178],[29, 29, 28],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[190, 190, 179],[237, 237, 224],[237, 237, 224],[174, 174, 164],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[95, 95, 90],[190, 190, 179],[47, 47, 45],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[48, 48, 45],[206, 206, 195],[222, 222, 210],[95, 95, 90],[0, 0, 0],[0, 0, 0],[48, 48, 45],[175, 175, 166],[127, 127, 121],[0, 0, 0],[0, 0, 0],[127, 127, 121],[238, 238, 226],[159, 159, 151],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[48, 48, 45],[206, 206, 195],[111, 111, 105],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[31, 31, 30],[199, 199, 192],[227, 227, 218],[226, 226, 217],[226, 226, 217],[226, 226, 218],[229, 229, 219],[232, 232, 221],[62, 62, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[124, 124, 118],[228, 228, 218],[224, 224, 216],[221, 221, 214],[220, 220, 214],[222, 222, 218],[225, 225, 221],[229, 229, 224],[230, 230, 224],[228, 228, 221],[228, 228, 219],[229, 229, 218],[139, 139, 131],[78, 78, 73],[0, 0, 0],[0, 0, 0],[0, 0, 0],[62, 62, 59],[125, 125, 118],[235, 235, 221],[236, 236, 222],[237, 237, 222],[174, 174, 164],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[79, 79, 75],[238, 238, 224],[238, 238, 225],[206, 206, 195],[111, 111, 105],[0, 0, 0],[0, 0, 0],[0, 0, 0],[47, 47, 45],[126, 126, 119],[238, 238, 225],[238, 238, 226],[239, 239, 227],[239, 239, 227],[240, 240, 229],[225, 225, 215],[96, 96, 92],[0, 0, 0],[0, 0, 0],[49, 49, 47],[162, 162, 156],[130, 130, 125],[0, 0, 0],[0, 0, 0],[48, 48, 46],[129, 129, 123],[210, 210, 200],[242, 242, 231],[242, 242, 232],[177, 177, 170],[49, 49, 47],[0, 0, 0],[0, 0, 0],[114, 114, 110],[211, 211, 205],[0, 0, 0],[0, 0, 0],[32, 32, 31],[126, 126, 123],[201, 202, 197],[227, 228, 222],[224, 224, 219],[222, 223, 220],[221, 222, 219],[222, 222, 219],[223, 224, 220],[225, 227, 221],[229, 230, 226],[232, 234, 228],[172, 174, 170],[111, 112, 108],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[124, 126, 122],[202, 205, 198],[231, 235, 227],[229, 232, 224],[227, 231, 223],[225, 230, 222],[226, 231, 221],[227, 232, 222],[151, 154, 148],[60, 62, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[101, 105, 98],[173, 178, 169],[214, 221, 209],[198, 204, 193],[69, 72, 68],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[104, 110, 100],[155, 164, 150],[26, 27, 25],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[108, 112, 102],[204, 212, 194],[209, 216, 198],[143, 147, 137],[59, 60, 56],[0, 0, 0],[0, 0, 0],[0, 0, 0],[108, 109, 102],[187, 187, 176],[235, 235, 222],[205, 205, 193],[63, 63, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[110, 110, 103],[235, 235, 220],[234, 234, 220],[234, 234, 221],[236, 236, 222],[126, 126, 118],[0, 0, 0],[0, 0, 0],[16, 16, 15],[142, 142, 133],[173, 173, 163],[16, 16, 15],[0, 0, 0],[94, 94, 89],[220, 220, 207],[236, 236, 222],[236, 236, 222],[236, 236, 222],[111, 111, 104],[0, 0, 0],[0, 0, 0],[0, 0, 0],[79, 79, 74],[142, 142, 133],[237, 237, 222],[237, 237, 222],[237, 237, 222],[126, 126, 118],[16, 16, 15],[0, 0, 0],[0, 0, 0],[16, 16, 15],[126, 126, 118],[205, 205, 192],[236, 236, 222],[236, 236, 222],[205, 205, 192],[63, 63, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[105, 105, 99],[217, 217, 208],[210, 210, 203],[204, 204, 198],[198, 198, 194],[193, 193, 190],[189, 189, 187],[186, 186, 185],[184, 184, 183],[182, 182, 182],[182, 182, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],
	[176, 176, 176],[176, 176, 176],[175, 175, 175],[175, 175, 175],[175, 175, 175],[176, 176, 176],[176, 176, 176],[176, 176, 176],[176, 176, 176],[176, 176, 176],[177, 177, 177],[177, 177, 177],[179, 179, 179],[181, 181, 181],[184, 184, 183],[187, 187, 186],[192, 192, 190],[196, 196, 193],[202, 202, 197],[208, 208, 202],[186, 186, 179],[44, 44, 42],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 149],[237, 237, 224],[237, 237, 224],[126, 126, 119],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[205, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[206, 206, 195],[95, 95, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[127, 127, 121],[95, 95, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[189, 189, 180],[234, 234, 223],[232, 232, 222],[232, 232, 222],[233, 233, 222],[235, 235, 224],[175, 175, 166],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[109, 109, 104],[229, 229, 219],[225, 225, 217],[225, 225, 219],[227, 227, 221],[230, 230, 225],[233, 233, 228],[235, 235, 229],[235, 235, 227],[187, 187, 179],[47, 47, 45],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[236, 236, 222],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[237, 237, 224],[190, 190, 179],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[191, 191, 181],[239, 239, 227],[239, 239, 227],[240, 240, 229],[96, 96, 92],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 62],[242, 242, 232],[162, 162, 156],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 62],[232, 233, 227],[229, 229, 223],[227, 227, 223],[227, 227, 223],[228, 228, 224],[229, 230, 224],[231, 233, 226],[219, 220, 214],[79, 80, 77],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[139, 140, 135],[230, 233, 223],[229, 232, 222],[227, 231, 220],[227, 231, 219],[121, 123, 116],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[141, 146, 137],[183, 189, 178],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[52, 55, 50],[143, 151, 139],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[69, 72, 67],[210, 217, 203],[115, 118, 111],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[157, 157, 149],[205, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[236, 236, 221],[236, 236, 222],[236, 236, 222],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[79, 79, 74],[236, 236, 222],[236, 236, 222],[94, 94, 89],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 163],[221, 221, 207],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[157, 157, 148],[236, 236, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[62, 62, 58],[223, 223, 212],[216, 216, 207],[209, 209, 201],[202, 202, 197],[196, 196, 192],[191, 191, 189],[187, 187, 186],[185, 185, 184],[183, 183, 183],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],
	[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[176, 176, 176],[176, 176, 176],[176, 176, 176],[177, 177, 177],[177, 177, 177],[177, 177, 177],[178, 178, 178],[179, 179, 179],[180, 180, 180],[183, 183, 183],[186, 186, 185],[188, 188, 187],[193, 193, 190],[197, 197, 194],[202, 202, 197],[208, 208, 202],[215, 215, 207],[223, 223, 213],[107, 107, 102],[0, 0, 0],[0, 0, 0],[142, 142, 134],[237, 237, 224],[237, 237, 224],[237, 237, 224],[238, 238, 224],[95, 95, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[47, 47, 45],[237, 237, 224],[237, 237, 224],[158, 158, 149],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[159, 159, 150],[238, 238, 225],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[191, 191, 181],[96, 96, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 64, 60],[239, 239, 226],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[48, 48, 45],[239, 239, 226],[191, 191, 181],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[127, 127, 121],[239, 239, 227],[239, 239, 227],[238, 238, 226],[237, 237, 226],[238, 238, 226],[239, 239, 227],[112, 112, 106],[0, 0, 0],[48, 48, 45],[223, 223, 212],[112, 112, 106],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[235, 235, 225],[231, 231, 222],[231, 231, 224],[233, 233, 226],[234, 234, 228],[236, 236, 230],[238, 238, 231],[206, 206, 199],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 133],[236, 236, 222],[110, 110, 104],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[221, 221, 207],[237, 237, 223],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[221, 221, 209],[95, 95, 90],[0, 0, 0],[0, 0, 0],[16, 16, 15],[207, 207, 197],[240, 240, 229],[240, 240, 229],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[98, 98, 94],[244, 244, 236],[49, 49, 47],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 16],[211, 211, 203],[227, 227, 218],[16, 16, 16],[0, 0, 0],[0, 0, 0],[0, 0, 0],[115, 115, 112],[246, 247, 241],[33, 33, 32],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 16],[209, 210, 205],[236, 237, 232],[233, 234, 230],[230, 232, 227],[229, 231, 226],[230, 233, 228],[233, 235, 229],[235, 237, 230],[48, 48, 47],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[151, 154, 148],[226, 230, 222],[225, 230, 220],[164, 168, 160],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[143, 147, 137],[210, 218, 202],[153, 159, 147],[42, 43, 41],[111, 116, 109],[208, 217, 205],[124, 129, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[132, 139, 126],[199, 209, 189],[67, 70, 64],[0, 0, 0],[0, 0, 0],[0, 0, 0],[14, 14, 13],[199, 203, 188],[159, 163, 151],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[156, 157, 147],[235, 235, 222],[173, 173, 164],[47, 47, 45],[126, 126, 119],[237, 237, 224],[142, 142, 134],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[237, 236, 222],[237, 236, 222],[236, 236, 222],[236, 236, 222],[63, 63, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 222],[221, 221, 207],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[79, 79, 74],[236, 236, 222],[236, 236, 222],[236, 236, 222],[94, 94, 89],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[190, 190, 178],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[157, 157, 148],[236, 236, 222],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[236, 236, 222],[229, 229, 217],[221, 221, 211],[212, 212, 204],[205, 205, 199],[199, 199, 194],[193, 193, 190],[189, 189, 187],[186, 186, 185],[184, 184, 183],[182, 182, 182],[182, 182, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[177, 177, 177],[178, 178, 178],[179, 179, 179],[179, 179, 179],[181, 181, 181],[183, 183, 183],[186, 186, 185],[189, 189, 188],[192, 192, 190],[197, 197, 193],[201, 201, 197],[208, 208, 202],[215, 215, 207],[224, 224, 214],[217, 217, 205],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 164],[237, 237, 224],[237, 237, 224],[205, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[239, 239, 226],[127, 127, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[239, 239, 227],[239, 239, 227],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 90],[239, 239, 226],[223, 223, 211],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[159, 159, 151],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[127, 127, 121],[48, 48, 45],[223, 223, 212],[127, 127, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[238, 238, 227],[235, 235, 226],[234, 234, 226],[234, 234, 228],[236, 236, 229],[238, 238, 231],[240, 240, 232],[65, 65, 62],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[125, 125, 117],[235, 235, 220],[235, 235, 220],[235, 235, 220],[125, 125, 117],[16, 16, 15],[141, 141, 132],[236, 236, 221],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[236, 236, 222],[157, 157, 148],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[127, 127, 119],[238, 238, 224],[127, 127, 120],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 91],[240, 240, 229],[241, 241, 229],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[146, 146, 142],[244, 244, 236],[98, 98, 94],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 31],[244, 244, 236],[130, 130, 126],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[231, 231, 226],[247, 247, 242],[115, 115, 113],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 64],[243, 244, 238],[240, 241, 235],[236, 237, 232],[234, 235, 230],[233, 234, 229],[234, 236, 230],[236, 238, 231],[111, 112, 108],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[122, 124, 119],[228, 232, 223],[182, 186, 178],[0, 0, 0],[0, 0, 0],[0, 0, 0],[30, 31, 29],[224, 228, 218],[223, 228, 215],[74, 75, 71],[0, 0, 0],[0, 0, 0],[0, 0, 0],[114, 117, 108],[211, 218, 200],[207, 215, 196],[206, 214, 196],[206, 214, 199],[206, 215, 200],[206, 215, 201],[151, 157, 147],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 184, 170],[203, 213, 196],[96, 100, 93],[0, 0, 0],[0, 0, 0],[0, 0, 0],[42, 44, 41],[216, 222, 207],[73, 75, 71],[0, 0, 0],[0, 0, 0],[0, 0, 0],[125, 125, 118],[235, 236, 223],[237, 236, 224],[237, 237, 224],[237, 237, 224],[237, 237, 224],[237, 237, 225],[190, 190, 179],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 148],[237, 236, 222],[237, 236, 222],[236, 236, 222],[237, 237, 222],[79, 79, 74],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 164],[174, 174, 164],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[205, 205, 192],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[236, 236, 222],[236, 236, 222],[220, 220, 207],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[237, 237, 222],[174, 174, 163],[0, 0, 0],[0, 0, 0],[0, 0, 0],[31, 31, 30],[236, 236, 222],[189, 189, 178],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[157, 157, 148],[236, 236, 222],[234, 234, 220],[225, 225, 214],[217, 217, 208],[209, 209, 201],[201, 201, 196],[196, 196, 192],[191, 191, 188],[187, 187, 185],[184, 184, 184],[183, 183, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[177, 177, 177],[177, 177, 177],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[178, 178, 178],[179, 179, 179],[180, 180, 180],[180, 180, 180],[182, 182, 182],[184, 184, 184],[186, 186, 185],[188, 188, 187],[191, 191, 189],[195, 195, 193],[201, 201, 197],[207, 207, 201],[215, 215, 207],[223, 223, 213],[232, 232, 220],[126, 126, 119],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[237, 237, 224],[237, 237, 224],[237, 237, 224],[205, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[143, 143, 135],[0, 0, 0],[0, 0, 0],[0, 0, 0],[159, 159, 151],[127, 127, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 64, 61],[239, 239, 227],[239, 239, 227],[80, 80, 76],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 91],[239, 239, 227],[239, 239, 227],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 166],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[192, 192, 182],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[112, 112, 107],[240, 240, 229],[236, 236, 227],[235, 235, 227],[236, 236, 228],[237, 237, 230],[240, 240, 232],[177, 177, 171],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[157, 157, 147],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[141, 141, 132],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[236, 236, 222],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 226],[127, 127, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[240, 240, 229],[241, 241, 230],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[146, 146, 142],[245, 245, 237],[131, 131, 126],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 32],[245, 245, 238],[82, 82, 80],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[247, 248, 244],[247, 248, 244],[132, 132, 130],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 64],[243, 244, 240],[241, 243, 238],[238, 239, 236],[235, 238, 235],[235, 237, 234],[235, 238, 233],[222, 224, 218],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[137, 140, 135],[227, 232, 222],[225, 230, 222],[209, 214, 205],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[177, 182, 173],[219, 226, 214],[29, 30, 28],[0, 0, 0],[0, 0, 0],[0, 0, 0],[155, 160, 147],[208, 216, 199],[205, 213, 196],[203, 211, 194],[202, 211, 196],[202, 212, 196],[203, 212, 197],[162, 170, 157],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[180, 186, 170],[209, 216, 198],[113, 116, 107],[0, 0, 0],[0, 0, 0],[0, 0, 0],[58, 59, 55],[222, 225, 210],[30, 30, 28],[0, 0, 0],[0, 0, 0],[0, 0, 0],[173, 173, 163],[237, 237, 222],[238, 237, 222],[237, 237, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[206, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 173, 163],[237, 237, 222],[237, 237, 222],[237, 237, 222],[237, 237, 222],[190, 190, 178],[16, 16, 15],[0, 0, 0],[0, 0, 0],[111, 111, 105],[174, 174, 164],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 222],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 133],[236, 236, 222],[236, 236, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[237, 237, 222],[174, 174, 163],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[205, 205, 192],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[173, 173, 163],[236, 236, 222],[236, 236, 222],[228, 228, 216],[220, 220, 210],[211, 211, 204],[204, 204, 198],[197, 197, 193],[192, 192, 189],[188, 188, 186],[185, 185, 184],[183, 183, 183],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[178, 178, 178],[178, 178, 178],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[179, 179, 179],[180, 180, 180],[180, 180, 180],[181, 181, 181],[182, 182, 182],[183, 183, 183],[185, 185, 185],[186, 186, 185],[189, 189, 188],[191, 191, 189],[194, 194, 192],[200, 200, 196],[206, 206, 200],[214, 214, 206],[222, 222, 212],[230, 230, 219],[205, 205, 194],[0, 0, 0],[0, 0, 0],[126, 126, 119],[237, 237, 224],[237, 237, 224],[111, 111, 105],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 134],[237, 237, 224],[237, 237, 224],[237, 237, 224],[238, 238, 224],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[238, 238, 225],[175, 175, 166],[79, 79, 75],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 64, 61],[239, 239, 227],[239, 239, 227],[96, 96, 91],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 91],[239, 239, 227],[239, 239, 227],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 166],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[128, 128, 122],[240, 240, 229],[240, 240, 229],[237, 237, 228],[236, 236, 228],[237, 237, 228],[238, 238, 229],[240, 240, 231],[113, 113, 108],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[236, 236, 221],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 219],[235, 235, 219],[235, 235, 220],[235, 235, 220],[172, 172, 161],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[236, 236, 222],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[223, 223, 212],[239, 239, 227],[80, 80, 76],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 31],[241, 241, 230],[242, 242, 231],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[146, 146, 142],[244, 244, 236],[131, 131, 126],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 32],[246, 246, 239],[82, 82, 80],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[247, 248, 244],[247, 248, 244],[148, 148, 146],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 64],[243, 244, 239],[242, 243, 238],[239, 240, 235],[236, 238, 234],[236, 238, 233],[237, 239, 232],[158, 159, 154],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[46, 47, 44],[227, 231, 222],[225, 230, 220],[30, 31, 29],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[116, 120, 113],[29, 30, 28],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[27, 28, 26],[200, 210, 193],[174, 183, 167],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[183, 189, 176],[212, 220, 205],[115, 118, 110],[0, 0, 0],[0, 0, 0],[0, 0, 0],[59, 60, 56],[225, 228, 215],[30, 31, 29],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[238, 238, 225],[206, 205, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 164],[237, 237, 223],[237, 237, 223],[237, 237, 223],[237, 237, 223],[237, 237, 224],[205, 205, 194],[111, 111, 105],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 222],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 133],[236, 236, 222],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 148],[237, 237, 222],[126, 126, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[205, 205, 192],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 163],[237, 237, 222],[237, 237, 222],[231, 231, 218],[223, 223, 212],[214, 214, 206],[206, 206, 200],[199, 199, 194],[193, 193, 190],[189, 189, 187],[186, 186, 184],[184, 184, 183],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[179, 179, 179],[179, 179, 179],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[181, 181, 181],[183, 183, 183],[183, 183, 183],[184, 184, 184],[186, 186, 185],[186, 186, 186],[188, 188, 187],[190, 190, 189],[194, 194, 192],[198, 198, 194],[204, 204, 199],[211, 211, 204],[219, 219, 210],[229, 229, 217],[237, 237, 224],[95, 95, 90],[0, 0, 0],[95, 95, 90],[237, 237, 224],[237, 237, 224],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[238, 238, 225],[238, 238, 226],[238, 238, 226],[239, 239, 226],[239, 239, 226],[127, 127, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 64, 61],[239, 239, 227],[239, 239, 227],[96, 96, 91],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 91],[239, 239, 227],[207, 207, 197],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[191, 191, 182],[239, 239, 227],[64, 64, 61],[32, 32, 30],[208, 208, 198],[240, 240, 228],[240, 240, 228],[192, 192, 182],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 92],[208, 208, 198],[240, 240, 229],[241, 241, 229],[240, 240, 230],[237, 237, 228],[236, 236, 227],[236, 236, 228],[238, 238, 228],[240, 240, 230],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 148],[236, 236, 222],[236, 236, 221],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[172, 172, 161],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[236, 236, 222],[205, 205, 193],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[239, 239, 227],[240, 240, 228],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[80, 80, 77],[242, 242, 232],[243, 243, 234],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[146, 146, 142],[245, 245, 237],[98, 98, 95],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 32],[247, 247, 241],[132, 132, 129],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[197, 198, 195],[246, 248, 244],[131, 132, 130],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 64],[241, 243, 239],[240, 242, 238],[237, 240, 235],[235, 237, 234],[235, 237, 233],[235, 238, 232],[125, 127, 123],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[105, 108, 104],[225, 230, 221],[223, 228, 218],[133, 136, 131],[44, 45, 43],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[58, 60, 57],[158, 164, 155],[71, 74, 70],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[187, 195, 181],[174, 182, 167],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[187, 192, 178],[218, 223, 207],[88, 90, 84],[0, 0, 0],[0, 0, 0],[0, 0, 0],[75, 76, 71],[228, 230, 215],[77, 77, 73],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[222, 222, 210],[206, 206, 194],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 149],[111, 111, 104],[0, 0, 0],[32, 32, 30],[190, 190, 179],[237, 237, 224],[237, 237, 224],[238, 238, 224],[238, 238, 224],[237, 237, 224],[174, 174, 164],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 222],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 133],[237, 237, 222],[237, 237, 222],[237, 237, 222],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 163],[237, 237, 222],[79, 79, 74],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[236, 236, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[158, 158, 148],[111, 111, 104],[0, 0, 0],[31, 31, 29],[179, 179, 170],[215, 215, 207],[207, 207, 201],[201, 201, 196],[195, 195, 192],[191, 191, 188],[187, 187, 185],[184, 184, 183],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[180, 180, 180],[180, 180, 180],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[182, 182, 182],[182, 182, 182],[183, 183, 183],[184, 184, 184],[184, 184, 184],[185, 185, 185],[186, 186, 186],[187, 187, 187],[188, 188, 187],[189, 189, 188],[192, 192, 191],[197, 197, 194],[202, 202, 198],[209, 209, 203],[217, 217, 209],[226, 226, 215],[235, 235, 222],[190, 190, 179],[0, 0, 0],[0, 0, 0],[205, 205, 194],[126, 126, 119],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 119],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[175, 175, 165],[0, 0, 0],[16, 16, 15],[143, 143, 136],[239, 239, 227],[80, 80, 76],[0, 0, 0],[0, 0, 0],[0, 0, 0],[96, 96, 91],[240, 240, 228],[191, 191, 182],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[239, 239, 227],[143, 143, 136],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[223, 223, 212],[240, 240, 228],[128, 128, 122],[0, 0, 0],[32, 32, 30],[224, 224, 214],[240, 240, 229],[192, 192, 183],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[240, 240, 229],[160, 160, 153],[16, 16, 15],[32, 32, 31],[238, 238, 228],[236, 236, 227],[235, 235, 226],[235, 235, 226],[236, 236, 227],[239, 239, 229],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[190, 190, 178],[236, 236, 222],[236, 236, 222],[236, 236, 221],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[235, 235, 220],[172, 172, 161],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 222],[237, 237, 224],[111, 111, 105],[0, 0, 0],[0, 0, 0],[0, 0, 0],[208, 208, 198],[145, 145, 137],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[178, 178, 172],[243, 243, 234],[163, 163, 157],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[147, 147, 142],[212, 212, 205],[33, 33, 32],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 32],[247, 247, 241],[231, 231, 225],[16, 16, 16],[0, 0, 0],[0, 0, 0],[0, 0, 0],[82, 83, 81],[246, 247, 243],[66, 66, 65],[0, 0, 0],[0, 0, 0],[0, 0, 0],[65, 65, 64],[241, 242, 237],[239, 241, 236],[235, 238, 232],[233, 235, 230],[233, 235, 229],[234, 237, 229],[124, 126, 122],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[135, 137, 131],[224, 228, 219],[221, 227, 215],[220, 226, 215],[219, 225, 213],[218, 224, 213],[218, 224, 213],[218, 224, 212],[217, 223, 211],[216, 223, 210],[215, 222, 210],[170, 176, 166],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 130, 121],[209, 216, 201],[55, 57, 53],[0, 0, 0],[0, 0, 0],[53, 56, 52],[200, 210, 194],[94, 98, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[132, 134, 126],[221, 226, 213],[30, 30, 29],[0, 0, 0],[0, 0, 0],[0, 0, 0],[106, 107, 101],[230, 232, 219],[186, 186, 176],[0, 0, 0],[0, 0, 0],[0, 0, 0],[143, 143, 135],[238, 238, 225],[63, 63, 60],[0, 0, 0],[0, 0, 0],[63, 63, 60],[238, 238, 225],[127, 127, 120],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[47, 47, 45],[222, 222, 209],[16, 16, 15],[0, 0, 0],[111, 111, 105],[238, 238, 224],[126, 126, 119],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[237, 237, 222],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[237, 237, 222],[237, 237, 222],[237, 237, 222],[158, 158, 148],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 133],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 118],[237, 237, 222],[126, 126, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[45, 45, 43],[216, 216, 208],[208, 208, 202],[201, 201, 196],[195, 195, 192],[191, 191, 189],[187, 187, 186],[185, 185, 184],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[181, 181, 181],[181, 181, 181],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[184, 184, 184],[185, 185, 185],[186, 186, 186],[186, 186, 186],[187, 187, 187],[188, 188, 187],[187, 187, 187],[189, 189, 188],[191, 191, 189],[195, 195, 192],[200, 200, 196],[206, 206, 200],[213, 213, 206],[222, 222, 212],[231, 231, 219],[237, 237, 224],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[221, 221, 209],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[206, 206, 195],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 165],[238, 238, 225],[206, 206, 196],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[176, 176, 168],[240, 240, 229],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 64, 61],[240, 240, 228],[240, 240, 228],[240, 240, 228],[64, 64, 61],[0, 0, 0],[96, 96, 92],[240, 240, 229],[240, 240, 229],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[48, 48, 45],[235, 235, 226],[233, 233, 224],[232, 232, 224],[233, 233, 225],[235, 235, 226],[238, 238, 228],[112, 112, 107],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[205, 205, 194],[237, 237, 222],[236, 236, 222],[236, 236, 222],[236, 236, 222],[205, 205, 192],[16, 16, 15],[0, 0, 0],[157, 157, 147],[236, 236, 221],[142, 142, 133],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[237, 237, 224],[238, 238, 225],[238, 238, 226],[80, 80, 76],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[130, 130, 126],[245, 245, 237],[180, 180, 174],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[147, 147, 143],[65, 65, 63],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[33, 33, 32],[247, 247, 241],[247, 247, 241],[180, 182, 178],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 65, 64],[239, 241, 236],[236, 239, 234],[231, 234, 229],[228, 231, 227],[228, 231, 226],[230, 233, 226],[123, 125, 121],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[119, 122, 116],[222, 227, 216],[219, 225, 213],[218, 224, 213],[217, 223, 211],[216, 223, 210],[216, 223, 210],[216, 223, 210],[216, 222, 209],[215, 222, 210],[214, 221, 211],[212, 219, 208],[127, 131, 123],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[13, 14, 13],[160, 167, 155],[146, 153, 142],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[169, 170, 161],[231, 234, 220],[234, 235, 223],[142, 142, 134],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[190, 190, 182],[206, 206, 196],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[238, 237, 224],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 119],[237, 237, 222],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 163],[237, 237, 222],[126, 126, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[237, 237, 222],[205, 205, 192],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 63, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[15, 15, 14],[215, 215, 207],[208, 208, 201],[201, 201, 197],[195, 195, 192],[191, 191, 189],[187, 187, 186],[185, 185, 184],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[184, 184, 184],[184, 184, 184],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 187],[190, 190, 189],[193, 193, 191],[197, 197, 194],[203, 203, 198],[210, 210, 203],[217, 217, 209],[226, 226, 216],[235, 235, 222],[142, 142, 134],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[126, 126, 119],[237, 237, 224],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 225],[190, 190, 180],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[175, 175, 166],[238, 238, 226],[238, 238, 226],[239, 239, 226],[127, 127, 121],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[80, 80, 76],[176, 176, 168],[240, 240, 229],[240, 240, 229],[240, 240, 229],[144, 144, 137],[128, 128, 122],[32, 32, 30],[0, 0, 0],[112, 112, 106],[192, 192, 182],[48, 48, 46],[0, 0, 0],[0, 0, 0],[80, 80, 76],[224, 224, 213],[240, 240, 228],[240, 240, 228],[240, 240, 229],[96, 96, 92],[0, 0, 0],[0, 0, 0],[224, 224, 214],[240, 240, 229],[224, 224, 214],[96, 96, 92],[0, 0, 0],[0, 0, 0],[0, 0, 0],[48, 48, 46],[16, 16, 15],[186, 186, 179],[231, 231, 223],[229, 229, 222],[229, 229, 222],[230, 230, 222],[232, 232, 224],[235, 235, 226],[175, 175, 167],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 134],[237, 237, 223],[237, 237, 223],[236, 236, 222],[236, 236, 222],[63, 63, 59],[0, 0, 0],[0, 0, 0],[47, 47, 44],[236, 236, 222],[126, 126, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[238, 238, 225],[239, 239, 226],[239, 239, 227],[240, 240, 229],[177, 177, 169],[96, 96, 92],[0, 0, 0],[0, 0, 0],[0, 0, 0],[82, 82, 79],[180, 180, 175],[246, 246, 239],[246, 246, 239],[246, 246, 239],[131, 131, 128],[115, 115, 112],[0, 0, 0],[0, 0, 0],[0, 0, 0],[148, 148, 143],[212, 212, 206],[131, 131, 127],[49, 49, 48],[0, 0, 0],[0, 0, 0],[66, 66, 64],[247, 247, 241],[247, 247, 241],[247, 248, 243],[198, 198, 195],[82, 83, 81],[0, 0, 0],[0, 0, 0],[65, 66, 65],[65, 66, 65],[0, 0, 0],[0, 0, 0],[0, 0, 0],[64, 65, 63],[238, 240, 234],[232, 235, 230],[226, 230, 226],[223, 226, 223],[223, 226, 221],[223, 227, 220],[213, 217, 207],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[59, 60, 57],[219, 224, 212],[217, 223, 210],[216, 222, 210],[215, 221, 208],[200, 206, 193],[57, 59, 55],[0, 0, 0],[86, 88, 83],[214, 220, 208],[213, 220, 210],[212, 219, 207],[211, 217, 203],[169, 174, 162],[84, 86, 80],[0, 0, 0],[0, 0, 0],[0, 0, 0],[95, 99, 94],[175, 183, 171],[199, 209, 193],[198, 208, 191],[173, 180, 164],[109, 113, 104],[85, 87, 82],[0, 0, 0],[0, 0, 0],[211, 214, 203],[121, 123, 117],[0, 0, 0],[0, 0, 0],[15, 16, 15],[154, 155, 148],[233, 234, 224],[233, 235, 223],[235, 236, 226],[237, 237, 226],[190, 190, 182],[95, 96, 91],[0, 0, 0],[0, 0, 0],[0, 0, 0],[112, 112, 106],[207, 207, 197],[238, 239, 227],[238, 238, 227],[175, 175, 166],[127, 127, 121],[79, 79, 75],[0, 0, 0],[0, 0, 0],[190, 190, 180],[126, 126, 120],[0, 0, 0],[0, 0, 0],[175, 175, 164],[238, 238, 225],[238, 238, 225],[143, 143, 135],[95, 95, 90],[0, 0, 0],[0, 0, 0],[0, 0, 0],[47, 47, 45],[142, 142, 134],[237, 237, 224],[237, 237, 222],[142, 142, 133],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[174, 174, 163],[237, 237, 222],[237, 237, 222],[205, 205, 192],[126, 126, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[47, 47, 44],[142, 142, 133],[237, 237, 222],[237, 237, 222],[237, 237, 222],[174, 174, 163],[126, 126, 118],[79, 79, 74],[0, 0, 0],[0, 0, 0],[190, 190, 178],[125, 125, 118],[0, 0, 0],[0, 0, 0],[161, 161, 154],[212, 212, 205],[205, 205, 200],[200, 200, 196],[195, 195, 192],[190, 190, 188],[187, 187, 186],[185, 185, 184],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[180, 180, 180],[180, 180, 180],[180, 180, 180],
	[182, 182, 182],[182, 182, 182],[183, 183, 183],[184, 184, 184],[184, 184, 184],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[190, 190, 189],[192, 192, 190],[196, 196, 193],[201, 201, 197],[207, 207, 201],[214, 214, 206],[222, 222, 212],[230, 230, 218],[221, 221, 209],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[221, 221, 209],[238, 238, 224],[238, 238, 224],[238, 238, 224],[238, 238, 225],[222, 222, 210],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[143, 143, 136],[238, 238, 226],[239, 239, 226],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 227],[239, 239, 228],[238, 238, 227],[238, 238, 228],[238, 238, 228],[238, 238, 227],[237, 237, 227],[237, 237, 226],[237, 237, 226],[238, 238, 227],[239, 239, 228],[239, 239, 228],[239, 239, 228],[239, 239, 228],[240, 240, 229],[240, 240, 229],[240, 240, 229],[240, 240, 229],[240, 240, 229],[240, 240, 229],[240, 240, 229],[128, 128, 122],[0, 0, 0],[0, 0, 0],[0, 0, 0],[160, 160, 153],[240, 240, 230],[240, 240, 230],[240, 240, 230],[241, 241, 231],[239, 239, 230],[236, 236, 226],[233, 233, 224],[230, 231, 223],[228, 228, 221],[225, 226, 219],[224, 225, 218],[224, 225, 218],[225, 226, 219],[227, 228, 221],[231, 232, 223],[235, 235, 227],[64, 64, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[32, 32, 30],[190, 190, 178],[237, 237, 222],[189, 190, 178],[63, 63, 59],[0, 0, 0],[0, 0, 0],[0, 0, 0],[31, 32, 30],[190, 190, 179],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[207, 207, 196],[239, 239, 227],[240, 240, 229],[241, 241, 231],[242, 242, 233],[243, 243, 234],[243, 243, 236],[244, 244, 238],[244, 244, 239],[245, 245, 240],[246, 246, 242],[247, 247, 243],[246, 247, 242],[246, 247, 242],[246, 247, 242],[246, 247, 242],[246, 247, 241],[246, 246, 240],[246, 246, 239],[245, 246, 239],[245, 245, 239],[245, 245, 239],[245, 246, 239],[245, 245, 239],[246, 246, 241],[246, 246, 241],[246, 247, 242],[247, 247, 243],[247, 248, 244],[247, 248, 244],[246, 247, 243],[246, 247, 243],[246, 247, 243],[245, 246, 241],[114, 115, 112],[0, 0, 0],[0, 0, 0],[0, 0, 0],[63, 64, 63],[232, 236, 234],[223, 229, 227],[217, 224, 224],[214, 221, 220],[212, 219, 217],[214, 221, 216],[218, 223, 216],[90, 92, 87],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[101, 104, 99],[214, 221, 208],[213, 220, 207],[156, 161, 152],[42, 44, 41],[0, 0, 0],[0, 0, 0],[0, 0, 0],[199, 205, 192],[212, 220, 208],[211, 219, 206],[210, 217, 202],[208, 215, 199],[203, 212, 197],[201, 210, 196],[197, 207, 197],[194, 205, 195],[192, 204, 195],[191, 202, 191],[189, 201, 188],[188, 198, 184],[191, 200, 182],[200, 209, 193],[211, 219, 207],[217, 223, 211],[225, 227, 212],[228, 229, 216],[229, 231, 219],[229, 231, 220],[229, 231, 219],[229, 230, 219],[229, 230, 217],[230, 230, 218],[230, 230, 219],[231, 231, 221],[232, 232, 221],[234, 234, 222],[234, 234, 222],[234, 234, 222],[235, 235, 223],[236, 236, 223],[236, 236, 225],[236, 236, 225],[236, 236, 224],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[236, 236, 223],[235, 235, 222],[235, 235, 222],[235, 235, 222],[235, 235, 223],[235, 235, 223],[235, 235, 223],[235, 235, 223],[236, 236, 224],[236, 236, 224],[236, 236, 224],[236, 236, 223],[237, 237, 223],[237, 237, 224],[237, 237, 224],[237, 237, 224],[237, 237, 224],[237, 237, 224],[237, 237, 223],[95, 95, 89],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 134],[237, 237, 223],[237, 237, 223],[237, 237, 223],[237, 237, 223],[236, 236, 223],[235, 235, 222],[234, 234, 222],[234, 234, 221],[234, 234, 221],[234, 234, 221],[234, 234, 220],[234, 234, 220],[234, 234, 220],[233, 233, 220],[233, 233, 220],[234, 234, 220],[234, 234, 221],[234, 234, 221],[234, 234, 221],[232, 232, 220],[230, 230, 218],[227, 227, 216],[224, 224, 214],[219, 219, 210],[214, 214, 207],[209, 209, 202],[203, 203, 199],[198, 198, 194],[194, 194, 191],[190, 190, 189],[187, 187, 185],[185, 185, 184],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],
	[183, 183, 183],[183, 183, 183],[184, 184, 184],[184, 184, 184],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[189, 189, 189],[191, 191, 189],[194, 194, 192],[198, 198, 195],[204, 204, 199],[210, 210, 204],[217, 217, 209],[225, 225, 214],[232, 232, 220],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[142, 142, 134],[237, 237, 224],[235, 235, 222],[233, 233, 221],[232, 232, 220],[232, 232, 221],[124, 124, 118],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[127, 127, 121],[238, 238, 225],[236, 236, 224],[235, 235, 224],[233, 233, 223],[232, 232, 222],[230, 230, 221],[230, 230, 221],[230, 230, 221],[229, 229, 221],[230, 230, 221],[229, 229, 221],[229, 229, 220],[228, 228, 220],[229, 229, 220],[229, 229, 221],[230, 230, 221],[230, 230, 221],[230, 230, 222],[231, 231, 222],[232, 232, 223],[233, 233, 224],[235, 235, 225],[236, 236, 226],[237, 237, 227],[238, 238, 228],[240, 240, 229],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[145, 145, 139],[240, 240, 231],[239, 239, 230],[237, 237, 228],[235, 235, 227],[232, 232, 225],[230, 230, 222],[227, 227, 220],[224, 225, 218],[221, 223, 216],[220, 221, 215],[219, 220, 214],[219, 220, 214],[221, 222, 216],[223, 224, 217],[226, 227, 219],[231, 231, 224],[204, 204, 197],[32, 32, 30],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[95, 95, 89],[63, 63, 60],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[192, 192, 182],[240, 240, 229],[239, 239, 229],[238, 238, 229],[238, 238, 232],[238, 238, 233],[240, 240, 235],[241, 241, 237],[242, 242, 239],[243, 243, 241],[245, 245, 243],[246, 246, 244],[245, 246, 243],[245, 246, 242],[244, 245, 241],[244, 245, 241],[243, 243, 239],[242, 243, 238],[241, 242, 237],[241, 242, 237],[240, 241, 237],[240, 241, 237],[240, 240, 237],[240, 241, 237],[241, 241, 237],[241, 241, 238],[242, 242, 239],[243, 243, 240],[243, 245, 242],[244, 245, 242],[244, 246, 242],[245, 246, 242],[245, 246, 241],[65, 66, 64],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[62, 63, 62],[225, 230, 229],[215, 223, 222],[209, 217, 218],[205, 214, 214],[203, 211, 211],[204, 211, 208],[207, 213, 207],[199, 204, 196],[44, 45, 43],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[199, 204, 191],[207, 215, 204],[198, 207, 198],[190, 200, 192],[182, 193, 185],[176, 189, 182],[172, 185, 179],[167, 182, 180],[164, 181, 179],[161, 178, 179],[159, 176, 176],[157, 174, 172],[156, 171, 170],[161, 175, 168],[174, 187, 181],[189, 202, 199],[198, 207, 203],[210, 214, 205],[214, 217, 210],[217, 220, 213],[218, 221, 215],[219, 222, 214],[220, 221, 214],[220, 221, 213],[221, 221, 214],[221, 221, 213],[222, 222, 214],[223, 223, 215],[224, 224, 215],[225, 225, 215],[225, 225, 215],[226, 226, 216],[227, 227, 217],[227, 227, 218],[226, 226, 217],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 215],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[226, 226, 216],[227, 227, 216],[227, 227, 216],[228, 228, 217],[229, 229, 218],[230, 230, 219],[232, 232, 219],[233, 233, 220],[186, 186, 177],[62, 62, 59],[0, 0, 0],[140, 140, 132],[232, 232, 219],[230, 230, 218],[228, 228, 217],[227, 227, 216],[226, 226, 215],[225, 225, 214],[224, 224, 214],[223, 223, 213],[223, 223, 213],[223, 223, 213],[223, 223, 212],[223, 223, 212],[223, 223, 212],[222, 222, 212],[223, 223, 212],[223, 223, 212],[224, 224, 213],[224, 224, 214],[223, 223, 213],[223, 223, 213],[221, 221, 211],[219, 219, 210],[217, 217, 208],[213, 213, 206],[209, 209, 203],[205, 205, 199],[200, 200, 196],[195, 195, 192],[192, 192, 190],[189, 189, 188],[187, 187, 186],[184, 184, 183],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],[181, 181, 181],
	[184, 184, 184],[184, 184, 184],[184, 184, 184],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[189, 189, 188],[190, 190, 189],[192, 192, 191],[196, 196, 194],[201, 201, 197],[207, 207, 201],[213, 213, 206],[220, 220, 211],[226, 226, 216],[15, 15, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[78, 78, 74],[233, 233, 220],[230, 230, 219],[228, 228, 218],[226, 226, 216],[225, 225, 216],[225, 225, 216],[227, 227, 217],[137, 137, 131],[123, 123, 117],[46, 46, 44],[0, 0, 0],[0, 0, 0],[139, 139, 133],[230, 230, 220],[228, 228, 218],[226, 226, 218],[225, 225, 217],[224, 224, 216],[222, 222, 215],[222, 222, 215],[222, 222, 215],[222, 222, 215],[222, 222, 216],[222, 222, 216],[222, 222, 215],[221, 221, 215],[221, 221, 215],[221, 221, 215],[222, 222, 215],[222, 222, 215],[222, 222, 216],[223, 223, 216],[224, 224, 217],[226, 226, 218],[227, 227, 219],[230, 230, 221],[231, 231, 222],[233, 233, 223],[235, 235, 225],[16, 16, 15],[0, 0, 0],[0, 0, 0],[0, 0, 0],[189, 189, 182],[235, 235, 226],[233, 233, 225],[231, 231, 223],[229, 229, 221],[226, 226, 219],[223, 223, 217],[221, 221, 216],[218, 219, 214],[216, 218, 213],[215, 216, 212],[214, 216, 211],[215, 216, 212],[216, 218, 213],[218, 220, 213],[222, 223, 216],[226, 226, 220],[230, 230, 223],[203, 203, 196],[63, 63, 61],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[16, 16, 15],[205, 205, 194],[237, 237, 225],[126, 127, 120],[127, 127, 121],[16, 16, 15],[0, 0, 0],[0, 0, 0],[204, 205, 197],[234, 235, 227],[234, 234, 228],[233, 234, 229],[236, 236, 232],[237, 237, 234],[239, 239, 237],[241, 241, 239],[242, 242, 241],[243, 243, 243],[245, 245, 245],[246, 246, 245],[246, 247, 245],[245, 246, 244],[244, 245, 243],[242, 244, 241],[241, 242, 240],[240, 240, 238],[239, 239, 237],[237, 238, 236],[236, 237, 234],[236, 237, 234],[236, 237, 234],[236, 237, 234],[237, 237, 236],[237, 238, 237],[238, 238, 237],[239, 239, 239],[239, 241, 240],[239, 241, 240],[239, 241, 239],[239, 241, 238],[240, 242, 239],[239, 241, 237],[127, 128, 125],[126, 127, 125],[16, 16, 15],[0, 0, 0],[74, 76, 75],[216, 222, 222],[205, 215, 215],[199, 208, 211],[194, 205, 205],[191, 201, 204],[190, 201, 201],[192, 201, 201],[198, 207, 202],[192, 199, 194],[57, 59, 56],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[55, 58, 54],[198, 207, 200],[185, 198, 196],[175, 190, 190],[164, 181, 181],[155, 172, 173],[146, 165, 168],[142, 162, 165],[136, 158, 164],[134, 155, 164],[132, 155, 165],[130, 152, 161],[129, 149, 157],[129, 148, 153],[138, 155, 156],[157, 171, 174],[175, 188, 190],[187, 196, 196],[198, 203, 199],[204, 208, 206],[207, 211, 209],[208, 212, 211],[210, 214, 210],[212, 213, 210],[211, 214, 209],[215, 214, 211],[215, 215, 210],[215, 215, 210],[216, 216, 210],[217, 217, 211],[218, 218, 210],[218, 218, 210],[218, 218, 210],[220, 220, 212],[218, 218, 211],[218, 218, 211],[217, 217, 210],[217, 217, 209],[216, 216, 208],[217, 217, 209],[217, 217, 209],[217, 217, 209],[217, 217, 209],[217, 217, 209],[217, 217, 210],[218, 218, 210],[218, 218, 210],[217, 217, 210],[217, 217, 210],[217, 217, 210],[217, 217, 210],[217, 217, 210],[217, 217, 210],[217, 217, 210],[217, 217, 209],[218, 218, 209],[218, 218, 210],[219, 219, 211],[220, 220, 211],[221, 221, 211],[222, 222, 212],[224, 224, 214],[224, 224, 214],[224, 224, 214],[224, 224, 214],[224, 224, 214],[222, 222, 213],[221, 221, 212],[220, 220, 211],[218, 218, 210],[217, 217, 209],[216, 216, 208],[215, 215, 207],[215, 215, 207],[214, 214, 207],[214, 214, 206],[214, 214, 206],[214, 214, 206],[213, 213, 205],[213, 213, 206],[213, 213, 206],[214, 214, 206],[214, 214, 207],[215, 215, 207],[215, 215, 207],[214, 214, 206],[214, 214, 206],[212, 212, 205],[210, 210, 203],[207, 207, 201],[204, 204, 199],[200, 200, 196],[197, 197, 194],[193, 193, 191],[190, 190, 189],[188, 188, 187],[186, 186, 185],[185, 185, 184],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],
	[185, 185, 185],[185, 185, 185],[185, 185, 185],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[190, 190, 189],[192, 192, 191],[195, 195, 193],[199, 199, 196],[204, 204, 200],[209, 209, 203],[214, 214, 207],[220, 220, 211],[164, 164, 157],[15, 15, 14],[0, 0, 0],[91, 91, 87],[212, 212, 202],[225, 225, 216],[223, 223, 214],[221, 221, 212],[219, 219, 211],[218, 218, 211],[219, 219, 211],[220, 220, 212],[222, 222, 213],[223, 223, 215],[224, 224, 215],[225, 225, 216],[225, 225, 216],[224, 224, 215],[222, 222, 214],[221, 221, 213],[219, 219, 212],[217, 217, 211],[216, 216, 210],[216, 216, 210],[216, 216, 211],[215, 215, 210],[215, 215, 210],[216, 216, 211],[216, 216, 211],[215, 215, 211],[215, 215, 211],[215, 215, 210],[215, 215, 211],[216, 216, 211],[216, 216, 211],[216, 216, 211],[217, 217, 212],[218, 218, 213],[220, 220, 214],[221, 221, 215],[223, 223, 216],[225, 225, 218],[227, 227, 219],[229, 229, 221],[154, 154, 148],[0, 0, 0],[0, 0, 0],[124, 124, 119],[231, 231, 223],[229, 229, 221],[227, 227, 220],[225, 225, 219],[223, 223, 217],[221, 221, 216],[218, 218, 214],[216, 216, 212],[213, 214, 210],[212, 214, 210],[212, 213, 209],[210, 212, 208],[211, 213, 209],[212, 214, 210],[214, 215, 210],[217, 218, 212],[220, 221, 216],[224, 225, 219],[228, 228, 221],[231, 231, 223],[155, 156, 149],[78, 79, 75],[0, 0, 0],[0, 0, 0],[0, 0, 0],[95, 95, 89],[173, 174, 164],[236, 237, 223],[236, 236, 223],[235, 235, 223],[235, 235, 223],[234, 234, 223],[234, 234, 224],[233, 233, 223],[232, 232, 224],[231, 231, 224],[230, 231, 225],[230, 230, 225],[230, 231, 227],[231, 231, 228],[235, 235, 233],[237, 237, 235],[239, 239, 238],[242, 242, 241],[243, 243, 243],[243, 243, 243],[245, 245, 245],[245, 245, 246],[245, 247, 245],[245, 246, 244],[244, 245, 243],[242, 243, 241],[240, 241, 241],[238, 238, 238],[236, 236, 235],[234, 235, 234],[234, 234, 233],[234, 234, 233],[234, 234, 233],[234, 234, 233],[233, 234, 235],[234, 235, 235],[234, 235, 235],[235, 236, 236],[235, 237, 238],[234, 236, 237],[233, 236, 235],[233, 236, 234],[233, 235, 234],[231, 234, 231],[229, 232, 229],[227, 230, 229],[223, 227, 225],[218, 223, 221],[211, 219, 217],[205, 212, 215],[194, 205, 207],[188, 199, 204],[182, 194, 197],[178, 190, 195],[176, 189, 191],[177, 188, 191],[182, 193, 192],[189, 199, 197],[195, 205, 198],[136, 141, 135],[84, 87, 82],[0, 0, 0],[0, 0, 0],[0, 0, 0],[0, 0, 0],[28, 29, 27],[27, 28, 26],[13, 14, 13],[150, 158, 154],[176, 189, 187],[163, 180, 184],[152, 171, 179],[143, 163, 170],[131, 152, 161],[122, 145, 154],[115, 139, 150],[109, 135, 148],[107, 132, 150],[106, 133, 152],[104, 131, 149],[104, 127, 144],[106, 128, 143],[120, 140, 150],[142, 159, 169],[162, 178, 186],[176, 187, 192],[187, 194, 193],[195, 199, 201],[199, 203, 204],[200, 205, 207],[203, 208, 207],[206, 208, 208],[206, 209, 206],[210, 209, 208],[210, 210, 207],[210, 210, 207],[211, 211, 206],[211, 211, 206],[211, 211, 204],[211, 211, 204],[212, 212, 206],[212, 212, 206],[210, 210, 205],[210, 210, 205],[210, 210, 204],[209, 209, 203],[208, 208, 202],[209, 209, 203],[209, 209, 203],[209, 209, 203],[210, 210, 204],[210, 210, 204],[210, 210, 204],[210, 210, 204],[211, 211, 205],[210, 210, 205],[210, 210, 205],[210, 210, 204],[210, 210, 204],[210, 210, 204],[210, 210, 204],[210, 210, 204],[210, 210, 204],[210, 210, 204],[211, 211, 205],[211, 211, 205],[212, 212, 205],[213, 213, 206],[214, 214, 207],[214, 214, 207],[215, 215, 207],[215, 215, 207],[215, 215, 207],[215, 215, 207],[214, 214, 206],[213, 213, 205],[211, 211, 204],[210, 210, 203],[209, 209, 202],[208, 208, 202],[207, 207, 201],[207, 207, 201],[207, 207, 201],[206, 206, 201],[206, 206, 201],[205, 205, 200],[205, 205, 200],[205, 205, 200],[205, 205, 200],[206, 206, 201],[207, 207, 201],[207, 207, 201],[207, 207, 201],[207, 207, 201],[206, 206, 200],[205, 205, 200],[203, 203, 199],[202, 202, 197],[199, 199, 195],[197, 197, 194],[194, 194, 191],[192, 192, 190],[189, 189, 187],[187, 187, 186],[186, 186, 185],[184, 184, 184],[184, 184, 183],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],
	[185, 185, 185],[186, 186, 186],[186, 186, 186],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 188],[191, 191, 190],[194, 194, 192],[197, 197, 194],[201, 201, 197],[205, 205, 200],[209, 209, 204],[214, 214, 207],[217, 217, 210],[220, 220, 211],[221, 221, 212],[220, 220, 212],[219, 219, 211],[218, 218, 211],[216, 216, 209],[214, 214, 208],[213, 213, 207],[213, 213, 206],[213, 213, 207],[214, 214, 208],[215, 215, 209],[216, 216, 209],[217, 217, 210],[218, 218, 211],[218, 218, 211],[217, 217, 210],[215, 215, 209],[214, 214, 208],[212, 212, 207],[211, 211, 207],[210, 210, 206],[209, 209, 205],[209, 209, 206],[209, 209, 206],[210, 210, 206],[210, 210, 206],[210, 210, 207],[210, 210, 207],[210, 210, 207],[210, 210, 207],[210, 210, 206],[210, 210, 207],[210, 210, 207],[210, 210, 207],[211, 211, 207],[212, 212, 208],[214, 214, 209],[215, 215, 210],[217, 217, 212],[219, 219, 213],[220, 220, 214],[223, 223, 216],[224, 224, 217],[225, 225, 218],[226, 226, 219],[226, 226, 219],[224, 224, 218],[223, 223, 217],[221, 221, 215],[220, 220, 215],[218, 218, 213],[216, 216, 212],[213, 213, 210],[212, 212, 209],[209, 211, 208],[208, 210, 207],[208, 209, 206],[208, 209, 206],[208, 208, 208],[209, 210, 209],[210, 211, 209],[213, 213, 211],[215, 215, 213],[218, 219, 216],[221, 222, 216],[224, 225, 218],[226, 226, 218],[227, 228, 218],[228, 228, 217],[228, 229, 217],[228, 229, 217],[228, 229, 217],[228, 228, 218],[227, 228, 217],[227, 227, 217],[226, 226, 216],[225, 226, 217],[224, 225, 217],[223, 225, 219],[223, 225, 219],[223, 225, 221],[223, 225, 222],[223, 225, 223],[225, 227, 226],[227, 229, 229],[228, 230, 231],[231, 234, 234],[234, 237, 237],[238, 238, 240],[240, 240, 243],[241, 242, 244],[241, 242, 244],[242, 242, 245],[242, 242, 245],[242, 242, 243],[242, 243, 244],[242, 243, 244],[242, 243, 243],[240, 241, 242],[237, 238, 238],[234, 235, 237],[232, 233, 234],[231, 232, 233],[231, 232, 233],[231, 232, 233],[231, 232, 233],[231, 232, 233],[231, 232, 233],[231, 232, 233],[232, 233, 235],[229, 233, 234],[229, 232, 233],[228, 231, 233],[227, 230, 231],[225, 229, 229],[222, 226, 226],[219, 224, 223],[217, 221, 222],[212, 218, 218],[206, 212, 215],[199, 208, 210],[193, 201, 207],[182, 194, 200],[176, 188, 197],[169, 183, 190],[165, 178, 187],[158, 175, 186],[157, 172, 185],[161, 177, 183],[168, 182, 187],[172, 187, 187],[180, 192, 191],[185, 198, 192],[189, 200, 194],[191, 203, 194],[192, 203, 194],[189, 200, 192],[184, 197, 190],[178, 192, 188],[171, 186, 184],[162, 178, 180],[151, 169, 176],[137, 160, 171],[127, 152, 167],[118, 144, 160],[107, 134, 152],[98, 126, 144],[92, 121, 140],[87, 118, 137],[83, 114, 138],[84, 115, 141],[84, 111, 138],[83, 109, 133],[90, 113, 134],[108, 128, 144],[131, 148, 160],[153, 166, 176],[167, 178, 183],[179, 186, 188],[189, 192, 196],[194, 197, 200],[197, 200, 203],[202, 205, 205],[203, 206, 206],[204, 205, 205],[206, 207, 207],[206, 206, 203],[206, 206, 203],[205, 205, 202],[205, 205, 202],[206, 206, 202],[206, 206, 203],[205, 205, 202],[206, 206, 202],[203, 203, 200],[203, 203, 200],[203, 203, 199],[202, 202, 198],[202, 202, 198],[202, 202, 198],[202, 202, 198],[202, 202, 198],[203, 203, 199],[203, 203, 199],[203, 203, 199],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[203, 203, 199],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[204, 204, 200],[205, 205, 200],[205, 205, 200],[206, 206, 201],[206, 206, 201],[207, 207, 202],[207, 207, 201],[208, 208, 202],[208, 208, 202],[207, 207, 201],[207, 207, 201],[206, 206, 200],[205, 205, 200],[203, 203, 199],[203, 203, 198],[202, 202, 197],[201, 201, 197],[201, 201, 197],[200, 200, 196],[200, 200, 196],[199, 199, 195],[199, 199, 195],[199, 199, 195],[199, 199, 195],[199, 199, 195],[199, 199, 195],[200, 200, 196],[200, 200, 196],[200, 200, 196],[200, 200, 196],[200, 200, 196],[199, 199, 195],[198, 198, 194],[197, 197, 194],[195, 195, 192],[193, 193, 191],[191, 191, 189],[189, 189, 188],[188, 188, 187],[186, 186, 185],[185, 185, 184],[184, 184, 184],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[182, 182, 182],[182, 182, 182],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],
	[186, 186, 186],[186, 186, 186],[187, 187, 187],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[189, 189, 188],[189, 189, 189],[191, 191, 190],[193, 193, 192],[195, 195, 193],[199, 199, 196],[201, 201, 197],[204, 204, 200],[208, 208, 203],[210, 210, 205],[212, 212, 206],[213, 213, 206],[213, 213, 206],[213, 213, 206],[211, 211, 205],[210, 210, 205],[209, 209, 204],[208, 208, 203],[208, 208, 203],[208, 208, 203],[209, 209, 205],[210, 210, 205],[211, 211, 206],[211, 211, 206],[212, 212, 206],[211, 211, 206],[210, 210, 205],[209, 209, 205],[208, 208, 204],[207, 207, 203],[206, 206, 203],[205, 205, 202],[204, 204, 202],[204, 204, 202],[204, 204, 202],[204, 204, 202],[205, 205, 203],[205, 205, 203],[206, 206, 204],[206, 206, 204],[206, 206, 204],[206, 206, 204],[206, 206, 204],[206, 206, 204],[207, 207, 204],[207, 207, 205],[208, 208, 205],[209, 209, 206],[211, 211, 207],[212, 212, 208],[213, 213, 209],[214, 214, 210],[217, 217, 212],[218, 218, 213],[220, 220, 214],[220, 220, 215],[220, 220, 215],[219, 219, 214],[217, 217, 213],[216, 216, 212],[214, 214, 211],[213, 213, 209],[211, 211, 208],[209, 209, 207],[208, 208, 206],[207, 208, 206],[206, 208, 205],[206, 208, 205],[206, 208, 205],[206, 206, 207],[206, 206, 206],[207, 208, 208],[209, 210, 208],[210, 211, 210],[212, 213, 212],[214, 214, 211],[216, 216, 211],[217, 217, 211],[218, 219, 212],[218, 219, 210],[219, 220, 210],[219, 220, 210],[219, 220, 209],[218, 219, 210],[217, 218, 209],[217, 218, 210],[217, 217, 210],[216, 216, 211],[215, 215, 210],[214, 217, 212],[214, 217, 214],[215, 217, 215],[217, 219, 218],[219, 221, 221],[222, 224, 225],[225, 227, 229],[228, 230, 232],[230, 233, 235],[232, 235, 237],[236, 237, 239],[238, 239, 242],[239, 240, 242],[239, 239, 242],[239, 239, 242],[239, 240, 243],[239, 239, 241],[240, 241, 242],[241, 242, 243],[241, 242, 243],[240, 241, 242],[237, 238, 239],[235, 236, 238],[233, 234, 236],[230, 231, 234],[230, 231, 234],[230, 231, 234],[229, 230, 233],[229, 230, 233],[228, 229, 232],[229, 229, 232],[229, 230, 232],[225, 229, 232],[225, 229, 230],[223, 227, 230],[221, 225, 227],[217, 223, 224],[214, 219, 220],[210, 216, 217],[207, 212, 216],[202, 209, 211],[196, 202, 208],[188, 198, 202],[182, 192, 200],[171, 184, 192],[165, 178, 190],[158, 173, 182],[152, 166, 179],[144, 163, 177],[141, 158, 175],[143, 161, 171],[148, 165, 175],[150, 169, 174],[155, 172, 177],[160, 177, 179],[163, 178, 180],[163, 179, 180],[163, 179, 178],[161, 177, 177],[156, 173, 175],[151, 169, 172],[145, 164, 170],[137, 157, 167],[127, 148, 162],[112, 138, 156],[105, 133, 154],[97, 127, 148],[89, 120, 143],[81, 113, 136],[75, 107, 132],[69, 104, 129],[66, 100, 130],[66, 100, 132],[69, 98, 132],[68, 96, 127],[76, 102, 130],[98, 119, 141],[121, 140, 156],[142, 156, 170],[158, 169, 177],[171, 179, 183],[182, 186, 192],[189, 192, 197],[193, 196, 202],[199, 202, 204],[201, 203, 206],[201, 202, 204],[203, 203, 205],[201, 201, 199],[201, 201, 199],[201, 201, 199],[201, 201, 199],[201, 201, 199],[201, 201, 199],[200, 200, 198],[200, 200, 197],[198, 198, 196],[198, 198, 196],[197, 197, 195],[197, 197, 194],[197, 197, 194],[196, 196, 193],[197, 197, 195],[197, 197, 195],[198, 198, 196],[198, 198, 196],[198, 198, 196],[199, 199, 196],[198, 198, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[199, 199, 196],[200, 200, 197],[200, 200, 196],[200, 200, 196],[201, 201, 197],[201, 201, 197],[201, 201, 197],[201, 201, 197],[201, 201, 197],[201, 201, 197],[201, 201, 197],[200, 200, 197],[199, 199, 196],[198, 198, 195],[198, 198, 195],[197, 197, 194],[197, 197, 194],[196, 196, 193],[196, 196, 193],[195, 195, 192],[195, 195, 192],[195, 195, 192],[193, 193, 191],[194, 194, 192],[194, 194, 192],[195, 195, 192],[195, 195, 192],[194, 194, 192],[194, 194, 192],[194, 194, 192],[194, 194, 191],[194, 194, 191],[193, 193, 191],[192, 192, 190],[191, 191, 189],[190, 190, 189],[189, 189, 188],[188, 188, 187],[187, 187, 186],[186, 186, 185],[185, 185, 185],[185, 185, 184],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[182, 182, 182],[182, 182, 182],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],
	[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[190, 190, 190],[191, 191, 191],[194, 194, 192],[196, 196, 194],[198, 198, 196],[201, 201, 198],[203, 203, 199],[206, 206, 201],[207, 207, 202],[207, 207, 202],[207, 207, 202],[206, 206, 202],[206, 206, 201],[205, 205, 201],[204, 204, 201],[204, 204, 201],[204, 204, 201],[204, 204, 201],[204, 204, 201],[205, 205, 201],[206, 206, 202],[207, 207, 203],[207, 207, 203],[206, 206, 203],[206, 206, 202],[204, 204, 201],[203, 203, 200],[203, 203, 200],[201, 201, 199],[201, 201, 199],[200, 200, 199],[200, 200, 198],[200, 200, 199],[200, 200, 199],[200, 200, 199],[201, 201, 200],[201, 201, 200],[202, 202, 201],[202, 202, 201],[202, 202, 201],[203, 203, 201],[203, 203, 202],[203, 203, 202],[203, 203, 202],[204, 204, 203],[205, 205, 203],[206, 206, 204],[208, 208, 205],[209, 209, 206],[210, 210, 207],[212, 212, 209],[214, 214, 210],[215, 215, 211],[215, 215, 211],[215, 215, 212],[215, 215, 211],[213, 213, 210],[212, 212, 209],[211, 211, 208],[209, 209, 207],[208, 208, 206],[207, 207, 205],[206, 206, 204],[205, 207, 205],[204, 206, 204],[204, 205, 206],[203, 204, 205],[204, 205, 206],[204, 205, 206],[204, 205, 206],[206, 207, 207],[203, 206, 207],[205, 207, 208],[206, 209, 208],[207, 209, 208],[207, 210, 206],[208, 210, 206],[208, 210, 205],[209, 211, 205],[208, 211, 205],[208, 210, 205],[208, 210, 205],[207, 209, 204],[206, 208, 204],[206, 208, 205],[206, 207, 205],[205, 207, 205],[203, 208, 208],[204, 209, 211],[206, 211, 213],[209, 214, 217],[212, 217, 221],[215, 220, 225],[219, 225, 229],[222, 228, 232],[225, 230, 234],[227, 232, 237],[230, 234, 239],[231, 236, 241],[233, 236, 240],[234, 236, 241],[235, 237, 242],[236, 238, 243],[235, 238, 241],[236, 239, 242],[236, 240, 243],[237, 241, 244],[237, 240, 243],[236, 239, 242],[234, 237, 241],[232, 235, 239],[229, 232, 236],[229, 232, 236],[229, 231, 235],[228, 230, 234],[227, 229, 233],[225, 228, 232],[224, 227, 233],[224, 227, 233],[219, 226, 230],[218, 224, 229],[217, 223, 228],[214, 220, 225],[211, 216, 221],[206, 212, 215],[201, 208, 212],[197, 204, 210],[190, 199, 205],[183, 193, 202],[175, 188, 196],[170, 182, 194],[160, 176, 187],[154, 169, 184],[145, 163, 176],[139, 156, 172],[130, 151, 173],[124, 148, 170],[124, 148, 165],[127, 149, 168],[129, 152, 167],[133, 154, 170],[136, 158, 170],[138, 159, 169],[139, 159, 168],[138, 158, 167],[134, 154, 163],[132, 152, 161],[128, 149, 159],[123, 145, 158],[116, 138, 155],[108, 132, 151],[100, 125, 146],[94, 121, 144],[89, 117, 140],[83, 111, 136],[76, 107, 132],[70, 101, 131],[61, 96, 126],[56, 91, 124],[52, 87, 125],[55, 89, 128],[56, 88, 123],[66, 96, 128],[91, 116, 141],[115, 133, 154],[132, 151, 166],[152, 165, 175],[165, 173, 177],[176, 180, 187],[184, 188, 192],[189, 194, 198],[196, 200, 201],[198, 201, 203],[198, 199, 199],[198, 199, 200],[198, 198, 197],[198, 198, 197],[198, 198, 197],[198, 198, 197],[198, 198, 197],[199, 199, 197],[196, 198, 196],[194, 196, 194],[194, 194, 193],[194, 194, 193],[193, 193, 192],[192, 192, 191],[193, 193, 191],[193, 193, 191],[193, 193, 191],[194, 194, 192],[194, 194, 192],[194, 194, 192],[194, 194, 192],[194, 194, 192],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 193],[195, 195, 194],[196, 196, 194],[195, 195, 193],[196, 196, 193],[195, 195, 193],[196, 196, 193],[195, 195, 193],[196, 196, 193],[196, 196, 193],[195, 195, 193],[196, 196, 193],[195, 195, 193],[194, 194, 192],[194, 194, 192],[193, 193, 191],[192, 192, 191],[192, 192, 190],[192, 192, 190],[191, 191, 190],[191, 191, 190],[190, 190, 189],[190, 190, 189],[190, 190, 189],[190, 190, 189],[190, 190, 189],[190, 190, 189],[190, 190, 189],[191, 191, 189],[191, 191, 189],[191, 191, 189],[190, 190, 188],[190, 190, 188],[189, 189, 188],[189, 189, 188],[188, 188, 187],[187, 187, 186],[187, 187, 186],[186, 186, 185],[185, 185, 185],[185, 185, 185],[185, 185, 184],[184, 184, 184],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],
	[188, 188, 188],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[190, 190, 189],[190, 190, 190],[191, 191, 190],[192, 192, 191],[194, 194, 192],[195, 195, 193],[197, 197, 195],[199, 199, 196],[201, 201, 197],[202, 202, 198],[202, 202, 198],[202, 202, 198],[202, 202, 198],[201, 201, 198],[201, 201, 198],[200, 200, 197],[200, 200, 198],[200, 200, 198],[200, 200, 198],[201, 201, 199],[201, 201, 199],[202, 202, 199],[202, 202, 199],[202, 202, 200],[202, 202, 199],[201, 201, 199],[200, 200, 198],[199, 199, 197],[199, 199, 198],[198, 198, 197],[198, 198, 197],[197, 197, 196],[197, 197, 196],[196, 196, 196],[197, 197, 197],[197, 197, 197],[198, 198, 197],[199, 199, 198],[199, 199, 198],[200, 200, 199],[200, 200, 199],[200, 200, 200],[200, 200, 200],[200, 200, 200],[201, 201, 200],[201, 201, 200],[202, 202, 201],[203, 203, 202],[204, 204, 203],[206, 206, 204],[207, 207, 205],[209, 209, 207],[210, 210, 208],[211, 211, 209],[212, 212, 209],[211, 211, 209],[211, 211, 208],[210, 210, 208],[209, 209, 207],[208, 208, 207],[207, 207, 206],[206, 206, 205],[205, 205, 204],[204, 204, 204],[204, 206, 204],[203, 205, 204],[203, 204, 205],[203, 204, 205],[202, 203, 205],[202, 203, 204],[202, 203, 204],[204, 204, 205],[200, 203, 205],[200, 203, 205],[200, 202, 203],[200, 202, 202],[200, 203, 201],[199, 202, 199],[199, 201, 198],[200, 202, 199],[200, 203, 199],[199, 202, 198],[199, 202, 198],[198, 201, 197],[198, 200, 199],[197, 200, 199],[198, 200, 200],[197, 199, 199],[194, 200, 203],[195, 201, 204],[198, 204, 208],[202, 208, 212],[205, 211, 216],[210, 216, 222],[215, 221, 226],[218, 224, 229],[222, 226, 231],[223, 228, 233],[227, 231, 237],[228, 233, 239],[230, 233, 237],[231, 234, 238],[232, 235, 239],[233, 236, 240],[234, 238, 240],[234, 238, 240],[235, 239, 241],[235, 239, 241],[235, 239, 241],[234, 238, 240],[234, 237, 241],[233, 236, 240],[231, 234, 239],[231, 234, 239],[229, 232, 237],[228, 231, 236],[226, 229, 234],[224, 227, 232],[222, 225, 232],[222, 225, 231],[217, 224, 229],[215, 222, 227],[213, 219, 226],[209, 216, 222],[205, 211, 217],[200, 206, 211],[193, 201, 206],[188, 195, 204],[182, 192, 199],[175, 185, 196],[167, 180, 191],[161, 174, 188],[151, 168, 181],[145, 161, 179],[136, 155, 170],[129, 147, 166],[118, 140, 165],[109, 135, 160],[107, 133, 155],[110, 134, 157],[110, 135, 156],[113, 137, 158],[115, 140, 158],[115, 140, 156],[117, 140, 157],[115, 139, 156],[112, 136, 151],[110, 134, 150],[108, 132, 149],[105, 130, 148],[100, 125, 147],[94, 120, 144],[92, 120, 145],[88, 117, 144],[84, 113, 140],[79, 109, 137],[72, 105, 133],[66, 98, 132],[55, 92, 125],[49, 85, 122],[42, 78, 119],[47, 83, 124],[50, 83, 121],[60, 91, 126],[86, 112, 140],[111, 130, 153],[126, 146, 164],[148, 162, 173],[162, 170, 176],[174, 178, 186],[182, 187, 192],[188, 193, 198],[195, 199, 202],[196, 200, 202],[196, 197, 198],[196, 197, 198],[197, 197, 196],[196, 196, 195],[196, 196, 195],[197, 197, 196],[197, 197, 196],[196, 196, 195],[193, 195, 193],[191, 193, 192],[192, 192, 192],[191, 191, 191],[190, 190, 190],[190, 190, 190],[189, 189, 189],[190, 190, 189],[191, 191, 190],[191, 191, 190],[191, 191, 190],[191, 191, 190],[191, 191, 190],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[193, 193, 192],[193, 193, 192],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 191],[193, 193, 192],[192, 192, 191],[192, 192, 191],[192, 192, 191],[192, 192, 190],[191, 191, 189],[191, 191, 190],[191, 191, 190],[191, 191, 189],[192, 192, 190],[191, 191, 190],[191, 191, 189],[190, 190, 189],[190, 190, 189],[189, 189, 188],[189, 189, 188],[189, 189, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[187, 187, 187],[188, 188, 187],[188, 188, 187],[188, 188, 187],[187, 187, 186],[187, 187, 186],[187, 187, 186],[186, 186, 186],[186, 186, 185],[186, 186, 185],[185, 185, 185],[185, 185, 185],[185, 185, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],[183, 183, 183],
	[188, 188, 188],[189, 189, 189],[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[189, 189, 189],[189, 189, 189],[190, 190, 190],[191, 191, 191],[192, 192, 191],[193, 193, 192],[194, 194, 193],[194, 194, 193],[195, 195, 194],[196, 196, 194],[198, 198, 195],[198, 198, 196],[198, 198, 196],[198, 198, 196],[198, 198, 196],[199, 199, 197],[198, 198, 196],[198, 198, 197],[199, 199, 197],[198, 198, 197],[198, 198, 197],[199, 199, 197],[199, 199, 198],[199, 199, 198],[198, 198, 197],[198, 198, 197],[199, 199, 198],[199, 199, 197],[198, 198, 197],[197, 197, 196],[197, 197, 196],[196, 196, 196],[196, 196, 195],[195, 195, 195],[195, 195, 195],[195, 195, 195],[194, 194, 194],[195, 195, 195],[195, 195, 195],[196, 196, 196],[197, 197, 197],[197, 197, 197],[197, 197, 197],[198, 198, 198],[200, 200, 199],[200, 200, 199],[199, 199, 199],[200, 200, 199],[200, 200, 199],[202, 202, 201],[203, 203, 202],[203, 203, 202],[205, 205, 204],[206, 206, 205],[208, 208, 206],[209, 209, 207],[209, 209, 207],[210, 210, 208],[208, 208, 205],[207, 207, 204],[207, 207, 205],[206, 206, 205],[206, 206, 205],[205, 205, 205],[204, 204, 204],[204, 204, 204],[203, 204, 205],[203, 204, 206],[204, 205, 206],[203, 204, 206],[203, 204, 206],[203, 204, 206],[202, 203, 205],[202, 203, 204],[196, 203, 209],[193, 199, 205],[190, 196, 202],[189, 195, 199],[188, 193, 196],[187, 192, 194],[187, 194, 192],[188, 194, 192],[188, 194, 193],[188, 194, 192],[187, 193, 194],[187, 192, 194],[187, 192, 196],[187, 193, 196],[187, 193, 196],[187, 193, 197],[184, 194, 202],[186, 196, 205],[190, 199, 208],[194, 202, 212],[197, 206, 214],[202, 211, 219],[207, 215, 225],[210, 218, 228],[214, 222, 231],[216, 225, 233],[220, 227, 234],[222, 228, 236],[222, 229, 237],[223, 230, 238],[226, 231, 237],[227, 232, 238],[228, 233, 237],[229, 234, 238],[230, 235, 239],[231, 236, 240],[231, 236, 242],[232, 237, 243],[232, 237, 243],[231, 236, 242],[232, 237, 243],[231, 236, 242],[230, 235, 241],[229, 234, 240],[227, 231, 240],[224, 228, 237],[221, 224, 233],[219, 223, 231],[215, 224, 232],[212, 220, 229],[207, 216, 224],[203, 212, 220],[199, 207, 214],[193, 201, 208],[186, 195, 203],[181, 189, 201],[172, 185, 195],[168, 179, 193],[158, 173, 186],[151, 167, 184],[143, 162, 178],[136, 154, 175],[127, 147, 165],[121, 141, 162],[105, 132, 160],[100, 125, 157],[94, 121, 150],[94, 120, 148],[92, 121, 148],[93, 121, 146],[95, 122, 145],[97, 124, 146],[98, 124, 145],[94, 120, 140],[90, 117, 135],[92, 115, 136],[94, 119, 136],[97, 118, 139],[96, 121, 141],[100, 122, 144],[102, 121, 142],[101, 120, 142],[95, 117, 138],[87, 110, 131],[80, 105, 128],[71, 100, 127],[59, 92, 124],[49, 85, 120],[44, 80, 122],[42, 78, 120],[43, 77, 117],[55, 86, 124],[81, 107, 140],[105, 129, 155],[126, 147, 168],[145, 161, 176],[165, 173, 176],[177, 180, 185],[186, 189, 194],[191, 194, 198],[194, 195, 197],[195, 196, 198],[196, 196, 196],[196, 196, 196],[195, 195, 195],[195, 195, 195],[195, 195, 195],[194, 194, 194],[194, 194, 194],[193, 193, 193],[192, 193, 192],[192, 193, 192],[191, 191, 190],[191, 191, 190],[190, 190, 189],[189, 189, 188],[189, 189, 188],[189, 189, 188],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[191, 191, 191],[190, 190, 190],[190, 190, 190],[191, 191, 190],[189, 189, 188],[189, 189, 189],[189, 189, 189],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 188],[189, 189, 189],[189, 189, 189],[190, 190, 189],[189, 189, 189],[189, 189, 189],[189, 189, 189],[188, 188, 188],[188, 188, 187],[188, 188, 187],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[187, 187, 186],[186, 186, 185],[186, 186, 185],[186, 186, 186],[186, 186, 186],[186, 186, 186],[185, 185, 185],[185, 185, 185],[185, 185, 184],[185, 185, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],
	[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[191, 191, 191],[190, 190, 190],[190, 190, 190],[190, 190, 190],[189, 189, 189],[190, 190, 190],[191, 191, 190],[191, 191, 191],[192, 192, 191],[192, 192, 192],[193, 193, 192],[194, 194, 193],[195, 195, 193],[194, 194, 193],[195, 195, 193],[195, 195, 193],[196, 196, 194],[195, 195, 194],[196, 196, 195],[196, 196, 195],[196, 196, 195],[196, 196, 195],[196, 196, 195],[197, 197, 196],[197, 197, 196],[196, 196, 195],[196, 196, 196],[197, 197, 196],[196, 196, 195],[196, 196, 196],[196, 196, 195],[196, 196, 195],[195, 195, 195],[196, 196, 195],[195, 195, 195],[195, 195, 195],[194, 194, 194],[194, 194, 194],[194, 194, 194],[194, 194, 194],[195, 195, 195],[195, 195, 195],[196, 196, 196],[197, 197, 197],[197, 197, 197],[197, 197, 197],[198, 198, 198],[199, 199, 199],[199, 199, 199],[199, 199, 199],[198, 198, 198],[200, 200, 200],[201, 201, 200],[201, 201, 200],[202, 202, 201],[203, 203, 202],[205, 205, 204],[205, 205, 204],[206, 206, 205],[207, 207, 206],[207, 207, 206],[206, 206, 203],[206, 206, 203],[206, 206, 205],[205, 205, 205],[204, 204, 204],[205, 205, 205],[204, 204, 204],[204, 204, 204],[203, 204, 206],[204, 205, 207],[203, 204, 206],[202, 203, 205],[202, 203, 205],[203, 204, 206],[201, 202, 204],[199, 200, 201],[193, 200, 207],[188, 195, 202],[184, 190, 196],[182, 188, 194],[180, 186, 190],[178, 184, 187],[179, 186, 186],[180, 187, 187],[180, 187, 187],[180, 187, 187],[179, 185, 188],[179, 185, 188],[179, 185, 190],[179, 186, 191],[179, 186, 191],[179, 186, 192],[176, 186, 196],[178, 189, 199],[183, 192, 202],[187, 196, 206],[190, 200, 209],[195, 204, 213],[201, 209, 219],[204, 212, 223],[209, 218, 226],[211, 219, 228],[215, 222, 230],[217, 224, 232],[218, 225, 233],[219, 226, 234],[222, 227, 233],[223, 228, 234],[227, 232, 236],[227, 232, 236],[229, 234, 238],[230, 235, 239],[231, 236, 242],[231, 236, 242],[231, 236, 242],[231, 236, 242],[232, 237, 243],[231, 236, 242],[231, 236, 242],[230, 235, 241],[227, 231, 240],[224, 228, 237],[220, 224, 233],[217, 221, 230],[212, 220, 229],[209, 218, 226],[203, 212, 221],[199, 207, 216],[194, 202, 210],[187, 196, 204],[180, 190, 198],[174, 183, 196],[165, 178, 190],[160, 172, 188],[151, 167, 181],[145, 162, 180],[137, 157, 174],[129, 148, 171],[119, 140, 160],[114, 135, 157],[97, 125, 154],[92, 118, 152],[85, 113, 145],[84, 111, 142],[80, 111, 142],[79, 109, 138],[78, 108, 136],[80, 109, 136],[79, 108, 135],[77, 106, 132],[75, 105, 129],[82, 108, 134],[89, 115, 138],[96, 119, 145],[100, 126, 150],[105, 128, 154],[108, 128, 151],[108, 128, 152],[102, 125, 147],[94, 117, 140],[87, 112, 136],[76, 106, 134],[63, 96, 128],[50, 86, 122],[42, 79, 121],[41, 78, 120],[43, 78, 118],[55, 87, 125],[82, 108, 141],[106, 130, 156],[129, 151, 172],[150, 166, 181],[169, 177, 180],[179, 182, 187],[187, 190, 195],[191, 194, 199],[193, 194, 196],[194, 195, 197],[195, 195, 195],[195, 195, 195],[195, 195, 195],[195, 195, 195],[195, 195, 195],[194, 194, 194],[194, 194, 194],[193, 193, 193],[192, 194, 193],[191, 193, 192],[191, 191, 191],[190, 190, 190],[189, 189, 189],[189, 189, 189],[188, 188, 188],[188, 188, 188],[189, 189, 189],[189, 189, 189],[189, 189, 189],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[190, 190, 190],[189, 189, 189],[189, 189, 189],[189, 189, 188],[188, 188, 188],[188, 188, 188],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[187, 187, 187],[188, 188, 188],[188, 188, 188],[188, 188, 188],[189, 189, 188],[188, 188, 188],[188, 188, 188],[188, 188, 188],[187, 187, 187],[187, 187, 187],[187, 187, 187],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[186, 186, 186],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[185, 185, 185],[186, 186, 186],[185, 185, 185],[185, 185, 185],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184],[184, 184, 184]]
imgHeaderBuf = imgldr(imgHeader, imgHeaderWidth, imgHeaderHeight)

##################
#Main Loop
##################
Register(draw, event, bevent)