package org.apache.dvsl;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.Map;
import java.util.Properties;
import java.util.Vector;
import java.util.HashMap;
import java.util.Enumeration;

import java.io.File;
import java.io.FileInputStream;
import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.FileOutputStream;

import org.apache.tools.ant.AntClassLoader;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.MatchingTask;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.Reference;


/**
 * A Task to process via DVSL a set of XML documents. This is
 * useful for building views of XML based documentation.
 * arguments:
 * <ul>
 * <li>basedir
 * <li>destdir
 * <li>style
 * <li>in
 * <li>out
 * <li>logfile
 * <li>includes
 * <li>excludes
 * <li>force
 * <li>extension
 * <li>outputencoding
 * <li>classpath
 * <li>classpathref
 * <li>toolboxfile
 * <li>velocityconfigclass
 * <li>
 * </ul>
 * <p>Of these arguments, the <b>sourcedir</b> and <b>destdir</b> are required, or,
 * <b>in</b> and <b>out</b> are required.</p>
 * <p>Following are the supported nested elements:</p>
 * <ul>
 * <li>&lt;include&gt;
 * <li>&lt;exclude&gt;
 * <li>&lt;classpath&gt;
 * <li>&lt;tool name="toolbox-property" value="value-or-object" /&gt;
 * <li>&lt;velconfig name="velocity-config-name" value="config-value" /&gt;
 * </ul>
 * <p>This task will recursively scan the sourcedir and destdir
 * looking for XML documents to process via DVSL. </p>
 *
 * <p>This task was adapted from Ant's &lt;style&gt; task (XSLTProcess class) from the
 * 1.4.1 release.</p>
 *
 * @author <a href="mailto:kvisco@exoffice.com">Keith Visco</a>
 * @author <a href="mailto:rubys@us.ibm.com">Sam Ruby</a>
 * @author <a href="mailto:russgold@acm.org">Russell Gold</a>
 * @author <a href="mailto:stefan.bodewig@epost.de">Stefan Bodewig</a>
 * @author <a href="mailto:geirm@apache.org">Geir Magnusson Jr.</a>
 * @author <a href="mailto:billb@progress.com">Bill Burton.</a>
 */

public class DVSLTask extends MatchingTask
{
    /**
     *  Supported app values
     */
    public static final String INFILENAME = "infilename";
    public static final String OUTFILENAME = "outfilename";

    private DVSL dvsl;

    private File destDir = null;
    private File baseDir = null;
    private File stylesheet = null;
    private String stylesheetEncoding = null;
    private File inFile = null;
    private File outFile = null;
    private File logFile = null;

    private String targetExtension = ".html";
    private String outputEncoding = "UTF-8";

    private Path classpath = null;
    private ClassLoader classLoader = null;

    private boolean force = false;

    private Vector toolAttr = new Vector();
    private File toolboxFile = null;
    private Properties toolboxProps = null;

    private String velConfigClass = null;
    private Map velConfigMap = null;
    private Vector velConfigAttr = new Vector();

    private boolean validatingParser = false;

    //private String outputtype = null;  // later when output type is supported

    /**
     * Creates a new DVSLTask Task.
     **/
    public DVSLTask()
    {
        classLoader = DVSLTask.class.getClassLoader();
    }

    /**
     * Executes the task.
     */
    public void execute()
            throws BuildException
    {
        DirectoryScanner scanner;
        String[]         list;
        String[]         dirs;

        if (stylesheet == null)
        {
            throw new BuildException("no stylesheet specified", getLocation());
        }

        if (baseDir == null)
        {
            baseDir = getProject().resolveFile(".");
        }

        /*
         * make a DVSL and set validation
         */
        dvsl = new DVSL();

        dvsl.setValidatingParser(validatingParser);

        /*
         *  Create a new Classloader for loading the Toolbox and the Velocity
         *  properties class.
         */
        if (classpath != null)
        {
            classLoader = new AntClassLoader(getProject(), classpath);
            dvsl.setClassLoader(classLoader);
        }

        /*
         * If the user gave us a velPropClass, we create an instance
         * of that to use to config the VelocityEngine inside te DVSL
         */
        Object velConfigObj = null;

        if (velConfigClass != null)
        {
            try
            {
                velConfigObj = Class.forName(velConfigClass, true, classLoader).newInstance();

                if (velConfigObj instanceof Map)
                {
                    velConfigMap = (Map) velConfigObj;
                }
                else
                {
                    throw new BuildException("VelocityPropClass is not instanceof java.util.Map");
                }
            }
            catch(Exception ex)
            {
                throw new BuildException("Error instantiating VelocityPropClass : "
                        + ex);
            }
        }

        /*
         * If any nested Velocity Config elements have been specified, overlay any settings
         * in the Velocity Config object.
         */
        if (!velConfigAttr.isEmpty())
        {
            if (velConfigMap == null)
            {
               velConfigMap = new HashMap();
            }

            /*
             * Now copy velocity config attributes into the Map
             */
            for (Enumeration e = velConfigAttr.elements(); e.hasMoreElements();)
            {
                VelocityConfig p = (VelocityConfig)e.nextElement();
                velConfigMap.put(p.getName(), p.getValue());
            }
        }

        /*
         * Finally, set the Velocity Config object in DVSL if it's valid.
         */
        if (velConfigMap != null)
        {
            dvsl.setVelocityConfig(velConfigMap);
        }

        /*
         * If a logfile attribute was specified, use that for the log file name,
         * otherwise use a Velocity to Ant logging adapter.
         */
        if (logFile != null)
        {
            dvsl.setLogFile(logFile);
        }
        else
        {
            dvsl.setLogChute(new AntLogChute(this));
        }

        /*
         * now the stylesheet
         */
        try
        {
            log("Loading stylesheet " + stylesheet, Project.MSG_INFO);
            dvsl.setStylesheet(stylesheet, stylesheetEncoding);
        }
        catch (Exception ex)
        {
            log("Failed to read stylesheet " + stylesheet, Project.MSG_INFO);
            throw new BuildException(ex);
        }

        /*
         *  now, if we were given a toolbox, set that up too
         */
        toolboxProps = new Properties();

        try
        {
            if (toolboxFile != null)
            {
                toolboxProps.load(new FileInputStream(toolboxFile));
            }

            /*
             *  Overlay any parameters
             */
            for (Enumeration e = toolAttr.elements(); e.hasMoreElements();)
            {
                Tool p = (Tool)e.nextElement();
                toolboxProps.setProperty(p.getName(), p.getValue());
            }

            dvsl.setToolbox(toolboxProps);
        }
        catch(Exception ee)
        {
            throw new BuildException("Error loading the toolbox : " + ee);
        }

        /*
         * if we have an in file and out then process them
         */

        if (inFile != null && outFile != null)
        {
            process(inFile, outFile, stylesheet);
            return;
        }

        /*
         * if we get here, in and out have not been specified, we are
         * in batch processing mode.
         */

        /*
         *   make sure Source directory exists...
         */
        if (destDir == null)
        {
            String msg = "destdir attributes must be set!";
            throw new BuildException(msg);
        }

        scanner = getDirectoryScanner(baseDir);
        log("Transforming into "+destDir, Project.MSG_INFO);

        /*
         *  Process all the files marked for styling
         */
        list = scanner.getIncludedFiles();

        for (int i = 0;i < list.length; ++i)
        {
            process(baseDir, list[i], destDir, stylesheet);
        }

        /*
         *  Process all the directoried marked for styling
         */
        dirs = scanner.getIncludedDirectories();

        for (int j = 0;j < dirs.length;++j)
        {
            list=new File(baseDir,dirs[j]).list();

            for (int i = 0;i < list.length;++i)
            {
                process(baseDir, list[i], destDir, stylesheet);
            }
        }
    } //-- execute

    /**
     * Set whether to check dependencies, or always generate.
     * @param force false to check dependencies, true to always generate
     */
    public void setForce(boolean force)
    {
        this.force = force;
    }

    /**
     * Set the base directory.
     * @param dir name of the base directory
     */
    public void setBasedir(File dir)
    {
        baseDir = dir;
    }

    /**
     * Set the destination directory where the generated
     * files should be directed.
     * @param dir name of the destination directory
     */
    public void setDestdir(File dir)
    {
        destDir = dir;
    }

    /**
     * Set the desired file extension to be used for the target files.
     * If not specified, &quot;<code>.html</code>&quot; is used.
     * @param name the extension to use
     */
    public void setExtension(String name)
    {
        targetExtension = name;
    }

    /**
     * Sets the file to use for stylesheet.
     * @param dvslFile stylesheet filename
     */
    public void setStyle(File dvslFile)
    {
        this.stylesheet = dvslFile;
    }

    /**
     * Sets the encoding of stylesheet file.
     * @param dvslFileEncoding encoding of stylesheet file
     */
    public void setStyleEncoding(String dvslFileEncoding)
    {
        this.stylesheetEncoding = dvslFileEncoding;
    }

    /**
     * Sets the file to use for logging.  If not specified, all logging
     * is directed through Ant's logging system.
     * @param logFile logging filename
     */
    public void setLogFile(File logFile)
    {
        this.logFile = logFile;
    }

    /**
     * Sets the Toolbox properties file to use.
     * @param toolboxFile properties file of tools
     * @deprecated use setToolboxFile instead
     */
    public void setToolbox(String toolboxFile)
    {
        log("DEPRECATED - use the toolboxfile attribute instead");
        this.toolboxFile = new File(toolboxFile);
    }

    /**
     * Sets the Toolbox properties file to use.
     * @param toolboxFile properties file of tools
     */
    public void setToolboxFile(File toolboxFile)
    {
        this.toolboxFile = toolboxFile;
    }

    /**
     * Allows the user to specify a class that implements
     * {@link java.util.Properties} that will have user properties
     * to be used when setting up DVSL.
     * @param classname Velocity configuration class to load
     */
    public void setVelocityConfigClass(String classname)
    {
        velConfigClass = classname;
    }

    /**
     * Sets an output file
     * @param outFile output file
     */
    public void setOut(File outFile)
    {
        this.outFile = outFile;
    }

    /**
     * Sets an input xml file to be styled
     * @param inFile input file
     */
    public void setIn(File inFile)
    {
        this.inFile = inFile;
    }

    /**
     * Sets the character encoding for output files.  If not specified,
     * output is written with UTF-8 encodin6g.
     * @param encoding Output encoding
     */
    public void setOutputEncoding(String encoding)
    {
        if (encoding != null)
            this.outputEncoding = encoding;
    }

    /**
     * Set the classpath to load the Processor through (attribute).
     * @param classpath classpath to set
     */
    public void setClasspath(Path classpath)
    {
        createClasspath().append(classpath);
    }

    /**
     * Set the classpath to load the Processor through (nested element).
     */
    public Path createClasspath()
    {
        if (classpath == null)
        {
            classpath = new Path(getProject());
        }
        return classpath.createPath();
    }

    /**
     * Set the classpath to load the Processor through via reference
     * (attribute).
     * @param r reference to classpath
     */
    public void setClasspathRef(Reference r)
    {
        createClasspath().setRefid(r);
    }

    /**
     *  Sets the flag to have DVSL use a validating parser for the
     *  input documents
     */
    public void setValidatingParser(boolean validating)
    {
        if (validating == true)
        {
            log("Parser is validating.");
        }

        validatingParser = validating;
    }

    /**
     *  Sets an application value from outside of the DVSL task
     */
    public void putAppValue(String name, Object o)
    {
        dvsl.putAppValue(name,o);
    }

    /**
     * Processes the given input XML file and stores the result
     * in the given resultFile.
     */
    private void process(File baseDir, String xmlFile, File destDir,
                         File stylesheet)
        throws BuildException
    {

        String fileExt=targetExtension;
        File   outFile=null;
        File   inFile=null;

        try
        {
            long styleSheetLastModified = stylesheet.lastModified();
            inFile = new File(baseDir,xmlFile);
            int dotPos = xmlFile.lastIndexOf('.');

            dvsl.putAppValue(INFILENAME, xmlFile);

            String outfilename;

            if (dotPos > 0)
            {
                outfilename = xmlFile.substring(0, xmlFile.lastIndexOf('.'))
                         + fileExt;
                outFile = new File(destDir, outfilename);
            }
            else
            {
                outfilename = xmlFile + fileExt;
                outFile = new File(destDir, outfilename);
            }

            dvsl.putAppValue(OUTFILENAME, outfilename);

            if (force ||
                inFile.lastModified() > outFile.lastModified() ||
                styleSheetLastModified > outFile.lastModified())
            {
                ensureDirectoryFor(outFile);
                log("Processing "+inFile+" to "+outFile, Project.MSG_INFO);
                long time = transform(inFile, outFile);
                log("Processed "+inFile+" in "+time+" ms.", Project.MSG_VERBOSE);
            }
        }
        catch (Exception ex)
        {
            /*
             * If failed to process document, must delete target document,
             * or it will not attempt to process it the second time
             */

            log("Failed to process " + inFile, Project.MSG_INFO);

            if (outFile != null)
            {
                outFile.delete();
            }

            throw new BuildException(ex);
        }
    }

    private void process(File inFile, File outFile, File stylesheet)
            throws BuildException
    {
        try
        {
            long styleSheetLastModified = stylesheet.lastModified();

            log("In file "+inFile+" time: " + inFile.lastModified() ,
                    Project.MSG_DEBUG);
            log("Out file "+outFile+" time: " + outFile.lastModified() ,
                    Project.MSG_DEBUG);
            log("Style file "+stylesheet+" time: " + styleSheetLastModified ,
                    Project.MSG_DEBUG);

            if (force ||
                inFile.lastModified() > outFile.lastModified() ||
                styleSheetLastModified > outFile.lastModified())
            {
                ensureDirectoryFor(outFile);
                log("Processing " + inFile + " to " + outFile, Project.MSG_INFO);
                long time = transform(inFile, outFile);
                log("Processed "+inFile+" in "+time+" ms.", Project.MSG_VERBOSE);
            }
        }
        catch (Exception ex)
        {
            log("Failed to process " + inFile, Project.MSG_INFO);

            if(outFile!=null)
            {
                outFile.delete();
            }
            throw new BuildException(ex);
        }
    }

    /**
     * <p>
     * Does the actual transform
     * </p>
     *
     * @param inFile  XML document source
     * @param outFile File for transformed input
     * @return elapsed time in ms. for transformation
     */
    private long transform(File inFile, File outFile)
        throws Exception
    {
        BufferedWriter writer =
                new BufferedWriter(new OutputStreamWriter(
                                       new FileOutputStream(outFile),
                                           outputEncoding));

        long time = dvsl.transform(inFile, writer);
        writer.close();
        return time;
    }

    private void ensureDirectoryFor(File targetFile) throws BuildException
    {
        File directory = new File(targetFile.getParent());

        if (!directory.exists())
        {
            if (!directory.mkdirs())
            {
                throw new BuildException("Unable to create directory: "
                                         + directory.getAbsolutePath());
            }
        }
    }

    /**
     * support for &lt;tool&gt; nested element
     */
    public Tool createTool()
    {
        Tool p = new Tool();
        toolAttr.addElement(p);
        return p;
    }

    public class Tool
    {
        private String name = null;
        private String value = null;

        public void setName(String name)
        {
            this.name = name;
        }

        public String getName() throws BuildException
        {
            if (name == null)
            {
                throw new BuildException("Name attribute is missing.");
            }
            return name;
        }

        public void setValue(String value)
        {
            this.value = value;
        }

        public String getValue() throws BuildException
        {
            if (value == null)
            {
                throw new BuildException("value for attribute for " + getName() + " is missing.");
            }
            return value;
        }
    }

    /**
     *  support for &lt;velconfig&gt; nested element
     */
    public VelocityConfig createVelConfig()
    {
        VelocityConfig p = new VelocityConfig();
        velConfigAttr.addElement(p);
        return p;
    }

    public class VelocityConfig
    {
        private String name = null;
        private String value = null;

        public void setName(String name)
        {
            this.name = name;
        }

        public String getName() throws BuildException
        {
            if (name == null)
            {
                throw new BuildException("Name attribute is missing.");
            }
            return name;
        }

        public void setValue(String value)
        {
            this.value = value;
        }

        public String getValue() throws BuildException
        {
            if (value == null)
            {
                throw new BuildException("Value attribute is missing.");
            }
            return value;
        }
    }

    /**
     * Set the output type to use for the transformation.  Only "xml" (the
     * default) is guaranteed to work for all parsers.  Xalan2 also
     * supports "html" and "text".<br>
     * <i>Not currently implemented.</i>
     * @param type the output method to use
     */
    /*
    public void setOutputtype(String type) {
        this.outputtype = type;
    }
    */

}
