/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.binarytuple;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.util.BitSet;
import java.util.UUID;
import org.apache.ignite.internal.binarytuple.BinaryTupleCommon;
import org.apache.ignite.internal.binarytuple.BinaryTupleFormatException;

public class BinaryTupleParser {
    public static final ByteOrder ORDER = ByteOrder.LITTLE_ENDIAN;
    private static final int UUID_SIZE = 16;
    private final int numElements;
    private final int entrySize;
    private final int entryBase;
    private final int valueBase;
    private final ByteBuffer buffer;

    public BinaryTupleParser(int numElements, ByteBuffer buffer) {
        this.numElements = numElements;
        assert (buffer.order() == ORDER);
        assert (buffer.position() == 0);
        this.buffer = buffer;
        byte flags = buffer.get(0);
        int base = 1;
        if ((flags & 4) != 0) {
            base += BinaryTupleCommon.nullMapSize(numElements);
        }
        this.entryBase = base;
        this.entrySize = 1 << (flags & 3);
        this.valueBase = base + this.entrySize * numElements;
    }

    public int size() {
        return this.valueBase + this.getOffset(this.valueBase - this.entrySize);
    }

    public int elementCount() {
        return this.numElements;
    }

    public boolean hasNullMap() {
        return this.entryBase > 1;
    }

    public ByteBuffer byteBuffer() {
        return this.buffer.slice().order(ORDER);
    }

    public void fetch(int index, Sink sink) {
        int nextOffset;
        assert (index >= 0);
        assert (index < this.numElements);
        int entry = this.entryBase + index * this.entrySize;
        int offset = this.valueBase;
        if (index > 0) {
            offset += this.getOffset(entry - this.entrySize);
        }
        if ((nextOffset = this.valueBase + this.getOffset(entry)) < offset) {
            throw new BinaryTupleFormatException("Corrupted offset table");
        }
        if (offset == nextOffset && this.hasNullMap()) {
            int nullIndex = BinaryTupleCommon.nullOffset(index);
            byte nullMask = BinaryTupleCommon.nullMask(index);
            if ((this.buffer.get(nullIndex) & nullMask) != 0) {
                sink.nextElement(index, 0, 0);
                return;
            }
        }
        sink.nextElement(index, offset, nextOffset);
    }

    public void parse(Sink sink) {
        int entry = this.entryBase;
        int offset = this.valueBase;
        for (int i = 0; i < this.numElements; ++i) {
            int nextOffset = this.valueBase + this.getOffset(entry);
            if (nextOffset < offset) {
                throw new BinaryTupleFormatException("Corrupted offset table");
            }
            if (offset == nextOffset && this.hasNullMap()) {
                int nullIndex = BinaryTupleCommon.nullOffset(i);
                byte nullMask = BinaryTupleCommon.nullMask(i);
                if ((this.buffer.get(nullIndex) & nullMask) != 0) {
                    sink.nextElement(i, 0, 0);
                    entry += this.entrySize;
                    continue;
                }
            }
            sink.nextElement(i, offset, nextOffset);
            offset = nextOffset;
            entry += this.entrySize;
        }
    }

    public final byte byteValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0;
            }
            case 1: {
                return this.buffer.get(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final short shortValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0;
            }
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final int intValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0;
            }
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
            case 4: {
                return this.buffer.getInt(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final long longValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0L;
            }
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
            case 4: {
                return this.buffer.getInt(begin);
            }
            case 8: {
                return this.buffer.getLong(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final float floatValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0.0f;
            }
            case 4: {
                return this.buffer.getFloat(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final double doubleValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return 0.0;
            }
            case 4: {
                return this.buffer.getFloat(begin);
            }
            case 8: {
                return this.buffer.getDouble(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    public final BigInteger numberValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (len == 0) {
            return BigInteger.ZERO;
        }
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.bytesValue(begin, end);
            begin = 0;
        }
        return new BigInteger(bytes, begin, len);
    }

    public final String stringValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.bytesValue(begin, end);
            begin = 0;
        }
        return new String(bytes, begin, len, StandardCharsets.UTF_8);
    }

    public final byte[] bytesValue(int begin, int end) {
        byte[] bytes = new byte[end - begin];
        this.buffer.duplicate().position(begin).limit(end).get(bytes);
        return bytes;
    }

    public final UUID uuidValue(int begin, int end) {
        int len = end - begin;
        if (len != 16) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_UUID;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        long msb = this.buffer.getLong(begin);
        long lsb = this.buffer.getLong(begin + 8);
        return new UUID(msb, lsb);
    }

    public final BitSet bitmaskValue(int begin, int end) {
        return BitSet.valueOf(this.buffer.duplicate().position(begin).limit(end));
    }

    public final LocalDate dateValue(int begin, int end) {
        int len = end - begin;
        if (len != 3) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_DATE;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        return this.getDate(begin);
    }

    public final LocalTime timeValue(int begin, int end) {
        int len = end - begin;
        if (len < 4 || len > 6) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_TIME;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        return this.getTime(begin, len);
    }

    public final LocalDateTime dateTimeValue(int begin, int end) {
        int len = end - begin;
        if (len < 7 || len > 9) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_DATE_TIME;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        return LocalDateTime.of(this.getDate(begin), this.getTime(begin + 3, len - 3));
    }

    public final Instant timestampValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_TIMESTAMP;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        long seconds = this.buffer.getLong(begin);
        int nanos = len == 8 ? 0 : this.buffer.getInt(begin + 8);
        return Instant.ofEpochSecond(seconds, nanos);
    }

    public final Duration durationValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            if (len == 0) {
                return BinaryTupleCommon.DEFAULT_DURATION;
            }
            throw new BinaryTupleFormatException("Invalid length for a tuple element");
        }
        long seconds = this.buffer.getLong(begin);
        int nanos = len == 8 ? 0 : this.buffer.getInt(begin + 8);
        return Duration.ofSeconds(seconds, nanos);
    }

    public final Period periodValue(int begin, int end) {
        switch (end - begin) {
            case 0: {
                return BinaryTupleCommon.DEFAULT_PERIOD;
            }
            case 3: {
                return Period.of(this.buffer.get(begin), this.buffer.get(begin + 1), this.buffer.get(begin + 2));
            }
            case 6: {
                return Period.of(this.buffer.getShort(begin), this.buffer.getShort(begin + 2), this.buffer.getShort(begin + 4));
            }
            case 12: {
                return Period.of(this.buffer.getInt(begin), this.buffer.getInt(begin + 4), this.buffer.getInt(begin + 8));
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element");
    }

    private int getOffset(int index) {
        switch (this.entrySize) {
            case 1: {
                return Byte.toUnsignedInt(this.buffer.get(index));
            }
            case 2: {
                return Short.toUnsignedInt(this.buffer.getShort(index));
            }
            case 4: {
                int offset = this.buffer.getInt(index);
                if (offset < 0) {
                    throw new BinaryTupleFormatException("Unsupported offset table size");
                }
                return offset;
            }
            case 8: {
                throw new BinaryTupleFormatException("Unsupported offset table size");
            }
        }
        throw new BinaryTupleFormatException("Invalid offset table size");
    }

    private LocalDate getDate(int offset) {
        int date = Short.toUnsignedInt(this.buffer.getShort(offset));
        int day = (date |= this.buffer.get(offset + 2) << 16) & 0x1F;
        int month = date >> 5 & 0xF;
        int year = date >> 9;
        return LocalDate.of(year, month, day);
    }

    private LocalTime getTime(int offset, int length) {
        int nanos;
        long time = Integer.toUnsignedLong(this.buffer.getInt(offset));
        if (length == 4) {
            nanos = ((int)time & 0x3FF) * 1000 * 1000;
            time >>>= 10;
        } else if (length == 5) {
            nanos = ((int)(time |= Byte.toUnsignedLong(this.buffer.get(offset + 4)) << 32) & 0xFFFFF) * 1000;
            time >>>= 20;
        } else {
            nanos = (int)(time |= Short.toUnsignedLong(this.buffer.getShort(offset + 4)) << 32) & 0x3FFFFFFF;
            time >>>= 30;
        }
        int second = (int)time & 0x3F;
        int minute = (int)time >>> 6 & 0x3F;
        int hour = (int)time >>> 12 & 0x1F;
        return LocalTime.of(hour, minute, second, nanos);
    }

    public static interface Sink {
        public void nextElement(int var1, int var2, int var3);
    }
}

