/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.MapDifference;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.functions.Function;
import org.apache.cassandra.cql3.functions.FunctionName;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.virtual.VirtualKeyspaceRegistry;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.LocalStrategy;
import org.apache.cassandra.schema.DefaultSchemaUpdateHandler;
import org.apache.cassandra.schema.DistributedSchema;
import org.apache.cassandra.schema.KeyspaceMetadata;
import org.apache.cassandra.schema.Keyspaces;
import org.apache.cassandra.schema.SchemaChangeListener;
import org.apache.cassandra.schema.SchemaChangeNotifier;
import org.apache.cassandra.schema.SchemaConstants;
import org.apache.cassandra.schema.SchemaDiagnostics;
import org.apache.cassandra.schema.SchemaKeyspace;
import org.apache.cassandra.schema.SchemaProvider;
import org.apache.cassandra.schema.SchemaTransformation;
import org.apache.cassandra.schema.SchemaUpdateHandler;
import org.apache.cassandra.schema.SchemaUpdateHandlerFactoryProvider;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.schema.TableMetadataRef;
import org.apache.cassandra.schema.TableMetadataRefCache;
import org.apache.cassandra.schema.Tables;
import org.apache.cassandra.schema.ViewMetadata;
import org.apache.cassandra.schema.Views;
import org.apache.cassandra.service.PendingRangeCalculatorService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.concurrent.Awaitable;
import org.apache.cassandra.utils.concurrent.LoadingMap;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Schema
implements SchemaProvider {
    private static final Logger logger = LoggerFactory.getLogger(Schema.class);
    public static final Schema instance = new Schema();
    private volatile Keyspaces distributedKeyspaces = Keyspaces.none();
    private volatile Keyspaces distributedAndLocalKeyspaces;
    private final Keyspaces localKeyspaces;
    private volatile TableMetadataRefCache tableMetadataRefCache = TableMetadataRefCache.EMPTY;
    private final LoadingMap<String, Keyspace> keyspaceInstances = new LoadingMap();
    private volatile UUID version = SchemaConstants.emptyVersion;
    private final SchemaChangeNotifier schemaChangeNotifier = new SchemaChangeNotifier();
    public final SchemaUpdateHandler updateHandler;
    private final boolean online;

    private Schema() {
        this.online = DatabaseDescriptor.isDaemonInitialized();
        this.distributedAndLocalKeyspaces = this.localKeyspaces = CassandraRelevantProperties.FORCE_LOAD_LOCAL_KEYSPACES.getBoolean() || DatabaseDescriptor.isDaemonInitialized() || DatabaseDescriptor.isToolInitialized() ? Keyspaces.of(SchemaKeyspace.metadata(), SystemKeyspace.metadata()) : Keyspaces.none();
        this.localKeyspaces.forEach(this::loadNew);
        this.updateHandler = SchemaUpdateHandlerFactoryProvider.instance.get().getSchemaUpdateHandler(this.online, this::mergeAndUpdateVersion);
    }

    @VisibleForTesting
    public Schema(boolean online, Keyspaces localKeyspaces, SchemaUpdateHandler updateHandler) {
        this.online = online;
        this.distributedAndLocalKeyspaces = this.localKeyspaces = localKeyspaces;
        this.updateHandler = updateHandler;
    }

    public void startSync() {
        logger.debug("Starting update handler");
        this.updateHandler.start();
    }

    public boolean waitUntilReady(Duration timeout) {
        logger.debug("Waiting for update handler to be ready...");
        return this.updateHandler.waitUntilReady(timeout);
    }

    public void loadFromDisk() {
        SchemaDiagnostics.schemaLoading(this);
        this.updateHandler.reset(true);
        SchemaDiagnostics.schemaLoaded(this);
    }

    private synchronized void load(KeyspaceMetadata ksm) {
        Preconditions.checkArgument((!SchemaConstants.isLocalSystemKeyspace(ksm.name) ? 1 : 0) != 0);
        KeyspaceMetadata previous = this.distributedKeyspaces.getNullable(ksm.name);
        if (previous == null) {
            this.loadNew(ksm);
        } else {
            this.reload(previous, ksm);
        }
        this.distributedKeyspaces = this.distributedKeyspaces.withAddedOrUpdated(ksm);
        this.distributedAndLocalKeyspaces = this.distributedAndLocalKeyspaces.withAddedOrUpdated(ksm);
    }

    private synchronized void loadNew(KeyspaceMetadata ksm) {
        this.tableMetadataRefCache = this.tableMetadataRefCache.withNewRefs(ksm);
        SchemaDiagnostics.metadataInitialized(this, ksm);
    }

    private synchronized void reload(KeyspaceMetadata previous, KeyspaceMetadata updated) {
        Keyspace keyspace = this.getKeyspaceInstance(updated.name);
        if (null != keyspace) {
            keyspace.setMetadata(updated);
        }
        Tables.TablesDiff tablesDiff = Tables.diff(previous.tables, updated.tables);
        Views.ViewsDiff viewsDiff = Views.diff(previous.views, updated.views);
        MapDifference<String, TableMetadata> indexesDiff = previous.tables.indexesDiff(updated.tables);
        this.tableMetadataRefCache = this.tableMetadataRefCache.withUpdatedRefs(previous, updated);
        SchemaDiagnostics.metadataReloaded(this, previous, updated, tablesDiff, viewsDiff, indexesDiff);
    }

    public void registerListener(SchemaChangeListener listener) {
        this.schemaChangeNotifier.registerListener(listener);
    }

    public void unregisterListener(SchemaChangeListener listener) {
        this.schemaChangeNotifier.unregisterListener(listener);
    }

    @Override
    public Keyspace getKeyspaceInstance(String keyspaceName) {
        return this.keyspaceInstances.getIfReady(keyspaceName);
    }

    public ColumnFamilyStore getColumnFamilyStoreInstance(TableId id) {
        TableMetadata metadata = this.getTableMetadata(id);
        if (metadata == null) {
            return null;
        }
        Keyspace instance = this.getKeyspaceInstance(metadata.keyspace);
        if (instance == null) {
            return null;
        }
        return instance.hasColumnFamilyStore(metadata.id) ? instance.getColumnFamilyStore(metadata.id) : null;
    }

    @Override
    public Keyspace maybeAddKeyspaceInstance(String keyspaceName, Supplier<Keyspace> loadFunction) {
        return this.keyspaceInstances.blockingLoadIfAbsent(keyspaceName, loadFunction);
    }

    private Keyspace maybeRemoveKeyspaceInstance(String keyspaceName, Consumer<Keyspace> unloadFunction) {
        try {
            return this.keyspaceInstances.blockingUnloadIfPresent(keyspaceName, unloadFunction);
        }
        catch (LoadingMap.UnloadExecutionException e) {
            throw new AssertionError("Failed to unload the keyspace " + keyspaceName, e);
        }
    }

    public Keyspaces distributedAndLocalKeyspaces() {
        return this.distributedAndLocalKeyspaces;
    }

    public Keyspaces distributedKeyspaces() {
        return this.distributedKeyspaces;
    }

    public int largestGcgs() {
        return this.distributedAndLocalKeyspaces().stream().flatMap(ksm -> ksm.tables.stream()).mapToInt(tm -> tm.params.gcGraceSeconds).max().orElse(Integer.MIN_VALUE);
    }

    private synchronized void unload(KeyspaceMetadata ksm) {
        this.distributedKeyspaces = this.distributedKeyspaces.without(ksm.name);
        this.distributedAndLocalKeyspaces = this.distributedAndLocalKeyspaces.without(ksm.name);
        this.tableMetadataRefCache = this.tableMetadataRefCache.withRemovedRefs(ksm);
        SchemaDiagnostics.metadataRemoved(this, ksm);
    }

    public int getNumberOfTables() {
        return this.distributedAndLocalKeyspaces().stream().mapToInt(k -> Iterables.size(k.tablesAndViews())).sum();
    }

    public ViewMetadata getView(String keyspaceName, String viewName) {
        assert (keyspaceName != null);
        KeyspaceMetadata ksm = this.distributedAndLocalKeyspaces().getNullable(keyspaceName);
        return ksm == null ? null : ksm.views.getNullable(viewName);
    }

    @Override
    public KeyspaceMetadata getKeyspaceMetadata(String keyspaceName) {
        assert (keyspaceName != null);
        KeyspaceMetadata keyspace = this.distributedAndLocalKeyspaces().getNullable(keyspaceName);
        return null != keyspace ? keyspace : VirtualKeyspaceRegistry.instance.getKeyspaceMetadataNullable(keyspaceName);
    }

    @Deprecated
    public Keyspaces getNonSystemKeyspaces() {
        return this.distributedKeyspaces;
    }

    public Keyspaces getNonLocalStrategyKeyspaces() {
        return this.distributedKeyspaces.filter(keyspace -> keyspace.params.replication.klass != LocalStrategy.class);
    }

    public Keyspaces getUserKeyspaces() {
        return this.distributedKeyspaces.without(SchemaConstants.REPLICATED_SYSTEM_KEYSPACE_NAMES);
    }

    public Iterable<TableMetadata> getTablesAndViews(String keyspaceName) {
        Preconditions.checkNotNull((Object)keyspaceName);
        KeyspaceMetadata ksm = (KeyspaceMetadata)ObjectUtils.getFirstNonNull((Supplier[])new Supplier[]{() -> this.distributedKeyspaces.getNullable(keyspaceName), () -> this.localKeyspaces.getNullable(keyspaceName)});
        Preconditions.checkNotNull((Object)ksm, (String)"Keyspace %s not found", (Object)keyspaceName);
        return ksm.tablesAndViews();
    }

    public ImmutableSet<String> getKeyspaces() {
        return this.distributedAndLocalKeyspaces().names();
    }

    public Keyspaces getLocalKeyspaces() {
        return this.localKeyspaces;
    }

    @Override
    public TableMetadataRef getTableMetadataRef(String keyspace, String table) {
        return this.tableMetadataRefCache.getTableMetadataRef(keyspace, table);
    }

    public TableMetadataRef getIndexTableMetadataRef(String keyspace, String index) {
        return this.tableMetadataRefCache.getIndexTableMetadataRef(keyspace, index);
    }

    @Override
    public TableMetadataRef getTableMetadataRef(TableId id) {
        return this.tableMetadataRefCache.getTableMetadataRef(id);
    }

    @Override
    public TableMetadataRef getTableMetadataRef(Descriptor descriptor) {
        return this.getTableMetadataRef(descriptor.ksname, descriptor.cfname);
    }

    @Override
    public TableMetadata getTableMetadata(String keyspace, String table) {
        assert (keyspace != null);
        assert (table != null);
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(keyspace);
        return ksm == null ? null : ksm.getTableOrViewNullable(table);
    }

    @Override
    public TableMetadata getTableMetadata(TableId id) {
        return (TableMetadata)ObjectUtils.getFirstNonNull((Supplier[])new Supplier[]{() -> this.distributedKeyspaces.getTableOrViewNullable(id), () -> this.localKeyspaces.getTableOrViewNullable(id), () -> VirtualKeyspaceRegistry.instance.getTableMetadataNullable(id)});
    }

    public TableMetadata validateTable(String keyspaceName, String tableName) {
        if (tableName.isEmpty()) {
            throw new InvalidRequestException("non-empty table is required");
        }
        KeyspaceMetadata keyspace = this.getKeyspaceMetadata(keyspaceName);
        if (keyspace == null) {
            throw new KeyspaceNotDefinedException(String.format("keyspace %s does not exist", keyspaceName));
        }
        TableMetadata metadata = keyspace.getTableOrViewNullable(tableName);
        if (metadata == null) {
            throw new InvalidRequestException(String.format("table %s does not exist", tableName));
        }
        return metadata;
    }

    public TableMetadata getTableMetadata(Descriptor descriptor) {
        return this.getTableMetadata(descriptor.ksname, descriptor.cfname);
    }

    public Collection<Function> getFunctions(FunctionName name) {
        if (!name.hasKeyspace()) {
            throw new IllegalArgumentException(String.format("Function name must be fully qualified: got %s", name));
        }
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(name.keyspace);
        return ksm == null ? Collections.emptyList() : ksm.functions.get(name);
    }

    public Optional<Function> findFunction(FunctionName name, List<AbstractType<?>> argTypes) {
        if (!name.hasKeyspace()) {
            throw new IllegalArgumentException(String.format("Function name must be fully quallified: got %s", name));
        }
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(name.keyspace);
        return ksm == null ? Optional.empty() : ksm.functions.find(name, argTypes);
    }

    public UUID getVersion() {
        return this.version;
    }

    public boolean isSameVersion(UUID schemaVersion) {
        return schemaVersion != null && schemaVersion.equals(this.version);
    }

    public boolean isEmpty() {
        return SchemaConstants.emptyVersion.equals(this.version);
    }

    private synchronized void updateVersion(UUID version) {
        this.version = version;
        SchemaDiagnostics.versionUpdated(this);
    }

    private synchronized SchemaTransformation.SchemaTransformationResult localDiff(SchemaTransformation.SchemaTransformationResult result) {
        Keyspaces localBefore = this.distributedKeyspaces;
        UUID localVersion = this.version;
        boolean needNewDiff = false;
        if (!Objects.equals(localBefore, result.before.getKeyspaces())) {
            logger.info("Schema was different to what we expected: {}", (Object)Keyspaces.diff(result.before.getKeyspaces(), localBefore));
            needNewDiff = true;
        }
        if (!Objects.equals(localVersion, result.before.getVersion())) {
            logger.info("Schema version was different to what we expected: {} != {}", (Object)result.before.getVersion(), (Object)localVersion);
            needNewDiff = true;
        }
        if (needNewDiff) {
            return new SchemaTransformation.SchemaTransformationResult(new DistributedSchema(localBefore, localVersion), result.after, Keyspaces.diff(localBefore, result.after.getKeyspaces()));
        }
        return result;
    }

    public void reloadSchemaAndAnnounceVersion() {
        this.updateHandler.reset(true);
    }

    @VisibleForTesting
    public synchronized void mergeAndUpdateVersion(SchemaTransformation.SchemaTransformationResult result, boolean dropData) {
        result = this.localDiff(result);
        this.schemaChangeNotifier.notifyPreChanges(result);
        this.merge(result.diff, dropData);
        this.updateVersion(result.after.getVersion());
        if (this.online) {
            SystemKeyspace.updateSchemaVersion(result.after.getVersion());
        }
    }

    public SchemaTransformation.SchemaTransformationResult transform(SchemaTransformation transformation) {
        return this.transform(transformation, false);
    }

    public SchemaTransformation.SchemaTransformationResult transform(SchemaTransformation transformation, boolean local) {
        return this.updateHandler.apply(transformation, local);
    }

    public void resetLocalSchema() {
        logger.debug("Clearing local schema...");
        if (Gossiper.instance.getLiveMembers().stream().allMatch(ep -> FBUtilities.getBroadcastAddressAndPort().equals(ep))) {
            throw new InvalidRequestException("Cannot reset local schema when there are no other live nodes");
        }
        Awaitable clearCompletion = this.updateHandler.clear();
        try {
            if (!clearCompletion.await(StorageService.SCHEMA_DELAY_MILLIS, TimeUnit.MILLISECONDS)) {
                throw new RuntimeException("Schema reset failed - no schema received from other nodes");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Failed to reset schema - the thread has been interrupted");
        }
        SchemaDiagnostics.schemaCleared(this);
        logger.info("Local schema reset completed");
    }

    private void merge(Keyspaces.KeyspacesDiff diff, boolean removeData) {
        diff.dropped.forEach(keyspace -> this.dropKeyspace((KeyspaceMetadata)keyspace, removeData));
        diff.created.forEach(this::createKeyspace);
        diff.altered.forEach(delta -> this.alterKeyspace((KeyspaceMetadata.KeyspaceDiff)delta, removeData));
    }

    private void alterKeyspace(KeyspaceMetadata.KeyspaceDiff delta, boolean dropData) {
        Keyspace keyspace;
        SchemaDiagnostics.keyspaceAltering(this, delta);
        boolean initialized = Keyspace.isInitialized();
        Keyspace keyspace2 = keyspace = initialized ? this.getKeyspaceInstance(delta.before.name) : null;
        if (initialized) {
            assert (keyspace != null);
            assert (delta.before.name.equals(delta.after.name));
            ((Views)delta.views.dropped).forEach(v -> this.dropView(keyspace, (ViewMetadata)v, dropData));
            ((Tables)delta.tables.dropped).forEach(t -> this.dropTable(keyspace, (TableMetadata)t, dropData));
        }
        this.load(delta.after);
        if (initialized) {
            ((Tables)delta.tables.created).forEach(t -> this.createTable(keyspace, (TableMetadata)t));
            ((Views)delta.views.created).forEach(v -> this.createView(keyspace, (ViewMetadata)v));
            delta.tables.altered.forEach(diff -> this.alterTable(keyspace, (TableMetadata)diff.after));
            delta.views.altered.forEach(diff -> this.alterView(keyspace, (ViewMetadata)diff.after));
            Keyspace.open((String)delta.after.name, (SchemaProvider)this, (boolean)true).viewManager.reload(true);
        }
        this.schemaChangeNotifier.notifyKeyspaceAltered(delta, dropData);
        SchemaDiagnostics.keyspaceAltered(this, delta);
    }

    private void createKeyspace(KeyspaceMetadata keyspace) {
        SchemaDiagnostics.keyspaceCreating(this, keyspace);
        this.load(keyspace);
        if (Keyspace.isInitialized()) {
            Keyspace.open(keyspace.name, this, true);
        }
        this.schemaChangeNotifier.notifyKeyspaceCreated(keyspace);
        SchemaDiagnostics.keyspaceCreated(this, keyspace);
        if (keyspace.params.replication.klass != LocalStrategy.class && Keyspace.isInitialized()) {
            PendingRangeCalculatorService.calculatePendingRanges(Keyspace.open(keyspace.name, this, true).getReplicationStrategy(), keyspace.name);
        }
    }

    private void dropKeyspace(KeyspaceMetadata keyspaceMetadata, boolean dropData) {
        Keyspace keyspace;
        SchemaDiagnostics.keyspaceDropping(this, keyspaceMetadata);
        boolean initialized = Keyspace.isInitialized();
        Keyspace keyspace2 = keyspace = initialized ? Keyspace.open(keyspaceMetadata.name, this, false) : null;
        if (initialized) {
            if (keyspace == null) {
                return;
            }
            keyspaceMetadata.views.forEach(v -> this.dropView(keyspace, (ViewMetadata)v, dropData));
            keyspaceMetadata.tables.forEach(t -> this.dropTable(keyspace, (TableMetadata)t, dropData));
            Keyspace unloadedKeyspace = this.maybeRemoveKeyspaceInstance(keyspaceMetadata.name, ks -> {
                ks.unload(dropData);
                this.unload(keyspaceMetadata);
            });
            assert (unloadedKeyspace == keyspace);
            Keyspace.writeOrder.awaitNewBarrier();
        } else {
            this.unload(keyspaceMetadata);
        }
        this.schemaChangeNotifier.notifyKeyspaceDropped(keyspaceMetadata, dropData);
        SchemaDiagnostics.keyspaceDropped(this, keyspaceMetadata);
    }

    private void dropView(Keyspace keyspace, ViewMetadata metadata, boolean dropData) {
        keyspace.viewManager.dropView(metadata.name());
        this.dropTable(keyspace, metadata.metadata, dropData);
    }

    private void dropTable(Keyspace keyspace, TableMetadata metadata, boolean dropData) {
        SchemaDiagnostics.tableDropping(this, metadata);
        keyspace.dropCf(metadata.id, dropData);
        SchemaDiagnostics.tableDropped(this, metadata);
    }

    private void createTable(Keyspace keyspace, TableMetadata table) {
        SchemaDiagnostics.tableCreating(this, table);
        keyspace.initCf(this.tableMetadataRefCache.getTableMetadataRef(table.id), true);
        SchemaDiagnostics.tableCreated(this, table);
    }

    private void createView(Keyspace keyspace, ViewMetadata view) {
        SchemaDiagnostics.tableCreating(this, view.metadata);
        keyspace.initCf(this.tableMetadataRefCache.getTableMetadataRef(view.metadata.id), true);
        SchemaDiagnostics.tableCreated(this, view.metadata);
    }

    private void alterTable(Keyspace keyspace, TableMetadata updated) {
        SchemaDiagnostics.tableAltering(this, updated);
        keyspace.getColumnFamilyStore(updated.name).reload();
        SchemaDiagnostics.tableAltered(this, updated);
    }

    private void alterView(Keyspace keyspace, ViewMetadata updated) {
        SchemaDiagnostics.tableAltering(this, updated.metadata);
        keyspace.getColumnFamilyStore(updated.name()).reload();
        SchemaDiagnostics.tableAltered(this, updated.metadata);
    }

    public Map<UUID, Set<InetAddressAndPort>> getOutstandingSchemaVersions() {
        return this.updateHandler instanceof DefaultSchemaUpdateHandler ? ((DefaultSchemaUpdateHandler)this.updateHandler).getOutstandingSchemaVersions() : Collections.emptyMap();
    }
}

