/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.view;

import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.cassandra.db.Clustering;
import org.apache.cassandra.db.ClusteringPrefix;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.DeletionTime;
import org.apache.cassandra.db.LivenessInfo;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.ByteBufferAccessor;
import org.apache.cassandra.db.marshal.CompositeType;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.BTreeRow;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.ColumnData;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.db.view.View;
import org.apache.cassandra.schema.ColumnMetadata;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.TableMetadata;

public class ViewUpdateGenerator {
    private final View view;
    private final int nowInSec;
    private final TableMetadata baseMetadata;
    private final DecoratedKey baseDecoratedKey;
    private final ByteBuffer[] basePartitionKey;
    private final TableMetadata viewMetadata;
    private final boolean baseEnforceStrictLiveness;
    private final Map<DecoratedKey, PartitionUpdate.Builder> updates = new HashMap<DecoratedKey, PartitionUpdate.Builder>();
    private final ByteBuffer[] currentViewEntryPartitionKey;
    private final Row.Builder currentViewEntryBuilder;

    public ViewUpdateGenerator(View view, DecoratedKey basePartitionKey, int nowInSec) {
        this.view = view;
        this.nowInSec = nowInSec;
        this.baseMetadata = view.getDefinition().baseTableMetadata();
        this.baseEnforceStrictLiveness = this.baseMetadata.enforceStrictLiveness();
        this.baseDecoratedKey = basePartitionKey;
        this.basePartitionKey = ViewUpdateGenerator.extractKeyComponents(basePartitionKey, this.baseMetadata.partitionKeyType);
        this.viewMetadata = Schema.instance.getTableMetadata(view.getDefinition().metadata.id);
        this.currentViewEntryPartitionKey = new ByteBuffer[this.viewMetadata.partitionKeyColumns().size()];
        this.currentViewEntryBuilder = BTreeRow.sortedBuilder();
    }

    private static ByteBuffer[] extractKeyComponents(DecoratedKey partitionKey, AbstractType<?> type) {
        ByteBuffer[] byteBufferArray;
        if (type instanceof CompositeType) {
            byteBufferArray = ((CompositeType)type).split(partitionKey.getKey());
        } else {
            ByteBuffer[] byteBufferArray2 = new ByteBuffer[1];
            byteBufferArray = byteBufferArray2;
            byteBufferArray2[0] = partitionKey.getKey();
        }
        return byteBufferArray;
    }

    public void addBaseTableUpdate(Row existingBaseRow, Row mergedBaseRow) {
        switch (this.updateAction(existingBaseRow, mergedBaseRow)) {
            case NONE: {
                return;
            }
            case NEW_ENTRY: {
                this.createEntry(mergedBaseRow);
                return;
            }
            case DELETE_OLD: {
                this.deleteOldEntry(existingBaseRow, mergedBaseRow);
                return;
            }
            case UPDATE_EXISTING: {
                this.updateEntry(existingBaseRow, mergedBaseRow);
                return;
            }
            case SWITCH_ENTRY: {
                this.createEntry(mergedBaseRow);
                this.deleteOldEntry(existingBaseRow, mergedBaseRow);
                return;
            }
        }
    }

    public Collection<PartitionUpdate> generateViewUpdates() {
        return this.updates.values().stream().map(PartitionUpdate.Builder::build).collect(Collectors.toList());
    }

    public void clear() {
        this.updates.clear();
    }

    private UpdateAction updateAction(Row existingBaseRow, Row mergedBaseRow) {
        Cell<?> after;
        assert (!mergedBaseRow.isEmpty());
        if (this.baseMetadata.isCompactTable()) {
            ClusteringPrefix clustering = mergedBaseRow.clustering();
            for (int i = 0; i < clustering.size(); ++i) {
                if (clustering.get(i) != null) continue;
                return UpdateAction.NONE;
            }
        }
        assert (this.view.baseNonPKColumnsInViewPK.size() <= 1) : "We currently only support one base non-PK column in the view PK";
        if (this.view.baseNonPKColumnsInViewPK.isEmpty()) {
            boolean existingHasLiveData = existingBaseRow != null && existingBaseRow.hasLiveData(this.nowInSec, this.baseEnforceStrictLiveness);
            boolean mergedHasLiveData = mergedBaseRow.hasLiveData(this.nowInSec, this.baseEnforceStrictLiveness);
            return existingHasLiveData ? (mergedHasLiveData ? UpdateAction.UPDATE_EXISTING : UpdateAction.DELETE_OLD) : (mergedHasLiveData ? UpdateAction.NEW_ENTRY : UpdateAction.NONE);
        }
        ColumnMetadata baseColumn = this.view.baseNonPKColumnsInViewPK.get(0);
        assert (!baseColumn.isComplex()) : "A complex column couldn't be part of the view PK";
        Cell<?> before = existingBaseRow == null ? null : existingBaseRow.getCell(baseColumn);
        if (before == (after = mergedBaseRow.getCell(baseColumn))) {
            return this.isLive(before) ? UpdateAction.UPDATE_EXISTING : UpdateAction.NONE;
        }
        if (!this.isLive(before)) {
            return this.isLive(after) ? UpdateAction.NEW_ENTRY : UpdateAction.NONE;
        }
        if (!this.isLive(after)) {
            return UpdateAction.DELETE_OLD;
        }
        return baseColumn.cellValueType().compare(before.buffer(), after.buffer()) == 0 ? UpdateAction.UPDATE_EXISTING : UpdateAction.SWITCH_ENTRY;
    }

    private boolean matchesViewFilter(Row baseRow) {
        return this.view.matchesViewFilter(this.baseDecoratedKey, baseRow, this.nowInSec);
    }

    private boolean isLive(Cell<?> cell) {
        return cell != null && cell.isLive(this.nowInSec);
    }

    private void createEntry(Row baseRow) {
        if (!this.matchesViewFilter(baseRow)) {
            return;
        }
        this.startNewUpdate(baseRow);
        this.currentViewEntryBuilder.addPrimaryKeyLivenessInfo(this.computeLivenessInfoForEntry(baseRow));
        this.currentViewEntryBuilder.addRowDeletion(baseRow.deletion());
        for (ColumnData data : baseRow) {
            ColumnMetadata viewColumn = this.view.getViewColumn(data.column());
            if (viewColumn == null || viewColumn.isPrimaryKeyColumn()) continue;
            this.addColumnData(viewColumn, data);
        }
        this.submitUpdate();
    }

    private void updateEntry(Row existingBaseRow, Row mergedBaseRow) {
        if (!this.matchesViewFilter(existingBaseRow)) {
            this.createEntry(mergedBaseRow);
            return;
        }
        if (!this.matchesViewFilter(mergedBaseRow)) {
            this.deleteOldEntryInternal(existingBaseRow, mergedBaseRow);
            return;
        }
        this.startNewUpdate(mergedBaseRow);
        this.currentViewEntryBuilder.addPrimaryKeyLivenessInfo(this.computeLivenessInfoForEntry(mergedBaseRow));
        this.currentViewEntryBuilder.addRowDeletion(mergedBaseRow.deletion());
        this.addDifferentCells(existingBaseRow, mergedBaseRow);
        this.submitUpdate();
    }

    private void addDifferentCells(Row existingBaseRow, Row mergedBaseRow) {
        PeekingIterator existingIter = Iterators.peekingIterator(existingBaseRow.iterator());
        for (ColumnData mergedData : mergedBaseRow) {
            int cmp;
            ColumnMetadata baseColumn = mergedData.column();
            ColumnMetadata viewColumn = this.view.getViewColumn(baseColumn);
            if (viewColumn == null || viewColumn.isPrimaryKeyColumn()) continue;
            ColumnData existingData = null;
            while (existingIter.hasNext() && (cmp = baseColumn.compareTo(((ColumnData)existingIter.peek()).column())) >= 0) {
                ColumnData next = (ColumnData)existingIter.next();
                if (cmp != 0) continue;
                existingData = next;
                break;
            }
            if (existingData == null) {
                this.addColumnData(viewColumn, mergedData);
                continue;
            }
            if (mergedData == existingData) continue;
            if (baseColumn.isComplex()) {
                ComplexColumnData mergedComplexData = (ComplexColumnData)mergedData;
                ComplexColumnData existingComplexData = (ComplexColumnData)existingData;
                if (mergedComplexData.complexDeletion().supersedes(existingComplexData.complexDeletion())) {
                    this.currentViewEntryBuilder.addComplexDeletion(viewColumn, mergedComplexData.complexDeletion());
                }
                PeekingIterator existingCells = Iterators.peekingIterator(existingComplexData.iterator());
                for (Cell<?> mergedCell : mergedComplexData) {
                    int cmp2;
                    Cell existingCell = null;
                    while (existingCells.hasNext() && (cmp2 = baseColumn.cellPathComparator().compare(mergedCell.path(), ((Cell)existingCells.peek()).path())) <= 0) {
                        Cell next = (Cell)existingCells.next();
                        if (cmp2 != 0) continue;
                        existingCell = next;
                        break;
                    }
                    if (mergedCell == existingCell) continue;
                    this.addCell(viewColumn, mergedCell);
                }
                continue;
            }
            this.addCell(viewColumn, (Cell)mergedData);
        }
    }

    private void deleteOldEntry(Row existingBaseRow, Row mergedBaseRow) {
        if (!this.matchesViewFilter(existingBaseRow)) {
            return;
        }
        this.deleteOldEntryInternal(existingBaseRow, mergedBaseRow);
    }

    private void deleteOldEntryInternal(Row existingBaseRow, Row mergedBaseRow) {
        this.startNewUpdate(existingBaseRow);
        long timestamp = this.computeTimestampForEntryDeletion(existingBaseRow, mergedBaseRow);
        long rowDeletion = mergedBaseRow.deletion().time().markedForDeleteAt();
        assert (timestamp >= rowDeletion);
        if (timestamp > rowDeletion) {
            LivenessInfo info = LivenessInfo.withExpirationTime(timestamp, Integer.MAX_VALUE, this.nowInSec);
            this.currentViewEntryBuilder.addPrimaryKeyLivenessInfo(info);
        }
        this.currentViewEntryBuilder.addRowDeletion(mergedBaseRow.deletion());
        this.addDifferentCells(existingBaseRow, mergedBaseRow);
        this.submitUpdate();
    }

    private void startNewUpdate(Row baseRow) {
        ByteBuffer[] clusteringValues = new ByteBuffer[this.viewMetadata.clusteringColumns().size()];
        for (ColumnMetadata viewColumn : this.viewMetadata.primaryKeyColumns()) {
            ColumnMetadata baseColumn = this.view.getBaseColumn(viewColumn);
            ByteBuffer value = this.getValueForPK(baseColumn, baseRow);
            if (viewColumn.isPartitionKey()) {
                this.currentViewEntryPartitionKey[viewColumn.position()] = value;
                continue;
            }
            clusteringValues[viewColumn.position()] = value;
        }
        this.currentViewEntryBuilder.newRow(Clustering.make(clusteringValues));
    }

    private LivenessInfo computeLivenessInfoForEntry(Row baseRow) {
        assert (this.view.baseNonPKColumnsInViewPK.size() <= 1);
        LivenessInfo baseLiveness = baseRow.primaryKeyLivenessInfo();
        if (this.view.baseNonPKColumnsInViewPK.isEmpty()) {
            if (this.view.getDefinition().includeAllColumns) {
                return baseLiveness;
            }
            long timestamp = baseLiveness.timestamp();
            boolean hasNonExpiringLiveCell = false;
            Cell<?> biggestExpirationCell = null;
            for (Cell<?> cell : baseRow.cells()) {
                if (this.view.getViewColumn(cell.column()) != null || !this.isLive(cell)) continue;
                timestamp = Math.max(timestamp, cell.maxTimestamp());
                if (!cell.isExpiring()) {
                    hasNonExpiringLiveCell = true;
                    continue;
                }
                if (biggestExpirationCell == null) {
                    biggestExpirationCell = cell;
                    continue;
                }
                if (cell.localDeletionTime() <= biggestExpirationCell.localDeletionTime()) continue;
                biggestExpirationCell = cell;
            }
            if (baseLiveness.isLive(this.nowInSec) && !baseLiveness.isExpiring()) {
                return LivenessInfo.create(timestamp, this.nowInSec);
            }
            if (hasNonExpiringLiveCell) {
                return LivenessInfo.create(timestamp, this.nowInSec);
            }
            if (biggestExpirationCell == null) {
                return baseLiveness;
            }
            if (biggestExpirationCell.localDeletionTime() > baseLiveness.localExpirationTime() || !baseLiveness.isLive(this.nowInSec)) {
                return LivenessInfo.withExpirationTime(timestamp, biggestExpirationCell.ttl(), biggestExpirationCell.localDeletionTime());
            }
            return baseLiveness;
        }
        Cell<?> cell = baseRow.getCell(this.view.baseNonPKColumnsInViewPK.get(0));
        assert (this.isLive(cell)) : "We shouldn't have got there if the base row had no associated entry";
        return LivenessInfo.withExpirationTime(cell.timestamp(), cell.ttl(), cell.localDeletionTime());
    }

    private long computeTimestampForEntryDeletion(Row existingBaseRow, Row mergedBaseRow) {
        DeletionTime deletion = mergedBaseRow.deletion().time();
        if (this.view.hasSamePrimaryKeyColumnsAsBaseTable()) {
            long timestamp = Math.max(deletion.markedForDeleteAt(), existingBaseRow.primaryKeyLivenessInfo().timestamp());
            if (this.view.getDefinition().includeAllColumns) {
                return timestamp;
            }
            for (Cell<?> cell : existingBaseRow.cells()) {
                if (this.view.getViewColumn(cell.column()) != null) continue;
                timestamp = Math.max(timestamp, cell.maxTimestamp());
            }
            return timestamp;
        }
        Cell<?> before = existingBaseRow.getCell(this.view.baseNonPKColumnsInViewPK.get(0));
        assert (this.isLive(before)) : "We shouldn't have got there if the base row had no associated entry";
        return deletion.deletes(before) ? deletion.markedForDeleteAt() : before.timestamp();
    }

    private void addColumnData(ColumnMetadata viewColumn, ColumnData baseTableData) {
        assert (viewColumn.isComplex() == baseTableData.column().isComplex());
        if (!viewColumn.isComplex()) {
            this.addCell(viewColumn, (Cell)baseTableData);
            return;
        }
        ComplexColumnData complexData = (ComplexColumnData)baseTableData;
        this.currentViewEntryBuilder.addComplexDeletion(viewColumn, complexData.complexDeletion());
        for (Cell<?> cell : complexData) {
            this.addCell(viewColumn, cell);
        }
    }

    private void addCell(ColumnMetadata viewColumn, Cell<?> baseTableCell) {
        assert (!viewColumn.isPrimaryKeyColumn());
        this.currentViewEntryBuilder.addCell(baseTableCell.withUpdatedColumn(viewColumn));
    }

    private void submitUpdate() {
        Row row = this.currentViewEntryBuilder.build();
        if (row.isEmpty()) {
            return;
        }
        DecoratedKey partitionKey = this.makeCurrentPartitionKey();
        PartitionUpdate.Builder update = this.updates.computeIfAbsent(partitionKey, k -> new PartitionUpdate.Builder(this.viewMetadata, partitionKey, this.viewMetadata.regularAndStaticColumns(), 4));
        update.add(row);
    }

    private DecoratedKey makeCurrentPartitionKey() {
        ByteBuffer rawKey = this.viewMetadata.partitionKeyColumns().size() == 1 ? this.currentViewEntryPartitionKey[0] : CompositeType.build(ByteBufferAccessor.instance, this.currentViewEntryPartitionKey);
        return this.viewMetadata.partitioner.decorateKey(rawKey);
    }

    private ByteBuffer getValueForPK(ColumnMetadata column, Row row) {
        switch (column.kind) {
            case PARTITION_KEY: {
                return this.basePartitionKey[column.position()];
            }
            case CLUSTERING: {
                return row.clustering().bufferAt(column.position());
            }
        }
        return row.getCell(column).buffer();
    }

    private static enum UpdateAction {
        NONE,
        NEW_ENTRY,
        DELETE_OLD,
        UPDATE_EXISTING,
        SWITCH_ENTRY;

    }
}

