/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.RateLimiter;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.exceptions.CDCWriteException;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.DirectorySizeCalculator;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegmentManagerCDC
extends AbstractCommitLogSegmentManager {
    static final Logger logger = LoggerFactory.getLogger(CommitLogSegmentManagerCDC.class);
    private final CDCSizeTracker cdcSizeTracker = new CDCSizeTracker(new File(DatabaseDescriptor.getCDCLogLocation()));

    public CommitLogSegmentManagerCDC(CommitLog commitLog, String storageDirectory) {
        super(commitLog, storageDirectory);
    }

    @Override
    void start() {
        this.cdcSizeTracker.start();
        super.start();
    }

    @Override
    public void discard(CommitLogSegment segment, boolean delete) {
        segment.close();
        this.addSize(-segment.onDiskSize());
        this.cdcSizeTracker.processDiscardedSegment(segment);
        if (delete) {
            FileUtils.deleteWithConfirm(segment.logFile);
        }
        if (segment.getCDCState() != CommitLogSegment.CDCState.CONTAINS) {
            File cdcIndexFile;
            File cdcLink = segment.getCDCFile();
            if (cdcLink.exists()) {
                FileUtils.deleteWithConfirm(cdcLink);
            }
            if ((cdcIndexFile = segment.getCDCIndexFile()).exists()) {
                FileUtils.deleteWithConfirm(cdcIndexFile);
            }
        }
    }

    @Override
    public void shutdown() {
        this.cdcSizeTracker.shutdown();
        super.shutdown();
    }

    @Override
    public CommitLogSegment.Allocation allocate(Mutation mutation, int size) throws CDCWriteException {
        CommitLogSegment.Allocation alloc;
        CommitLogSegment segment = this.allocatingFrom();
        this.permitSegmentMaybe(segment);
        this.throwIfForbidden(mutation, segment);
        while (null == (alloc = segment.allocate(mutation, size))) {
            this.advanceAllocatingFrom(segment);
            segment = this.allocatingFrom();
            this.permitSegmentMaybe(segment);
            this.throwIfForbidden(mutation, segment);
        }
        if (mutation.trackedByCDC()) {
            segment.setCDCState(CommitLogSegment.CDCState.CONTAINS);
        }
        return alloc;
    }

    private void permitSegmentMaybe(CommitLogSegment segment) {
        CommitLogSegment.CDCState oldState;
        if (segment.getCDCState() != CommitLogSegment.CDCState.FORBIDDEN) {
            return;
        }
        if (this.cdcSizeTracker.hasSpaceForNewSegment() && (oldState = segment.setCDCState(CommitLogSegment.CDCState.PERMITTED)) == CommitLogSegment.CDCState.FORBIDDEN) {
            FileUtils.createHardLink(segment.logFile, segment.getCDCFile());
            this.cdcSizeTracker.addSize(DatabaseDescriptor.getCommitLogSegmentSize());
        }
    }

    private void throwIfForbidden(Mutation mutation, CommitLogSegment segment) throws CDCWriteException {
        if (mutation.trackedByCDC() && segment.getCDCState() == CommitLogSegment.CDCState.FORBIDDEN) {
            String logMsg = String.format("Rejecting mutation to keyspace %s. Free up space in %s by processing CDC logs. Total CDC bytes on disk is %s.", mutation.getKeyspaceName(), DatabaseDescriptor.getCDCLogLocation(), this.cdcSizeTracker.totalCDCSizeOnDisk());
            this.cdcSizeTracker.submitOverflowSizeRecalculation();
            NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 10L, TimeUnit.SECONDS, logMsg, new Object[0]);
            throw new CDCWriteException(logMsg);
        }
    }

    @Override
    public CommitLogSegment createSegment() {
        CommitLogSegment segment = CommitLogSegment.createSegment(this.commitLog, this);
        this.cdcSizeTracker.processNewSegment(segment);
        if (segment.getCDCState() == CommitLogSegment.CDCState.PERMITTED) {
            FileUtils.createHardLink(segment.logFile, segment.getCDCFile());
        }
        return segment;
    }

    @Override
    void handleReplayedSegment(File file) {
        super.handleReplayedSegment(file);
        File cdcFile = new File(DatabaseDescriptor.getCDCLogLocation(), file.getName());
        File cdcIndexFile = new File(DatabaseDescriptor.getCDCLogLocation(), CommitLogDescriptor.fromFileName(file.getName()).cdcIndexFileName());
        if (cdcFile.exists() && !cdcIndexFile.exists()) {
            logger.trace("(Unopened) CDC segment {} is no longer needed and will be deleted now", (Object)cdcFile);
            FileUtils.deleteWithConfirm(cdcFile);
        }
    }

    public void addCDCSize(long size) {
        this.cdcSizeTracker.addSize(size);
    }

    @VisibleForTesting
    public long updateCDCTotalSize() {
        this.cdcSizeTracker.submitOverflowSizeRecalculation();
        try {
            Thread.sleep(DatabaseDescriptor.getCDCDiskCheckInterval() + 10);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        if (this.allocatingFrom().getCDCState() == CommitLogSegment.CDCState.FORBIDDEN) {
            this.cdcSizeTracker.processNewSegment(this.allocatingFrom());
        }
        return this.cdcSizeTracker.totalCDCSizeOnDisk();
    }

    private static class CDCSizeTracker
    extends DirectorySizeCalculator {
        private final RateLimiter rateLimiter = RateLimiter.create((double)(1000.0 / (double)DatabaseDescriptor.getCDCDiskCheckInterval()));
        private ExecutorService cdcSizeCalculationExecutor;
        private volatile long sizeInProgress = 0L;

        CDCSizeTracker(File path) {
            super(path);
        }

        public void start() {
            this.size = 0L;
            this.cdcSizeCalculationExecutor = new ThreadPoolExecutor(1, 1, 1000L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), new ThreadPoolExecutor.DiscardPolicy());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void processNewSegment(CommitLogSegment segment) {
            Object object = segment.cdcStateLock;
            synchronized (object) {
                segment.setCDCState(this.hasSpaceForNewSegment() ? CommitLogSegment.CDCState.PERMITTED : CommitLogSegment.CDCState.FORBIDDEN);
                if (segment.getCDCState() == CommitLogSegment.CDCState.PERMITTED) {
                    this.size += (long)this.defaultSegmentSize();
                }
            }
            this.submitOverflowSizeRecalculation();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void processDiscardedSegment(CommitLogSegment segment) {
            Object object = segment.cdcStateLock;
            synchronized (object) {
                if (segment.getCDCState() == CommitLogSegment.CDCState.CONTAINS) {
                    this.size += segment.onDiskSize();
                }
                if (segment.getCDCState() != CommitLogSegment.CDCState.FORBIDDEN) {
                    this.size -= (long)this.defaultSegmentSize();
                }
            }
            this.submitOverflowSizeRecalculation();
        }

        long allowableCDCBytes() {
            return (long)DatabaseDescriptor.getCDCSpaceInMB() * 1024L * 1024L;
        }

        public void submitOverflowSizeRecalculation() {
            try {
                this.cdcSizeCalculationExecutor.submit(() -> {
                    this.rateLimiter.acquire();
                    this.calculateSize();
                });
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }

        private int defaultSegmentSize() {
            return DatabaseDescriptor.getCommitLogSegmentSize();
        }

        private void calculateSize() {
            try {
                this.sizeInProgress = 0L;
                Files.walkFileTree(this.path.toPath(), this);
                this.size = this.sizeInProgress;
            }
            catch (IOException ie) {
                CommitLog.handleCommitError("Failed CDC Size Calculation", ie);
            }
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            this.sizeInProgress += attrs.size();
            return FileVisitResult.CONTINUE;
        }

        public void shutdown() {
            if (this.cdcSizeCalculationExecutor != null && !this.cdcSizeCalculationExecutor.isShutdown()) {
                this.cdcSizeCalculationExecutor.shutdown();
            }
        }

        private void addSize(long toAdd) {
            this.size += toAdd;
        }

        private long totalCDCSizeOnDisk() {
            return this.size;
        }

        private boolean hasSpaceForNewSegment() {
            return (long)this.defaultSegmentSize() + this.totalCDCSizeOnDisk() <= this.allowableCDCBytes();
        }
    }
}

