/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.webmonitor.threadinfo;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.runtime.execution.ExecutionState;
import org.apache.flink.runtime.executiongraph.AccessExecution;
import org.apache.flink.runtime.executiongraph.AccessExecutionJobVertex;
import org.apache.flink.runtime.executiongraph.AccessExecutionVertex;
import org.apache.flink.runtime.executiongraph.ExecutionAttemptID;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.runtime.messages.ThreadInfoSample;
import org.apache.flink.runtime.resourcemanager.ResourceManagerGateway;
import org.apache.flink.runtime.taskexecutor.TaskExecutorThreadInfoGateway;
import org.apache.flink.runtime.taskmanager.TaskManagerLocation;
import org.apache.flink.runtime.webmonitor.retriever.GatewayRetriever;
import org.apache.flink.runtime.webmonitor.stats.VertexStatsTracker;
import org.apache.flink.runtime.webmonitor.threadinfo.ThreadInfoRequestCoordinator;
import org.apache.flink.runtime.webmonitor.threadinfo.VertexThreadInfoStats;
import org.apache.flink.shaded.guava31.com.google.common.cache.Cache;
import org.apache.flink.shaded.guava31.com.google.common.collect.ImmutableSet;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VertexThreadInfoTracker
implements VertexStatsTracker<VertexThreadInfoStats> {
    private static final Logger LOG = LoggerFactory.getLogger(VertexThreadInfoTracker.class);
    private final Object lock = new Object();
    @GuardedBy(value="lock")
    private final ThreadInfoRequestCoordinator coordinator;
    private final ExecutorService executor;
    private final GatewayRetriever<ResourceManagerGateway> resourceManagerGatewayRetriever;
    @GuardedBy(value="lock")
    private final Cache<JobVertexKey, VertexThreadInfoStats> jobVertexStatsCache;
    @GuardedBy(value="lock")
    private final Set<JobVertexKey> pendingJobVertexStats = new HashSet<JobVertexKey>();
    @GuardedBy(value="lock")
    private final Cache<ExecutionVertexKey, VertexThreadInfoStats> executionVertexStatsCache;
    @GuardedBy(value="lock")
    private final Set<ExecutionVertexKey> pendingExecutionVertexStats = new HashSet<ExecutionVertexKey>();
    private final int numSamples;
    private final Duration statsRefreshInterval;
    private final Duration delayBetweenSamples;
    private final int maxThreadInfoDepth;
    private final CompletableFuture<Void> resultAvailableFuture = new CompletableFuture();
    private boolean shutDown;
    private final Time rpcTimeout;

    VertexThreadInfoTracker(ThreadInfoRequestCoordinator coordinator, GatewayRetriever<ResourceManagerGateway> resourceManagerGatewayRetriever, ScheduledExecutorService executor, Duration cleanUpInterval, int numSamples, Duration statsRefreshInterval, Duration delayBetweenSamples, int maxStackTraceDepth, Time rpcTimeout, Cache<JobVertexKey, VertexThreadInfoStats> jobVertexStatsCache, Cache<ExecutionVertexKey, VertexThreadInfoStats> executionVertexStatsCache) {
        this.coordinator = (ThreadInfoRequestCoordinator)Preconditions.checkNotNull((Object)coordinator, (String)"Thread info samples coordinator");
        this.resourceManagerGatewayRetriever = (GatewayRetriever)Preconditions.checkNotNull(resourceManagerGatewayRetriever, (String)"Gateway retriever");
        this.executor = (ExecutorService)Preconditions.checkNotNull((Object)executor, (String)"Scheduled executor");
        this.statsRefreshInterval = (Duration)Preconditions.checkNotNull((Object)statsRefreshInterval, (String)"Statistics refresh interval");
        this.rpcTimeout = rpcTimeout;
        Preconditions.checkArgument((cleanUpInterval.toMillis() > 0L ? 1 : 0) != 0, (Object)"Clean up interval must be greater than 0");
        Preconditions.checkArgument((numSamples >= 1 ? 1 : 0) != 0, (Object)"Number of samples");
        this.numSamples = numSamples;
        Preconditions.checkArgument((statsRefreshInterval.toMillis() > 0L ? 1 : 0) != 0, (Object)"Stats refresh interval must be greater than 0");
        this.delayBetweenSamples = (Duration)Preconditions.checkNotNull((Object)delayBetweenSamples, (String)"Delay between samples");
        Preconditions.checkArgument((maxStackTraceDepth > 0 ? 1 : 0) != 0, (Object)"Max stack trace depth must be greater than 0");
        this.maxThreadInfoDepth = maxStackTraceDepth;
        this.jobVertexStatsCache = (Cache)Preconditions.checkNotNull(jobVertexStatsCache, (String)"Job vertex stats cache");
        this.executionVertexStatsCache = (Cache)Preconditions.checkNotNull(executionVertexStatsCache, (String)"Execution vertex stats cache");
        executor.scheduleWithFixedDelay(this::cleanUpStatsCache, cleanUpInterval.toMillis(), cleanUpInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<VertexThreadInfoStats> getJobVertexStats(JobID jobId, AccessExecutionJobVertex vertex) {
        Object object = this.lock;
        synchronized (object) {
            JobVertexKey jobVertexKey = VertexThreadInfoTracker.getKey(jobId, vertex);
            VertexThreadInfoStats stats = (VertexThreadInfoStats)this.jobVertexStatsCache.getIfPresent((Object)jobVertexKey);
            if (stats == null || System.currentTimeMillis() >= stats.getEndTime() + this.statsRefreshInterval.toMillis()) {
                this.triggerThreadInfoSampleInternal(jobVertexKey, vertex);
            }
            return Optional.ofNullable(stats);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<VertexThreadInfoStats> getExecutionVertexStats(JobID jobId, AccessExecutionJobVertex vertex, int subtaskIndex) {
        Object object = this.lock;
        synchronized (object) {
            ExecutionVertexKey executionVertexKey = VertexThreadInfoTracker.getKey(jobId, vertex, subtaskIndex);
            VertexThreadInfoStats stats = (VertexThreadInfoStats)this.executionVertexStatsCache.getIfPresent((Object)executionVertexKey);
            if (stats == null || System.currentTimeMillis() >= stats.getEndTime() + this.statsRefreshInterval.toMillis()) {
                this.triggerThreadInfoSampleInternal(executionVertexKey, vertex);
            }
            return Optional.ofNullable(stats);
        }
    }

    private void triggerThreadInfoSampleInternal(JobVertexKey jobVertexKey, AccessExecutionJobVertex vertex) {
        assert (Thread.holdsLock(this.lock));
        if (this.shutDown) {
            return;
        }
        if (this.pendingJobVertexStats.contains(jobVertexKey)) {
            return;
        }
        this.pendingJobVertexStats.add(jobVertexKey);
        this.triggerThreadInfoRequestForVertices(new JobVertexThreadInfoSampleCompletionCallback(jobVertexKey, vertex.getName()), vertex.getTaskVertices());
    }

    private void triggerThreadInfoSampleInternal(ExecutionVertexKey executionVertexKey, AccessExecutionJobVertex vertex) {
        assert (Thread.holdsLock(this.lock));
        if (this.shutDown) {
            return;
        }
        if (this.pendingJobVertexStats.contains(executionVertexKey.getJobVertexKey()) || this.pendingExecutionVertexStats.contains(executionVertexKey)) {
            return;
        }
        this.pendingExecutionVertexStats.add(executionVertexKey);
        AccessExecutionVertex[] executionVertices = (AccessExecutionVertex[])Arrays.stream(vertex.getTaskVertices()).filter(executionVertex -> executionVertex.getParallelSubtaskIndex() == executionVertexKey.subtaskIndex).toArray(AccessExecutionVertex[]::new);
        if (executionVertices.length == 0) {
            return;
        }
        this.triggerThreadInfoRequestForVertices(new ExecutionVertexThreadInfoSampleCompletionCallback(executionVertexKey, executionVertices[0].getTaskNameWithSubtaskIndex()), executionVertices);
    }

    private void triggerThreadInfoRequestForVertices(ThreadInfoSampleCompletionCallback completionCallback, AccessExecutionVertex[] executionVertices) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Triggering thread info sample for tasks: {}", (Object)Arrays.toString(executionVertices));
        }
        CompletableFuture<ResourceManagerGateway> gatewayFuture = this.resourceManagerGatewayRetriever.getFuture();
        ((CompletableFuture)gatewayFuture.thenCompose(resourceManagerGateway -> this.coordinator.triggerThreadInfoRequest(this.matchExecutionsWithGateways(executionVertices, (ResourceManagerGateway)resourceManagerGateway), this.numSamples, this.delayBetweenSamples, this.maxThreadInfoDepth))).whenCompleteAsync((BiConsumer)completionCallback, (Executor)this.executor);
    }

    private Map<ImmutableSet<ExecutionAttemptID>, CompletableFuture<TaskExecutorThreadInfoGateway>> matchExecutionsWithGateways(AccessExecutionVertex[] executionVertices, ResourceManagerGateway resourceManagerGateway) {
        Map<TaskManagerLocation, ImmutableSet<ExecutionAttemptID>> executionsByLocation = this.groupExecutionsByLocation(executionVertices);
        return this.mapExecutionsToGateways(resourceManagerGateway, executionsByLocation);
    }

    private Map<ImmutableSet<ExecutionAttemptID>, CompletableFuture<TaskExecutorThreadInfoGateway>> mapExecutionsToGateways(ResourceManagerGateway resourceManagerGateway, Map<TaskManagerLocation, ImmutableSet<ExecutionAttemptID>> verticesByLocation) {
        HashMap<ImmutableSet<ExecutionAttemptID>, CompletableFuture<TaskExecutorThreadInfoGateway>> executionsWithGateways = new HashMap<ImmutableSet<ExecutionAttemptID>, CompletableFuture<TaskExecutorThreadInfoGateway>>();
        for (Map.Entry<TaskManagerLocation, ImmutableSet<ExecutionAttemptID>> entry : verticesByLocation.entrySet()) {
            TaskManagerLocation tmLocation = entry.getKey();
            ImmutableSet<ExecutionAttemptID> attemptIds = entry.getValue();
            CompletableFuture<TaskExecutorThreadInfoGateway> taskExecutorGatewayFuture = resourceManagerGateway.requestTaskExecutorThreadInfoGateway(tmLocation.getResourceID(), this.rpcTimeout);
            executionsWithGateways.put(attemptIds, taskExecutorGatewayFuture);
        }
        return executionsWithGateways;
    }

    private Map<TaskManagerLocation, ImmutableSet<ExecutionAttemptID>> groupExecutionsByLocation(AccessExecutionVertex[] executionVertices) {
        HashMap<TaskManagerLocation, Set> executionAttemptsByLocation = new HashMap<TaskManagerLocation, Set>();
        for (AccessExecutionVertex executionVertex : executionVertices) {
            if (executionVertex.getExecutionState() != ExecutionState.RUNNING && executionVertex.getExecutionState() != ExecutionState.INITIALIZING) {
                LOG.trace("{} not running or initializing, but {}; not sampling", (Object)executionVertex.getTaskNameWithSubtaskIndex(), (Object)executionVertex.getExecutionState());
                continue;
            }
            for (AccessExecution execution : executionVertex.getCurrentExecutions()) {
                TaskManagerLocation tmLocation = execution.getAssignedResourceLocation();
                if (tmLocation == null) {
                    LOG.trace("ExecutionVertex {} is currently not assigned", (Object)executionVertex);
                    continue;
                }
                Set groupedAttemptIds = executionAttemptsByLocation.getOrDefault(tmLocation, new HashSet());
                ExecutionAttemptID attemptId = execution.getAttemptId();
                groupedAttemptIds.add(attemptId);
                executionAttemptsByLocation.put(tmLocation, groupedAttemptIds);
            }
        }
        return executionAttemptsByLocation.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ImmutableSet.copyOf((Collection)((Collection)e.getValue()))));
    }

    @VisibleForTesting
    void cleanUpStatsCache() {
        this.jobVertexStatsCache.cleanUp();
        this.executionVertexStatsCache.cleanUp();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void shutDown() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.shutDown) {
                this.jobVertexStatsCache.invalidateAll();
                this.pendingJobVertexStats.clear();
                this.executionVertexStatsCache.invalidateAll();
                this.pendingExecutionVertexStats.clear();
                this.shutDown = true;
            }
        }
    }

    @VisibleForTesting
    CompletableFuture<Void> getResultAvailableFuture() {
        return this.resultAvailableFuture;
    }

    private static JobVertexKey getKey(JobID jobId, AccessExecutionJobVertex vertex) {
        return new JobVertexKey(jobId, vertex.getJobVertexId());
    }

    private static ExecutionVertexKey getKey(JobID jobId, AccessExecutionJobVertex vertex, int subtaskIndex) {
        return new ExecutionVertexKey(jobId, vertex.getJobVertexId(), subtaskIndex);
    }

    private class ExecutionVertexThreadInfoSampleCompletionCallback
    extends ThreadInfoSampleCompletionCallback {
        private final ExecutionVertexKey executionVertexKey;

        ExecutionVertexThreadInfoSampleCompletionCallback(ExecutionVertexKey executionVertexKey, String sampleName) {
            super(sampleName);
            this.executionVertexKey = executionVertexKey;
        }

        @Override
        protected void handleResult(VertexThreadInfoStats threadInfoStats) {
            VertexThreadInfoTracker.this.executionVertexStatsCache.put((Object)this.executionVertexKey, (Object)threadInfoStats);
        }

        @Override
        protected void doFinally() {
            VertexThreadInfoTracker.this.pendingExecutionVertexStats.remove(this.executionVertexKey);
        }
    }

    private class JobVertexThreadInfoSampleCompletionCallback
    extends ThreadInfoSampleCompletionCallback {
        private final JobVertexKey jobVertexKey;

        JobVertexThreadInfoSampleCompletionCallback(JobVertexKey jobVertexKey, String sampleName) {
            super(sampleName);
            this.jobVertexKey = jobVertexKey;
        }

        @Override
        protected void handleResult(VertexThreadInfoStats threadInfoStats) {
            VertexThreadInfoTracker.this.jobVertexStatsCache.put((Object)this.jobVertexKey, (Object)threadInfoStats);
            for (Map.Entry<ExecutionAttemptID, Collection<ThreadInfoSample>> entry : threadInfoStats.getSamplesBySubtask().entrySet()) {
                ExecutionAttemptID executionAttemptID = entry.getKey();
                ExecutionVertexKey executionVertexKey = this.jobVertexKey.toExecutionVertexKey(executionAttemptID.getSubtaskIndex());
                VertexThreadInfoStats oldStats = (VertexThreadInfoStats)VertexThreadInfoTracker.this.executionVertexStatsCache.getIfPresent((Object)executionVertexKey);
                if (oldStats == null || oldStats.getRequestId() < threadInfoStats.getRequestId()) {
                    VertexThreadInfoTracker.this.executionVertexStatsCache.put((Object)executionVertexKey, (Object)this.generateExecutionVertexStats(threadInfoStats, executionAttemptID, entry.getValue()));
                    continue;
                }
                if (oldStats.getRequestId() != threadInfoStats.getRequestId()) continue;
                Map<ExecutionAttemptID, Collection<ThreadInfoSample>> samples = oldStats.getSamplesBySubtask();
                samples.put(executionAttemptID, entry.getValue());
            }
        }

        private VertexThreadInfoStats generateExecutionVertexStats(VertexThreadInfoStats threadInfoStats, ExecutionAttemptID executionAttemptID, Collection<ThreadInfoSample> sample) {
            HashMap<ExecutionAttemptID, Collection<ThreadInfoSample>> samples = new HashMap<ExecutionAttemptID, Collection<ThreadInfoSample>>();
            samples.put(executionAttemptID, sample);
            return new VertexThreadInfoStats(threadInfoStats.getRequestId(), threadInfoStats.getStartTime(), threadInfoStats.getEndTime(), samples);
        }

        @Override
        protected void doFinally() {
            VertexThreadInfoTracker.this.pendingJobVertexStats.remove(this.jobVertexKey);
        }
    }

    private abstract class ThreadInfoSampleCompletionCallback
    implements BiConsumer<VertexThreadInfoStats, Throwable> {
        private final String sampleName;

        protected ThreadInfoSampleCompletionCallback(String sampleName) {
            this.sampleName = sampleName;
        }

        protected abstract void handleResult(VertexThreadInfoStats var1);

        protected abstract void doFinally();

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void accept(VertexThreadInfoStats threadInfoStats, Throwable throwable) {
            Object object = VertexThreadInfoTracker.this.lock;
            synchronized (object) {
                try {
                    if (VertexThreadInfoTracker.this.shutDown) {
                        return;
                    }
                    if (threadInfoStats == null) {
                        LOG.error("Failed to gather a thread info sample for {}", (Object)this.sampleName, (Object)throwable);
                        return;
                    }
                    this.handleResult(threadInfoStats);
                    VertexThreadInfoTracker.this.resultAvailableFuture.complete(null);
                }
                catch (Throwable t) {
                    LOG.error("Error during stats completion.", t);
                }
                finally {
                    this.doFinally();
                }
                return;
            }
        }
    }

    static class ExecutionVertexKey {
        private final JobVertexKey jobVertexKey;
        private final int subtaskIndex;

        private ExecutionVertexKey(JobID jobId, JobVertexID jobVertexId, int subtaskIndex) {
            this.jobVertexKey = new JobVertexKey(jobId, jobVertexId);
            this.subtaskIndex = subtaskIndex;
        }

        private JobVertexKey getJobVertexKey() {
            return this.jobVertexKey;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ExecutionVertexKey executionVertexKey = (ExecutionVertexKey)o;
            return Objects.equals(this.jobVertexKey, executionVertexKey.jobVertexKey) && Objects.equals(this.subtaskIndex, executionVertexKey.subtaskIndex);
        }

        public int hashCode() {
            return Objects.hash(this.jobVertexKey, this.subtaskIndex);
        }
    }

    static class JobVertexKey {
        private final JobID jobId;
        private final JobVertexID jobVertexId;

        private JobVertexKey(JobID jobId, JobVertexID jobVertexId) {
            this.jobId = jobId;
            this.jobVertexId = jobVertexId;
        }

        private ExecutionVertexKey toExecutionVertexKey(int subtaskIndex) {
            return new ExecutionVertexKey(this.jobId, this.jobVertexId, subtaskIndex);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            JobVertexKey jobVertexKey = (JobVertexKey)o;
            return Objects.equals(this.jobId, jobVertexKey.jobId) && Objects.equals(this.jobVertexId, jobVertexKey.jobVertexId);
        }

        public int hashCode() {
            return Objects.hash(this.jobId, this.jobVertexId);
        }
    }
}

