/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.Closeable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Locale;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.constants.AbfsHttpConstants;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.SASTokenProviderException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AppendRequestParameters;
import org.apache.hadoop.fs.azurebfs.extensions.ExtensionHelper;
import org.apache.hadoop.fs.azurebfs.extensions.SASTokenProvider;
import org.apache.hadoop.fs.azurebfs.oauth2.AccessTokenProvider;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientContext;
import org.apache.hadoop.fs.azurebfs.services.AbfsCounters;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpHeader;
import org.apache.hadoop.fs.azurebfs.services.AbfsPerfTracker;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperationType;
import org.apache.hadoop.fs.azurebfs.services.AbfsUriQueryBuilder;
import org.apache.hadoop.fs.azurebfs.services.AuthType;
import org.apache.hadoop.fs.azurebfs.services.ExponentialRetryPolicy;
import org.apache.hadoop.fs.azurebfs.services.SharedKeyCredentials;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.ssl.DelegatingSSLSocketFactory;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.FutureCallback;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.Futures;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableScheduledFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListeningScheduledExecutorService;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.MoreExecutors;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AbfsClient
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(AbfsClient.class);
    private final URL baseUrl;
    private final SharedKeyCredentials sharedKeyCredentials;
    private final String xMsVersion = "2019-12-12";
    private final ExponentialRetryPolicy retryPolicy;
    private final String filesystem;
    private final AbfsConfiguration abfsConfiguration;
    private final String userAgent;
    private final AbfsPerfTracker abfsPerfTracker;
    private final String clientProvidedEncryptionKey;
    private final String clientProvidedEncryptionKeySHA;
    private final String accountName;
    private final AuthType authType;
    private AccessTokenProvider tokenProvider;
    private SASTokenProvider sasTokenProvider;
    private final AbfsCounters abfsCounters;
    private final ListeningScheduledExecutorService executorService;

    private AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, AbfsClientContext abfsClientContext) throws IOException {
        this.baseUrl = baseUrl;
        this.sharedKeyCredentials = sharedKeyCredentials;
        String baseUrlString = baseUrl.toString();
        this.filesystem = baseUrlString.substring(baseUrlString.lastIndexOf("/") + 1);
        this.abfsConfiguration = abfsConfiguration;
        this.retryPolicy = abfsClientContext.getExponentialRetryPolicy();
        this.accountName = abfsConfiguration.getAccountName().substring(0, abfsConfiguration.getAccountName().indexOf("."));
        this.authType = abfsConfiguration.getAuthType(this.accountName);
        String encryptionKey = this.abfsConfiguration.getClientProvidedEncryptionKey();
        if (encryptionKey != null) {
            this.clientProvidedEncryptionKey = this.getBase64EncodedString(encryptionKey);
            this.clientProvidedEncryptionKeySHA = this.getBase64EncodedString(this.getSHA256Hash(encryptionKey));
        } else {
            this.clientProvidedEncryptionKey = null;
            this.clientProvidedEncryptionKeySHA = null;
        }
        String sslProviderName = null;
        if (this.baseUrl.toString().startsWith("https")) {
            try {
                LOG.trace("Initializing DelegatingSSLSocketFactory with {} SSL Channel Mode", (Object)this.abfsConfiguration.getPreferredSSLFactoryOption());
                DelegatingSSLSocketFactory.initializeDefaultFactory((DelegatingSSLSocketFactory.SSLChannelMode)this.abfsConfiguration.getPreferredSSLFactoryOption());
                sslProviderName = DelegatingSSLSocketFactory.getDefaultFactory().getProviderName();
            }
            catch (IOException e) {
                LOG.trace("NonCritFailure: DelegatingSSLSocketFactory Init failed : {}", (Object)e.getMessage());
            }
        }
        this.userAgent = this.initializeUserAgent(abfsConfiguration, sslProviderName);
        this.abfsPerfTracker = abfsClientContext.getAbfsPerfTracker();
        this.abfsCounters = abfsClientContext.getAbfsCounters();
        ThreadFactory tf = new ThreadFactoryBuilder().setNameFormat("AbfsClient Lease Ops").setDaemon(true).build();
        this.executorService = MoreExecutors.listeningDecorator((ScheduledExecutorService)HadoopExecutors.newScheduledThreadPool((int)this.abfsConfiguration.getNumLeaseThreads(), (ThreadFactory)tf));
    }

    public AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, AccessTokenProvider tokenProvider, AbfsClientContext abfsClientContext) throws IOException {
        this(baseUrl, sharedKeyCredentials, abfsConfiguration, abfsClientContext);
        this.tokenProvider = tokenProvider;
    }

    public AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, SASTokenProvider sasTokenProvider, AbfsClientContext abfsClientContext) throws IOException {
        this(baseUrl, sharedKeyCredentials, abfsConfiguration, abfsClientContext);
        this.sasTokenProvider = sasTokenProvider;
    }

    private byte[] getSHA256Hash(String key) throws IOException {
        try {
            MessageDigest digester = MessageDigest.getInstance("SHA-256");
            return digester.digest(key.getBytes(StandardCharsets.UTF_8));
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
    }

    private String getBase64EncodedString(String key) {
        return this.getBase64EncodedString(key.getBytes(StandardCharsets.UTF_8));
    }

    private String getBase64EncodedString(byte[] bytes) {
        return Base64.getEncoder().encodeToString(bytes);
    }

    @Override
    public void close() throws IOException {
        if (this.tokenProvider instanceof Closeable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{(Closeable)((Object)this.tokenProvider)});
        }
        HadoopExecutors.shutdown((ExecutorService)this.executorService, (Logger)LOG, (long)0L, (TimeUnit)TimeUnit.SECONDS);
    }

    public String getFileSystem() {
        return this.filesystem;
    }

    protected AbfsPerfTracker getAbfsPerfTracker() {
        return this.abfsPerfTracker;
    }

    ExponentialRetryPolicy getRetryPolicy() {
        return this.retryPolicy;
    }

    SharedKeyCredentials getSharedKeyCredentials() {
        return this.sharedKeyCredentials;
    }

    List<AbfsHttpHeader> createDefaultHeaders() {
        ArrayList<AbfsHttpHeader> requestHeaders = new ArrayList<AbfsHttpHeader>();
        requestHeaders.add(new AbfsHttpHeader("x-ms-version", "2019-12-12"));
        requestHeaders.add(new AbfsHttpHeader("Accept", "application/json, application/octet-stream"));
        requestHeaders.add(new AbfsHttpHeader("Accept-Charset", "utf-8"));
        requestHeaders.add(new AbfsHttpHeader("Content-Type", ""));
        requestHeaders.add(new AbfsHttpHeader("User-Agent", this.userAgent));
        return requestHeaders;
    }

    private void addCustomerProvidedKeyHeaders(List<AbfsHttpHeader> requestHeaders) {
        if (this.clientProvidedEncryptionKey != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-key", this.clientProvidedEncryptionKey));
            requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-key-sha256", this.clientProvidedEncryptionKeySHA));
            requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-algorithm", "AES256"));
        }
    }

    AbfsUriQueryBuilder createDefaultUriQueryBuilder() {
        AbfsUriQueryBuilder abfsUriQueryBuilder = new AbfsUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("timeout", "90");
        return abfsUriQueryBuilder;
    }

    public AbfsRestOperation createFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = new AbfsUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.CreateFileSystem, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setFilesystemProperties(String properties, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-properties", properties));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.SetFileSystemProperties, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation listPath(String relativePath, boolean recursive, int listMaxResults, String continuation, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        abfsUriQueryBuilder.addQuery("directory", AbfsClient.getDirectoryQueryParameter(relativePath));
        abfsUriQueryBuilder.addQuery("recursive", String.valueOf(recursive));
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        abfsUriQueryBuilder.addQuery("maxResults", String.valueOf(listMaxResults));
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(this.abfsConfiguration.isUpnUsed()));
        this.appendSASTokenToQuery(relativePath, "list", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.ListPaths, this, "GET", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getFilesystemProperties(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.GetFileSystemProperties, this, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation deleteFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.DeleteFileSystem, this, "DELETE", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation createPath(String path, boolean isFile, boolean overwrite, String permission, String umask, boolean isAppendBlob, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        if (isFile) {
            this.addCustomerProvidedKeyHeaders(requestHeaders);
        }
        if (!overwrite) {
            requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        }
        if (permission != null && !permission.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-permissions", permission));
        }
        if (umask != null && !umask.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-umask", umask));
        }
        if (eTag != null && !eTag.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", isFile ? "file" : "directory");
        if (isAppendBlob) {
            abfsUriQueryBuilder.addQuery("blobtype", "appendblob");
        }
        String operation = isFile ? "create-file" : "create-directory";
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.CreatePath, this, "PUT", url, requestHeaders);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            String existingResource;
            if (!op.hasResult()) {
                throw ex;
            }
            if (!isFile && op.getResult().getStatusCode() == 409 && (existingResource = op.getResult().getResponseHeader("x-ms-existing-resource-type")) != null && existingResource.equals("directory")) {
                return op;
            }
            throw ex;
        }
        return op;
    }

    public AbfsRestOperation acquireLease(String path, int duration, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "acquire"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-duration", Integer.toString(duration)));
        requestHeaders.add(new AbfsHttpHeader("x-ms-proposed-lease-id", UUID.randomUUID().toString()));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.LeasePath, this, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation renewLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "renew"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.LeasePath, this, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation releaseLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "release"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.LeasePath, this, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation breakLease(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "break"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-break-period", "0"));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.LeasePath, this, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation renamePath(String source, String destination, String continuation, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        String encodedRenameSource = AbfsClient.urlEncode("/" + this.getFileSystem() + source);
        if (this.authType == AuthType.SAS) {
            AbfsUriQueryBuilder srcQueryBuilder = new AbfsUriQueryBuilder();
            this.appendSASTokenToQuery(source, "rename-source", srcQueryBuilder);
            encodedRenameSource = encodedRenameSource + srcQueryBuilder.toString();
        }
        LOG.trace("Rename source queryparam added {}", (Object)encodedRenameSource);
        requestHeaders.add(new AbfsHttpHeader("x-ms-rename-source", encodedRenameSource));
        requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        this.appendSASTokenToQuery(destination, "rename-destination", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(destination, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.RenamePath, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation append(String path, byte[] buffer, AppendRequestParameters reqParams, String cachedSasToken, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addCustomerProvidedKeyHeaders(requestHeaders);
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (reqParams.getLeaseId() != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", reqParams.getLeaseId()));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "append");
        abfsUriQueryBuilder.addQuery("position", Long.toString(reqParams.getPosition()));
        if (reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_MODE || reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_CLOSE_MODE) {
            abfsUriQueryBuilder.addQuery("flush", "true");
            if (reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_CLOSE_MODE) {
                abfsUriQueryBuilder.addQuery("close", "true");
            }
        }
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.Append, this, "PUT", url, requestHeaders, buffer, reqParams.getoffset(), reqParams.getLength(), sasTokenForReuse);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException e) {
            if (!op.hasResult()) {
                throw e;
            }
            if (reqParams.isAppendBlob() && this.appendSuccessCheckOp(op, path, reqParams.getPosition() + (long)reqParams.getLength(), tracingContext)) {
                AbfsRestOperation successOp = new AbfsRestOperation(AbfsRestOperationType.Append, this, "PUT", url, requestHeaders, buffer, reqParams.getoffset(), reqParams.getLength(), sasTokenForReuse);
                successOp.hardSetResult(200);
                return successOp;
            }
            throw e;
        }
        return op;
    }

    public boolean appendSuccessCheckOp(AbfsRestOperation op, String path, long length, TracingContext tracingContext) throws AzureBlobFileSystemException {
        String fileLength;
        AbfsRestOperation destStatusOp;
        if (op.isARetriedRequest() && op.getResult().getStatusCode() == 400 && (destStatusOp = this.getPathStatus(path, false, tracingContext)).getResult().getStatusCode() == 200 && length <= Long.parseLong(fileLength = destStatusOp.getResult().getResponseHeader("Content-Length"))) {
            LOG.debug("Returning success response from append blob idempotency code");
            return true;
        }
        return false;
    }

    public AbfsRestOperation flush(String path, long position, boolean retainUncommittedData, boolean isClose, String cachedSasToken, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addCustomerProvidedKeyHeaders(requestHeaders);
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (leaseId != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "flush");
        abfsUriQueryBuilder.addQuery("position", Long.toString(position));
        abfsUriQueryBuilder.addQuery("retainUncommittedData", String.valueOf(retainUncommittedData));
        abfsUriQueryBuilder.addQuery("close", String.valueOf(isClose));
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.Flush, this, "PUT", url, requestHeaders, sasTokenForReuse);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setPathProperties(String path, String properties, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addCustomerProvidedKeyHeaders(requestHeaders);
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-properties", properties));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setProperties");
        this.appendSASTokenToQuery(path, "set-properties", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.SetPathProperties, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getPathStatus(String path, boolean includeProperties, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String operation = "get-properties";
        if (!includeProperties) {
            abfsUriQueryBuilder.addQuery("action", "getStatus");
            operation = "get-status";
        } else {
            this.addCustomerProvidedKeyHeaders(requestHeaders);
        }
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(this.abfsConfiguration.isUpnUsed()));
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.GetPathStatus, this, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation read(String path, long position, byte[] buffer, int bufferOffset, int bufferLength, String eTag, String cachedSasToken, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addCustomerProvidedKeyHeaders(requestHeaders);
        requestHeaders.add(new AbfsHttpHeader("Range", String.format("bytes=%d-%d", position, position + (long)bufferLength - 1L)));
        requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "read", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.ReadFile, this, "GET", url, requestHeaders, buffer, bufferOffset, bufferLength, sasTokenForReuse);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation deletePath(String path, boolean recursive, String continuation, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("recursive", String.valueOf(recursive));
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        String operation = recursive ? "delete-recursive" : "delete";
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.DeletePath, this, "DELETE", url, requestHeaders);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException e) {
            if (!op.hasResult()) {
                throw e;
            }
            AbfsRestOperation idempotencyOp = this.deleteIdempotencyCheckOp(op);
            if (idempotencyOp.getResult().getStatusCode() == op.getResult().getStatusCode()) {
                throw e;
            }
            return idempotencyOp;
        }
        return op;
    }

    public AbfsRestOperation deleteIdempotencyCheckOp(AbfsRestOperation op) {
        Preconditions.checkArgument((boolean)op.hasResult(), (Object)"Operations has null HTTP response");
        if (op.isARetriedRequest() && op.getResult().getStatusCode() == 404) {
            AbfsRestOperation successOp = new AbfsRestOperation(AbfsRestOperationType.DeletePath, this, "DELETE", op.getUrl(), op.getRequestHeaders());
            successOp.hardSetResult(200);
            LOG.debug("Returning success response from delete idempotency logic");
            return successOp;
        }
        return op;
    }

    public AbfsRestOperation setOwner(String path, String owner, String group, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (owner != null && !owner.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-owner", owner));
        }
        if (group != null && !group.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-group", group));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-owner", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.SetOwner, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setPermission(String path, String permission, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-permissions", permission));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-permission", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.SetPermissions, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setAcl(String path, String aclSpecString, TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.setAcl(path, aclSpecString, "", tracingContext);
    }

    public AbfsRestOperation setAcl(String path, String aclSpecString, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-acl", aclSpecString));
        if (eTag != null && !eTag.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-acl", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.SetAcl, this, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getAclStatus(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.getAclStatus(path, this.abfsConfiguration.isUpnUsed(), tracingContext);
    }

    public AbfsRestOperation getAclStatus(String path, boolean useUPN, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "getAccessControl");
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(useUPN));
        this.appendSASTokenToQuery(path, "get-acl", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.GetAcl, this, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation checkAccess(String path, String rwx, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "checkAccess");
        abfsUriQueryBuilder.addQuery("fsAction", rwx);
        this.appendSASTokenToQuery(path, "check-access", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.CheckAccess, this, "HEAD", url, this.createDefaultHeaders());
        op.execute(tracingContext);
        return op;
    }

    public static String getDirectoryQueryParameter(String path) {
        String directory = path;
        if (Strings.isNullOrEmpty((String)directory)) {
            directory = "";
        } else if (directory.charAt(0) == '/') {
            directory = directory.substring(1);
        }
        return directory;
    }

    private String appendSASTokenToQuery(String path, String operation, AbfsUriQueryBuilder queryBuilder) throws SASTokenProviderException {
        return this.appendSASTokenToQuery(path, operation, queryBuilder, null);
    }

    private String appendSASTokenToQuery(String path, String operation, AbfsUriQueryBuilder queryBuilder, String cachedSasToken) throws SASTokenProviderException {
        String sasToken = null;
        if (this.authType == AuthType.SAS) {
            try {
                LOG.trace("Fetch SAS token for {} on {}", (Object)operation, (Object)path);
                if (cachedSasToken == null) {
                    sasToken = this.sasTokenProvider.getSASToken(this.accountName, this.filesystem, path, operation);
                    if (sasToken == null || sasToken.isEmpty()) {
                        throw new UnsupportedOperationException("SASToken received is empty or null");
                    }
                } else {
                    sasToken = cachedSasToken;
                    LOG.trace("Using cached SAS token.");
                }
                queryBuilder.setSASToken(sasToken);
                LOG.trace("SAS token fetch complete for {} on {}", (Object)operation, (Object)path);
            }
            catch (Exception ex) {
                throw new SASTokenProviderException(String.format("Failed to acquire a SAS token for %s on %s due to %s", operation, path, ex.toString()));
            }
        }
        return sasToken;
    }

    private URL createRequestUrl(String query) throws AzureBlobFileSystemException {
        return this.createRequestUrl("", query);
    }

    @VisibleForTesting
    protected URL createRequestUrl(String path, String query) throws AzureBlobFileSystemException {
        URL url;
        String base = this.baseUrl.toString();
        String encodedPath = path;
        try {
            encodedPath = AbfsClient.urlEncode(path);
        }
        catch (AzureBlobFileSystemException ex) {
            LOG.debug("Unexpected error.", (Throwable)ex);
            throw new InvalidUriException(path);
        }
        StringBuilder sb = new StringBuilder();
        sb.append(base);
        sb.append(encodedPath);
        sb.append(query);
        try {
            url = new URL(sb.toString());
        }
        catch (MalformedURLException ex) {
            throw new InvalidUriException(sb.toString());
        }
        return url;
    }

    public static String urlEncode(String value) throws AzureBlobFileSystemException {
        String encodedString;
        try {
            encodedString = URLEncoder.encode(value, "utf-8").replace("+", "%20").replace("%2F", "/");
        }
        catch (UnsupportedEncodingException ex) {
            throw new InvalidUriException(value);
        }
        return encodedString;
    }

    public synchronized String getAccessToken() throws IOException {
        if (this.tokenProvider != null) {
            return "Bearer " + this.tokenProvider.getToken().getAccessToken();
        }
        return null;
    }

    public AuthType getAuthType() {
        return this.authType;
    }

    @VisibleForTesting
    String initializeUserAgent(AbfsConfiguration abfsConfiguration, String sslProviderName) {
        StringBuilder sb = new StringBuilder();
        sb.append("APN/1.0");
        sb.append(" ");
        sb.append(AbfsHttpConstants.CLIENT_VERSION);
        sb.append(" ");
        sb.append("(");
        sb.append(System.getProperty("java.vendor").replaceAll(" ", ""));
        sb.append(" ");
        sb.append("JavaJRE");
        sb.append(" ");
        sb.append(System.getProperty("java.version"));
        sb.append(";");
        sb.append(" ");
        sb.append(System.getProperty("os.name").replaceAll(" ", ""));
        sb.append(" ");
        sb.append(System.getProperty("os.version"));
        sb.append("/");
        sb.append(System.getProperty("os.arch"));
        sb.append(";");
        this.appendIfNotEmpty(sb, sslProviderName, true);
        this.appendIfNotEmpty(sb, ExtensionHelper.getUserAgentSuffix(this.tokenProvider, ""), true);
        sb.append(" ");
        sb.append(abfsConfiguration.getClusterName());
        sb.append("/");
        sb.append(abfsConfiguration.getClusterType());
        sb.append(")");
        this.appendIfNotEmpty(sb, abfsConfiguration.getCustomUserAgentPrefix(), false);
        return String.format(Locale.ROOT, sb.toString(), new Object[0]);
    }

    private void appendIfNotEmpty(StringBuilder sb, String regEx, boolean shouldAppendSemiColon) {
        if (regEx == null || regEx.trim().isEmpty()) {
            return;
        }
        sb.append(" ");
        sb.append(regEx);
        if (shouldAppendSemiColon) {
            sb.append(";");
        }
    }

    @VisibleForTesting
    URL getBaseUrl() {
        return this.baseUrl;
    }

    @VisibleForTesting
    public SASTokenProvider getSasTokenProvider() {
        return this.sasTokenProvider;
    }

    protected AbfsCounters getAbfsCounters() {
        return this.abfsCounters;
    }

    public int getNumLeaseThreads() {
        return this.abfsConfiguration.getNumLeaseThreads();
    }

    public <V> ListenableScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit timeUnit) {
        return this.executorService.schedule(callable, delay, timeUnit);
    }

    public ListenableFuture<?> submit(Runnable runnable) {
        return this.executorService.submit(runnable);
    }

    public <V> void addCallback(ListenableFuture<V> future, FutureCallback<V> callback) {
        Futures.addCallback(future, callback, (Executor)this.executorService);
    }
}

