/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIESOR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.aries.tx.control.service.common.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

import org.osgi.service.transaction.control.TransactionContext;
import org.osgi.service.transaction.control.TransactionStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractTransactionContextImpl implements TransactionContext {

	private static final Logger logger = LoggerFactory.getLogger(AbstractTransactionContextImpl.class);
	
	protected final AtomicReference<Throwable> firstUnexpectedException = new AtomicReference<>();

	protected final List<Throwable> subsequentExceptions = new ArrayList<>();

	protected final Set<Throwable> ignoredExceptions = Collections.newSetFromMap(new IdentityHashMap<>());

	protected final List<Runnable> preCompletion = new ArrayList<>();

	protected final List<Consumer<TransactionStatus>> postCompletion = new ArrayList<>();
	
	protected final Map<Object, Object> scopedVariables = new HashMap<>();

	@Override
	public Object getScopedValue(Object key) {
		return scopedVariables.get(key);
	}

	@Override
	public void putScopedValue(Object key, Object value) {
		scopedVariables.put(key, value);
	}

	protected void beforeCompletion(Runnable onFirstError) {
		preCompletion.stream().forEach(r -> {
			try {
				r.run();
			} catch (Exception e) {
				if (firstUnexpectedException.compareAndSet(null, e)) {
					onFirstError.run();
				} else {
					subsequentExceptions.add(e);
				}
				logger.warn("A pre-completion callback failed with an exception", e);
			}
		});
	}

	protected void afterCompletion(TransactionStatus status) {
		postCompletion.stream().forEach(c -> {
			try {
				c.accept(status);
			} catch (Exception e) {
				// Post completion failures do not affect the outcome
				logger.warn("A post-completion callback failed with an exception", e);
			}
		});
	}

	protected abstract boolean isAlive();

	protected void recordFailure(Throwable failure) {
		if (!firstUnexpectedException.compareAndSet(null, failure)) {
			subsequentExceptions.add(failure);
		}
	}

	protected abstract void safeSetRollbackOnly();

	public abstract void finish();

	protected void ignoreException(Throwable t) {
		ignoredExceptions.add(t);
	}
}
