/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.plugin.context;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.config.java.plugin.Plugin;
import org.springframework.jmx.export.MBeanExporter;
import org.springframework.jmx.export.annotation.ManagedResource;


/**
 * Activates default exporting of MBeans by detecting standard MBeans in the Spring
 * context as well as {@link ManagedResource @ManagedResource} annotations on
 * Spring-defined beans.
 * <p/>
 * The resulting {@link MBeanExporter} bean is defined under the name "mbeanExporter".
 * Alternatively, consider defining a custom AnnotationMBeanExporter bean explicitly.
 * <p/>
 * This annotation is modeled after and functionally equivalent to Spring XML's
 * {@code <context:mbean-export/>} element
 *
 * @author Chris Beams
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Inherited
@Documented
@Plugin(handler=MBeanExportHandler.class)
public @interface MBeanExport {

    /**
     * The default domain to use when generating JMX ObjectNames.
     */
    String defaultDomain() default "";

    /**
     * The bean name of the MBeanServer to which MBeans should be exported.
     * Default is to use the platform's default MBeanServer (autodetecting
     * WebLogic 9+, WebSphere 5.1+ and the JDK 1.5+ platform MBeanServer).
     */
    String server() default "";

    /**
     * The registration behavior, indicating how to deal with existing MBeans
     * of the same name: fail with an exception, ignore and keep the existing
     * MBean, or replace the existing one with the new MBean.
     * <p/>
     * Default is to fail with an exception.
     */
     RegistrationPolicy registration() default RegistrationPolicy.FAIL_ON_EXISTING;

}
