/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.packagescanning;

import static test.common.util.Assert.assertBeanDefinitionCount;

import org.junit.After;
import org.junit.Test;

import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.context.ConfigurableJavaConfigApplicationContext;
import org.springframework.config.java.context.JavaConfigApplicationContext;

import test.common.domain.company.SimpleConfiguration;

import test.feature.packagescanning.scanpkg.complex.ComplexConfiguration;


/**
 * Integration tests related to package scanning that picks up
 * {@link Configuration} classes (a feature of all
 * {@link ConfigurableJavaConfigApplicationContext} implementations.
 *
 * @author  Chris Beams
 */
public class PackageScanningTests {

    private JavaConfigApplicationContext ctx;

    @After
    public void tearDown() {
        try {
            // just in case some test fails during loading and refresh is not
            // called
            ctx.refresh();
            ctx.close();
        } catch (Exception ex) {
            // ignore - it's cleanup time
        }

        ctx = null;
    }

    @Test
    public void testReadSimplePackage() {
        String pkgName = test.feature.packagescanning.scanpkg.simple._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(pkgName);

        int classesInPackage = 2;
        int beansInClasses = 3;

        assertBeanDefinitionCount(ctx, (classesInPackage + beansInClasses));
    }

    /**
     * complex.* package contains only one top-level configuration class. Inner classes should not
     * be included.
     */
    @Test
    public void testScanningPackageDoesNotIncludeInnerConfigurationClasses() {
        String pkgName = test.feature.packagescanning.scanpkg.complex._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(pkgName);
        assertBeanDefinitionCount(ctx, 2);
    }

    @Test
    public void testOpenEndedConstructionSupplyingBasePackagesBeforeRefresh() {
        ctx = new JavaConfigApplicationContext();
        String pkgName = test.feature.packagescanning.scanpkg.a._package.class.getPackage().getName();
        ctx.addBasePackage(pkgName);
        ctx.refresh();

        assertBeanDefinitionCount(ctx, 2);
    }

    /**
     * Sets up a base package with two classes in it, one of which is abstract and should thus be
     * ignored.
     */
    @Test
    public void testConstructionWithSingleBasePackage() {
        String packageName = test.feature.packagescanning.scanpkg.complex._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(packageName);
        assertBeanDefinitionCount(ctx, 2);
    }

    @Test
    public void testConstructionWithMultipleBasePackages() {
        String packageName1 = test.feature.packagescanning.scanpkg.simple._package.class.getPackage().getName();
        String packageName2 = test.feature.packagescanning.scanpkg.complex._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(packageName1, packageName2);
        assertBeanDefinitionCount(ctx, 7);
    }

    @Test
    public void testConstructionWithSingleClassAndSingleBasePackage() {
        String pkgName = test.common.domain.company._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(
            new Class<?>[] { SimpleConfiguration.class }, new String[] { pkgName });
        assertBeanDefinitionCount(ctx, 5);
    }

    @Test
    public void testConstructionWithMixOfClassesAndBasePackages() {
        String pkg1 = test.feature.packagescanning.scanpkg.simple._package.class.getPackage().getName();
        ctx = new JavaConfigApplicationContext(new Class<?>[] { ComplexConfiguration.class }, new String[] { pkg1 });

        assertBeanDefinitionCount(ctx, 7);
    }

}
