/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.valuesource;

import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;


/**
 * Base class for use by all {@link ValueResolver} implementations that will return String.
 *
 * @author  Rod Johnson
 * @author  Chris Beams
 * @see     #getString(String)
 */
public abstract class AbstractStringBasedValueResolver implements ValueResolver {

    private BeanWrapper beanWrapperImpl = new BeanWrapperImpl();

    /**
     * Return value associated with <var>name</var>, where value is of type <var>requiredType</var>.
     * For String-based values, consider {@link #getString(String)}.
     */
    @SuppressWarnings("unchecked")
    public <T> T resolve(String name, Class<?> requiredType) throws ValueResolutionException {
        String rawValue = getString(name);

        return (T) beanWrapperImpl.convertIfNecessary(rawValue, requiredType);
    }

    /**
     * Return value associated with <var>name</var>.
     *
     * @throws  ValueResolutionException  if <var>name</var> cannot be found within this
     *                                    {@link ValueResolver} - it is incumbent upon all subclasses
     *                                    to adhere to these semantics.
     */
    public abstract String getString(String name) throws ValueResolutionException;

}
