/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.locking;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.gradle.api.file.RegularFile;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.internal.DocumentationRegistry;
import org.gradle.api.internal.DomainObjectContext;
import org.gradle.api.internal.file.FileResolver;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.internal.resource.local.FileResourceListener;
import org.gradle.util.GFileUtils;

public class LockFileReaderWriter {
    private static final Logger LOGGER = Logging.getLogger(LockFileReaderWriter.class);
    private static final DocumentationRegistry DOC_REG = new DocumentationRegistry();
    static final String UNIQUE_LOCKFILE_NAME = "gradle.lockfile";
    static final String FILE_SUFFIX = ".lockfile";
    static final String DEPENDENCY_LOCKING_FOLDER = "gradle/dependency-locks";
    static final Charset CHARSET = StandardCharsets.UTF_8;
    static final List<String> LOCKFILE_HEADER_LIST = ImmutableList.of((Object)"# This is a Gradle generated file for dependency locking.", (Object)"# Manual edits can break the build and are not advised.", (Object)"# This file is expected to be part of source control.");
    static final String EMPTY_CONFIGURATIONS_ENTRY = "empty=";
    static final String BUILD_SCRIPT_PREFIX = "buildscript-";
    static final String SETTINGS_SCRIPT_PREFIX = "settings-";
    private final Path lockFilesRoot;
    private final DomainObjectContext context;
    private final RegularFileProperty lockFile;
    private final FileResourceListener listener;

    public LockFileReaderWriter(FileResolver fileResolver, DomainObjectContext context, RegularFileProperty lockFile, FileResourceListener listener) {
        this.context = context;
        this.lockFile = lockFile;
        this.listener = listener;
        Path resolve = null;
        if (fileResolver.canResolveRelativePath()) {
            resolve = fileResolver.resolve((Object)DEPENDENCY_LOCKING_FOLDER).toPath();
            lockFile.set(fileResolver.resolve((Object)this.decorate(UNIQUE_LOCKFILE_NAME)));
        }
        this.lockFilesRoot = resolve;
        LOGGER.debug("Lockfiles root: {}", (Object)this.lockFilesRoot);
    }

    @Nullable
    public List<String> readLockFile(String configurationName) {
        this.checkValidRoot(configurationName);
        Path lockFile = this.lockFilesRoot.resolve(this.decorate(configurationName) + FILE_SUFFIX);
        this.listener.fileObserved(lockFile.toFile());
        if (Files.exists(lockFile, new LinkOption[0])) {
            List<String> result;
            try {
                result = Files.readAllLines(lockFile, CHARSET);
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to load lock file", e);
            }
            List<String> lines = result;
            this.filterNonModuleLines(lines);
            return lines;
        }
        return null;
    }

    private String decorate(String configurationName) {
        if (this.context.isScript()) {
            if (this.context.isRootScript()) {
                return SETTINGS_SCRIPT_PREFIX + configurationName;
            }
            return BUILD_SCRIPT_PREFIX + configurationName;
        }
        return configurationName;
    }

    private void checkValidRoot() {
        if (this.lockFilesRoot == null) {
            throw new IllegalStateException("Dependency locking cannot be used for project '" + this.context.getProjectPath() + "'. See limitations in the documentation (" + DOC_REG.getDocumentationFor("dependency_locking", "locking_limitations") + ").");
        }
    }

    private void checkValidRoot(String configurationName) {
        if (this.lockFilesRoot == null) {
            throw new IllegalStateException("Dependency locking cannot be used for configuration '" + this.context.identityPath(configurationName) + "'. See limitations in the documentation (" + DOC_REG.getDocumentationFor("dependency_locking", "locking_limitations") + ").");
        }
    }

    private void filterNonModuleLines(List<String> lines) {
        Iterator<String> iterator = lines.iterator();
        while (iterator.hasNext()) {
            String value = iterator.next().trim();
            if (!value.startsWith("#") && !value.isEmpty()) continue;
            iterator.remove();
        }
    }

    public Map<String, List<String>> readUniqueLockFile() {
        this.checkValidRoot();
        Predicate<String> empty = String::isEmpty;
        Predicate<String> comment = s -> s.startsWith("#");
        Path uniqueLockFile = this.getUniqueLockfilePath();
        ArrayList emptyConfigurations = new ArrayList();
        HashMap<String, List<String>> uniqueLockState = new HashMap<String, List<String>>(10);
        this.listener.fileObserved(uniqueLockFile.toFile());
        if (Files.exists(uniqueLockFile, new LinkOption[0])) {
            try {
                Files.lines(uniqueLockFile, CHARSET).filter(empty.or(comment).negate()).filter(line -> {
                    if (line.startsWith(EMPTY_CONFIGURATIONS_ENTRY)) {
                        this.collectEmptyConfigurations((String)line, emptyConfigurations);
                        return false;
                    }
                    return true;
                }).forEach(line -> this.parseLine((String)line, (Map<String, List<String>>)uniqueLockState));
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to load unique lockfile", e);
            }
            for (String emptyConfiguration : emptyConfigurations) {
                uniqueLockState.computeIfAbsent(emptyConfiguration, k -> new ArrayList());
            }
            return uniqueLockState;
        }
        return new HashMap<String, List<String>>();
    }

    private void collectEmptyConfigurations(String line, List<String> emptyConfigurations) {
        if (line.length() > EMPTY_CONFIGURATIONS_ENTRY.length()) {
            String[] configurations = line.substring(EMPTY_CONFIGURATIONS_ENTRY.length()).split(",");
            Collections.addAll(emptyConfigurations, configurations);
        }
    }

    private Path getUniqueLockfilePath() {
        return ((RegularFile)this.lockFile.get()).getAsFile().toPath();
    }

    private void parseLine(String line, Map<String, List<String>> result) {
        String[] configurations;
        String[] split = line.split("=");
        for (String configuration : configurations = split[1].split(",")) {
            result.compute(configuration, (k, v) -> {
                ArrayList<String> mapping = v == null ? new ArrayList<String>() : v;
                mapping.add(split[0]);
                return mapping;
            });
        }
    }

    public boolean canWrite() {
        return this.lockFilesRoot != null;
    }

    public void writeUniqueLockfile(Map<String, List<String>> lockState) {
        this.checkValidRoot();
        Path lockfilePath = this.getUniqueLockfilePath();
        if (lockState.isEmpty()) {
            GFileUtils.deleteQuietly((File)lockfilePath.toFile());
            return;
        }
        TreeMap<String, List<String>> dependencyToConfigurations = new TreeMap<String, List<String>>();
        ArrayList<String> emptyConfigurations = new ArrayList<String>();
        this.mapLockStateFromDependencyToConfiguration(lockState, dependencyToConfigurations, emptyConfigurations);
        this.writeUniqueLockfile(lockfilePath, dependencyToConfigurations, emptyConfigurations);
        this.cleanupLegacyLockFiles(lockState.keySet());
    }

    private void cleanupLegacyLockFiles(Set<String> lockedConfigurations) {
        lockedConfigurations.stream().map(f -> this.lockFilesRoot.resolve(this.decorate((String)f) + FILE_SUFFIX)).map(Path::toFile).forEach(GFileUtils::deleteQuietly);
    }

    private void writeUniqueLockfile(Path lockfilePath, Map<String, List<String>> dependencyToConfigurations, List<String> emptyConfigurations) {
        try {
            Files.createDirectories(lockfilePath.getParent(), new FileAttribute[0]);
            ArrayList<String> content = new ArrayList<String>(50);
            content.addAll(LOCKFILE_HEADER_LIST);
            for (Map.Entry<String, List<String>> entry : dependencyToConfigurations.entrySet()) {
                String builder = entry.getKey() + "=" + entry.getValue().stream().sorted().collect(Collectors.joining(","));
                content.add(builder);
            }
            content.add(EMPTY_CONFIGURATIONS_ENTRY + emptyConfigurations.stream().sorted().collect(Collectors.joining(",")));
            Files.write(lockfilePath, content, CHARSET, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to write unique lockfile", e);
        }
    }

    private void mapLockStateFromDependencyToConfiguration(Map<String, List<String>> lockState, Map<String, List<String>> dependencyToConfigurations, List<String> emptyConfigurations) {
        for (Map.Entry<String, List<String>> entry : lockState.entrySet()) {
            List<String> dependencies = entry.getValue();
            if (dependencies.isEmpty()) {
                emptyConfigurations.add(entry.getKey());
                continue;
            }
            for (String dependency : dependencies) {
                dependencyToConfigurations.compute(dependency, (k, v) -> {
                    ArrayList<String> confs = v;
                    if (v == null) {
                        confs = new ArrayList<String>();
                    }
                    confs.add((String)entry.getKey());
                    return confs;
                });
            }
        }
    }
}

