/*
 * Decompiled with CFR 0.152.
 */
package ghidra.formats.gfilesystem;

import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileImpl;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.util.Msg;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

public class FileSystemIndexHelper<METADATATYPE> {
    private GFile rootDir;
    protected Map<GFile, FileData<METADATATYPE>> fileToEntryMap = new HashMap<GFile, FileData<METADATATYPE>>();
    protected Map<Long, FileData<METADATATYPE>> fileIndexToEntryMap = new HashMap<Long, FileData<METADATATYPE>>();
    protected Map<GFile, Map<String, FileData<METADATATYPE>>> directoryToListing = new HashMap<GFile, Map<String, FileData<METADATATYPE>>>();

    public FileSystemIndexHelper(GFileSystem fs, FSRLRoot fsFSRL) {
        this.rootDir = GFileImpl.fromFSRL(fs, null, fsFSRL.withPath("/"), true, -1L);
        this.initRootDir(null);
    }

    private void initRootDir(METADATATYPE metadata) {
        FileData fileData = new FileData();
        fileData.file = this.rootDir;
        fileData.fileIndex = -1L;
        fileData.metaData = metadata;
        this.fileToEntryMap.put(this.rootDir, fileData);
        this.directoryToListing.put(this.rootDir, new HashMap());
    }

    public GFile getRootDir() {
        return this.rootDir;
    }

    public synchronized void clear() {
        this.fileToEntryMap.clear();
        this.directoryToListing.clear();
    }

    public synchronized int getFileCount() {
        return this.fileToEntryMap.size();
    }

    public synchronized METADATATYPE getMetadata(GFile f) {
        FileData<METADATATYPE> fileData = this.fileToEntryMap.get(f);
        return fileData != null ? (METADATATYPE)fileData.metaData : null;
    }

    public synchronized void setMetadata(GFile f, METADATATYPE metaData) throws IOException {
        FileData<METADATATYPE> fileData = this.fileToEntryMap.get(f);
        if (fileData == null) {
            throw new IOException("Unknown file: " + f);
        }
        fileData.metaData = metaData;
    }

    public synchronized GFile getFileByIndex(long fileIndex) {
        FileData<METADATATYPE> fileData = this.fileIndexToEntryMap.get(fileIndex);
        return fileData != null ? fileData.file : null;
    }

    public synchronized List<GFile> getListing(GFile directory) {
        Map<String, FileData<METADATATYPE>> dirListing = this.getDirectoryContents(directory, false);
        if (dirListing == null) {
            return List.of();
        }
        return dirListing.values().stream().map(fd -> fd.file).collect(Collectors.toList());
    }

    public synchronized GFile lookup(String path) {
        return this.lookup(null, path, null);
    }

    public synchronized GFile lookup(GFile baseDir, String path, Comparator<String> nameComp) {
        String name;
        String[] nameparts = (path != null ? path : "").split("/");
        GFile parent = this.lookupParent(baseDir, nameparts, false, nameComp);
        if (parent == null) {
            return null;
        }
        String string = name = nameparts.length > 0 ? nameparts[nameparts.length - 1] : null;
        if (name == null || name.isEmpty()) {
            return parent;
        }
        FileData<METADATATYPE> fileData = this.lookupFileInDir(this.getDirectoryContents(parent, false), name, nameComp);
        return fileData != null ? fileData.file : null;
    }

    public synchronized GFile storeFile(String path, long fileIndex, boolean isDirectory, long length, METADATATYPE metadata) {
        String[] nameparts = path.replaceAll("[\\\\]", "/").split("/");
        GFile parent = this.lookupParent(this.rootDir, nameparts, true, null);
        String lastpart = nameparts[nameparts.length - 1];
        FileData<METADATATYPE> fileData = this.doStoreFile(lastpart, parent, fileIndex, isDirectory, length, metadata);
        return fileData.file;
    }

    public synchronized GFile storeFileWithParent(String filename, GFile parent, long fileIndex, boolean isDirectory, long length, METADATATYPE metadata) {
        FileData<METADATATYPE> fileData = this.doStoreFile(filename, parent, fileIndex, isDirectory, length, metadata);
        return fileData.file;
    }

    private FileData<METADATATYPE> doStoreMissingDir(String filename, GFile parent) {
        parent = parent == null ? this.rootDir : parent;
        Map<String, FileData<METADATATYPE>> dirContents = this.getDirectoryContents(parent, true);
        GFileImpl file = this.createNewFile(parent, filename, true, -1L, null);
        FileData fileData = new FileData();
        fileData.file = file;
        fileData.fileIndex = -1L;
        this.fileToEntryMap.put(file, fileData);
        dirContents.put(filename, fileData);
        this.getDirectoryContents(file, true);
        return fileData;
    }

    private FileData<METADATATYPE> doStoreFile(String filename, GFile parent, long fileIndex, boolean isDirectory, long length, METADATATYPE metadata) {
        Map<String, FileData<METADATATYPE>> dirContents;
        long fileNum;
        parent = parent == null ? this.rootDir : parent;
        long l = fileNum = fileIndex != -1L ? fileIndex : (long)this.fileToEntryMap.size();
        if (this.fileIndexToEntryMap.containsKey(fileNum)) {
            Msg.warn((Object)this, (Object)("Duplicate fileNum for file " + parent.getPath() + "/" + filename));
        }
        String uniqueName = this.makeUniqueFilename((dirContents = this.getDirectoryContents(parent, true)).containsKey(filename) && !isDirectory, filename, fileNum);
        GFileImpl file = this.createNewFile(parent, uniqueName, isDirectory, length, metadata);
        FileData fileData = new FileData();
        fileData.file = file;
        fileData.fileIndex = fileNum;
        fileData.metaData = metadata;
        this.fileToEntryMap.put(file, fileData);
        this.fileIndexToEntryMap.put(fileNum, fileData);
        dirContents.put(uniqueName, fileData);
        if (isDirectory) {
            this.getDirectoryContents(file, true);
        }
        return fileData;
    }

    private String makeUniqueFilename(boolean wasNameCollision, String filename, long fileIndex) {
        return wasNameCollision ? filename + "[" + Long.toString(fileIndex) + "]" : filename;
    }

    private Map<String, FileData<METADATATYPE>> getDirectoryContents(GFile directoryFile, boolean createIfMissing) {
        Map<String, FileData<METADATATYPE>> dirContents = this.directoryToListing.get(directoryFile = directoryFile != null ? directoryFile : this.rootDir);
        if (dirContents == null && createIfMissing) {
            dirContents = new HashMap<String, FileData<METADATATYPE>>();
            this.directoryToListing.put(directoryFile, dirContents);
        }
        return dirContents;
    }

    protected GFile lookupParent(GFile baseDir, String[] nameparts, boolean createIfMissing, Comparator<String> nameComp) {
        GFile currentDir = Objects.requireNonNullElse(baseDir, this.rootDir);
        for (int i = 0; i < nameparts.length - 1; ++i) {
            Map<String, FileData<METADATATYPE>> currentDirContents = this.getDirectoryContents(currentDir, true);
            String name = nameparts[i];
            if (name.isEmpty()) continue;
            FileData<METADATATYPE> fileData = this.lookupFileInDir(currentDirContents, name, nameComp);
            if (fileData == null) {
                if (!createIfMissing) {
                    return null;
                }
                fileData = this.doStoreMissingDir(name, currentDir);
            }
            currentDir = fileData.file;
        }
        return currentDir;
    }

    protected FileData<METADATATYPE> lookupFileInDir(Map<String, FileData<METADATATYPE>> dirContents, String filename, Comparator<String> nameComp) {
        if (dirContents == null) {
            return null;
        }
        if (nameComp == null) {
            return dirContents.get(filename);
        }
        ArrayList<FileData<METADATATYPE>> candidateFiles = new ArrayList<FileData<METADATATYPE>>();
        for (FileData<METADATATYPE> fd : dirContents.values()) {
            if (nameComp.compare(filename, fd.file.getName()) != 0) continue;
            if (fd.file.getName().equals(filename)) {
                return fd;
            }
            candidateFiles.add(fd);
        }
        Collections.sort(candidateFiles, (f1, f2) -> f1.file.getName().compareTo(f2.file.getName()));
        return !candidateFiles.isEmpty() ? (FileData)candidateFiles.get(0) : null;
    }

    protected GFileImpl createNewFile(GFile parentFile, String name, boolean isDirectory, long size, METADATATYPE metadata) {
        FSRL newFileFSRL = parentFile.getFSRL().appendPath(name);
        return GFileImpl.fromFSRL(this.rootDir.getFilesystem(), parentFile, newFileFSRL, isDirectory, size);
    }

    public synchronized void updateFSRL(GFile file, FSRL newFSRL) {
        Map<String, FileData<METADATATYPE>> dirListing;
        GFileImpl newFile = GFileImpl.fromFSRL(this.rootDir.getFilesystem(), file.getParentFile(), newFSRL, file.isDirectory(), file.getLength());
        FileData<METADATATYPE> fileData = this.fileToEntryMap.get(file);
        if (fileData != null) {
            this.fileToEntryMap.remove(file);
            this.fileIndexToEntryMap.remove(fileData.fileIndex);
            fileData.file = newFile;
            this.fileToEntryMap.put(newFile, fileData);
            if (fileData.fileIndex != -1L) {
                this.fileIndexToEntryMap.put(fileData.fileIndex, fileData);
            }
        }
        if ((dirListing = this.directoryToListing.get(file)) != null) {
            this.directoryToListing.remove(file);
            this.directoryToListing.put(newFile, dirListing);
        }
    }

    public String toString() {
        return "FileSystemIndexHelper for " + this.rootDir.getFilesystem();
    }

    static class FileData<METADATATYPE> {
        GFile file;
        METADATATYPE metaData;
        long fileIndex;

        FileData() {
        }
    }
}

