/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.file.formats.android.dex.format;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.StructConverter;
import ghidra.program.model.data.*;
import ghidra.util.exception.DuplicateNameException;

import java.io.IOException;

public class FilledArrayDataPayload implements StructConverter {

	public final static short MAGIC = 0x0300;

	private short ident;
	private short elementWidth;
	private int size;
	private byte[] data;

	public FilledArrayDataPayload(BinaryReader reader) throws IOException {
		ident = reader.readNextShort();
		elementWidth = reader.readNextShort();
		size = reader.readNextInt();
		data = reader.readNextByteArray(size * elementWidth);
	}

	public short getIdent() {
		return ident;
	}

	public short getElementWidth() {
		return elementWidth;
	}

	public int getSize() {
		return size;
	}

	public byte[] getData() {
		return data;
	}

	@Override
	public DataType toDataType() throws DuplicateNameException, IOException {
		Structure structure =
			new StructureDataType("filled_array_data_payload_" + elementWidth + "_" + size, 0);
		structure.add(WORD, "ident", null);
		structure.add(WORD, "element_width", null);
		structure.add(DWORD, "size", null);
		structure.add(new ArrayDataType(BYTE, size * elementWidth, BYTE.getLength()), "data", null);
		structure.setCategoryPath(new CategoryPath("/dex/filled_array_data_payload"));
		return structure;
	}

}
