/* ###
 * IP: GHIDRA
 * REVIEWED: YES
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.program.util;

import ghidra.framework.model.DomainObjectChangeRecord;
import ghidra.program.model.address.Address;

/**
 * Event data for a DomainObjectChangeEvent generated by a Program.
 * <p>See <a href="doc-files/events.html">Program Events</a> for more
 * information on event data.
 *</p>
 */
public class ProgramChangeRecord extends DomainObjectChangeRecord {
	private final static long serialVersionUID = 1;

	private Address start;
	private Address end;
	private Object affectedObj; // may be null

	/**
	 * Construct a new ProgramChangeRecord; any of the Address or
	 * Object params may be null, depending on what the type param is.
	 * @param type event type
	 * @param start starting address that is affected by the event
	 * @param end ending address that is affected by the event
	 * @param oldValue original value
	 * @param newValue new value
	 */
	public ProgramChangeRecord(int type, Address start, Address end, Object affectedObj,
			Object oldValue, Object newValue) {
		super(type, oldValue, newValue);
		this.start = start;
		this.end = end;
		this.affectedObj = affectedObj;
	}

	/**
	 * Construct a new ProgramChangeRecord; any of the Address or
	 * Object params may be null, depending on what the type param is.
	 * @param type event type
	 * @param subType event sub-type
	 * @param start starting address that is affected by the event
	 * @param end ending address that is affected by the event
	 * @param oldValue original value
	 * @param newValue new value
	 */
	public ProgramChangeRecord(int type, int subType, Address start, Address end,
			Object affectedObj, Object oldValue, Object newValue) {
		super(type, subType, oldValue, newValue);
		this.start = start;
		this.end = end;
		this.affectedObj = affectedObj;
	}

	/**
	 * Get the start address.
	 */
	public Address getStart() {
		return start;
	}

	/**
	 * Get the end address.
	 */
	public Address getEnd() {
		return end;
	}

	/**
	 * Return the object that is the subject of this change record.
	 *
	 * @return Object null if this change record does not have the
	 * affected object
	 */
	public Object getObject() {
		return affectedObj;
	}
}
