/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util;

import ghidra.util.SignednessFormatMode;
import ghidra.util.StringUtilities;
import java.math.BigInteger;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.IteratorUtils;
import org.apache.commons.collections4.ResettableIterator;
import org.apache.commons.lang3.StringUtils;
import util.CollectionUtils;

public final class NumericUtilities {
    public static final BigInteger MAX_UNSIGNED_LONG = new BigInteger("ffffffffffffffff", 16);
    public static final BigInteger MAX_SIGNED_LONG = new BigInteger("7fffffffffffffff", 16);
    public static final long MAX_UNSIGNED_INT32_AS_LONG = 0xFFFFFFFFL;
    private static final String HEX_PREFIX_X = "0X";
    private static final String HEX_PREFIX_x = "0x";
    private static final Set<Class<? extends Number>> INTEGER_TYPES = new HashSet<Class<? extends Number>>();
    private static final Set<Class<? extends Number>> FLOATINGPOINT_TYPES;
    private static IntegerRadixRenderer SIGNED_INTEGER_RENDERER;
    private static IntegerRadixRenderer UNSIGNED_INTEGER_RENDERER;
    private static IntegerRadixRenderer DEFAULT_INTEGER_RENDERER;

    private NumericUtilities() {
    }

    public static long parseNumber(String numStr) {
        return NumericUtilities.parseNumber(numStr, 0L);
    }

    public static Long parseNumber(String numStr, Long defaultValue) {
        String string = numStr = numStr == null ? "" : numStr.trim();
        if (numStr.length() == 0) {
            return defaultValue;
        }
        long value = 0L;
        try {
            value = numStr.startsWith(HEX_PREFIX_x) || numStr.startsWith(HEX_PREFIX_X) ? (long)Integer.parseInt(numStr.substring(2), 16) : (long)Integer.parseInt(numStr);
        }
        catch (NumberFormatException exc) {
            return defaultValue;
        }
        return value;
    }

    public static long parseLong(String numStr) {
        String origStr = numStr;
        long value = 0L;
        long sign = 1L;
        String string = numStr = numStr == null ? "" : numStr.trim();
        if (numStr.length() == 0) {
            return value;
        }
        if (numStr.startsWith("-")) {
            sign = -1L;
            numStr = numStr.substring(1);
        }
        int radix = 10;
        if (numStr.startsWith(HEX_PREFIX_x) || numStr.startsWith(HEX_PREFIX_X)) {
            if (numStr.length() > 18) {
                throw new NumberFormatException(numStr + " has too many digits.");
            }
            numStr = numStr.substring(2);
            radix = 16;
        }
        if (numStr.length() == 0) {
            return 0L;
        }
        try {
            BigInteger bi = new BigInteger(numStr, radix);
            return bi.longValue() * sign;
        }
        catch (NumberFormatException e) {
            NumberFormatException e2 = new NumberFormatException("Cannot parse long from " + origStr);
            e2.setStackTrace(e.getStackTrace());
            throw e2;
        }
        catch (ArithmeticException e) {
            throw new NumberFormatException(origStr + " is too big.");
        }
    }

    public static long parseOctLong(String numStr) {
        long value = 0L;
        long sign = 1L;
        String string = numStr = numStr == null ? "" : numStr.trim();
        if (numStr.length() == 0) {
            return value;
        }
        if (numStr.startsWith("-")) {
            sign = -1L;
            numStr = numStr.substring(1);
        }
        int radix = 8;
        if (numStr.startsWith("0")) {
            if (numStr.length() > 18) {
                throw new NumberFormatException(numStr + " has too many digits.");
            }
            numStr = numStr.substring(1);
        }
        BigInteger bi = new BigInteger(numStr, radix);
        return bi.longValue() * sign;
    }

    public static long parseHexLong(String numStr) {
        return NumericUtilities.parseHexBigInteger(numStr).longValue();
    }

    public static BigInteger parseHexBigInteger(String numStr) {
        Object object = numStr = numStr == null ? "" : ((String)numStr).trim();
        if (((String)numStr).length() == 0) {
            throw new NumberFormatException((String)numStr + " no digits.");
        }
        boolean negative = false;
        if (((String)numStr).startsWith("-")) {
            negative = true;
            numStr = ((String)numStr).substring(1);
        }
        if (((String)numStr).startsWith(HEX_PREFIX_x) || ((String)numStr).startsWith(HEX_PREFIX_X)) {
            numStr = ((String)numStr).substring(2);
        }
        if (negative) {
            numStr = "-" + (String)numStr;
        }
        return new BigInteger((String)numStr, 16);
    }

    public static final String toHexString(long value) {
        return HEX_PREFIX_x + Long.toHexString(value);
    }

    public static final String toHexString(long value, int size) {
        if (size > 0 && size < 8) {
            value &= -1L >>> 8 * (8 - size);
        }
        return HEX_PREFIX_x + Long.toHexString(value);
    }

    public static final String toSignedHexString(long value) {
        StringBuffer buf = new StringBuffer();
        if (value < 0L) {
            buf.append("-");
        }
        buf.append(HEX_PREFIX_x);
        buf.append(Long.toHexString(Math.abs(value)));
        return buf.toString();
    }

    public static BigInteger unsignedLongToBigInteger(long value) {
        if (value >= 0L) {
            return BigInteger.valueOf(value);
        }
        return MAX_UNSIGNED_LONG.add(BigInteger.valueOf(value + 1L));
    }

    public static long bigIntegerToUnsignedLong(BigInteger value) {
        if (value.compareTo(MAX_SIGNED_LONG) <= 0) {
            return value.longValue();
        }
        return value.subtract(MAX_UNSIGNED_LONG).longValue() - 1L;
    }

    public static double unsignedLongToDouble(long val) {
        double dVal = val & Long.MAX_VALUE;
        if (val < 0L) {
            dVal += 9.223372036854776E18;
        }
        return dVal;
    }

    public static long getUnsignedAlignedValue(long unsignedValue, long alignment) {
        boolean negative;
        if (alignment == 0L || unsignedValue % alignment == 0L) {
            return unsignedValue;
        }
        boolean bl = negative = unsignedValue < 0L;
        if (negative) {
            unsignedValue = -(unsignedValue + alignment);
        }
        long alignedValue = (unsignedValue + alignment - 1L) / alignment * alignment;
        if (negative) {
            alignedValue = -alignedValue;
        }
        return alignedValue;
    }

    public static String convertMaskedValueToHexString(long msk, long val, int n, boolean truncate, int spaceevery, String spacer) {
        StringBuilder sb = new StringBuilder(n * 4);
        int shamt = n * 4 - 4;
        long LEFT_NIBBLE = 15L << shamt;
        long LEFT_BIT = 8L << shamt;
        for (int i = 0; i < n; ++i) {
            long m;
            if (!truncate && spaceevery != 0 && i != 0 && i % spaceevery == 0) {
                sb.append(spacer);
            }
            if ((m = msk & LEFT_NIBBLE) == 0L) {
                msk <<= 4;
                val <<= 4;
                if (truncate) continue;
                sb.append('X');
            } else if (m == LEFT_NIBBLE) {
                sb.append(String.format("%1X", (val & LEFT_NIBBLE) >>> shamt));
                msk <<= 4;
                val <<= 4;
            } else {
                sb.append('[');
                for (int j = 0; j < 4; ++j) {
                    if ((msk & LEFT_BIT) == 0L) {
                        sb.append('x');
                    } else if ((val & LEFT_BIT) == 0L) {
                        sb.append('0');
                    } else {
                        sb.append('1');
                    }
                    msk <<= 1;
                    val <<= 1;
                }
                sb.append(']');
            }
            truncate = false;
        }
        if (sb.length() == 0 && n != 0) {
            return "X";
        }
        return sb.toString();
    }

    public static String convertMaskToHexString(long msk, int n, boolean truncate, int spaceevery, String spacer) {
        StringBuilder sb = new StringBuilder(n * 4);
        int shamt = n * 4 - 4;
        long LEFT_NIBBLE = 15L << shamt;
        long LEFT_BIT = 8L << shamt;
        for (int i = 0; i < n; ++i) {
            long m;
            if (!truncate && spaceevery != 0 && i != 0 && i % spaceevery == 0) {
                sb.append(spacer);
            }
            if ((m = msk & LEFT_NIBBLE) == 0L) {
                if (truncate) continue;
                sb.append('0');
                msk <<= 4;
            } else if (m == LEFT_NIBBLE) {
                sb.append('F');
                msk <<= 4;
            } else {
                sb.append('[');
                for (int j = 0; j < 4; ++j) {
                    if ((msk & LEFT_BIT) == 0L) {
                        sb.append('0');
                    } else {
                        sb.append('1');
                    }
                    msk <<= 1;
                }
                sb.append(']');
            }
            truncate = false;
        }
        if (sb.length() == 0 && n != 0) {
            return "X";
        }
        return sb.toString();
    }

    public static void convertHexStringToMaskedValue(AtomicLong msk, AtomicLong val, String hex, int n, int spaceevery, String spacer) {
        long lmsk = 0L;
        long lval = 0L;
        int pos = 0;
        int i = 0;
        while (pos < hex.length() && i < n) {
            lmsk <<= 4;
            lval <<= 4;
            if (i != 0 && spaceevery != 0 && i % spaceevery == 0) {
                boolean matchspacer = hex.regionMatches(pos, spacer, 0, spacer.length());
                if (!matchspacer) {
                    throw new NumberFormatException("Missing spacer at " + pos);
                }
                pos += spacer.length();
                continue;
            }
            char c = hex.charAt(pos);
            if (Character.isLetterOrDigit(c)) {
                if (c == 'X') {
                    ++i;
                    ++pos;
                    continue;
                }
                long nibble = Long.parseLong(hex.substring(pos, pos + 1), 16);
                lmsk |= 0xFL;
                lval |= nibble;
                ++i;
                ++pos;
                continue;
            }
            if (c == '[') {
                lmsk >>= 4;
                lval >>= 4;
                for (int j = 0; j < 4; ++j) {
                    lmsk <<= 1;
                    lval <<= 1;
                    if (++pos > hex.length()) {
                        throw new NumberFormatException("Missing one or more bits at " + pos);
                    }
                    c = hex.charAt(pos);
                    if (c == 'x') continue;
                    lmsk |= 1L;
                    if (c == '0') continue;
                    if (c == '1') {
                        lval |= 1L;
                        continue;
                    }
                    throw new NumberFormatException("Illegal character '" + c + "' at " + pos);
                }
                if (++pos > hex.length() || hex.charAt(pos) != ']') {
                    throw new NumberFormatException("Missing closing bracket after bits at " + pos);
                }
                ++pos;
                ++i;
                continue;
            }
            throw new NumberFormatException("Illegal character '" + c + "' at " + pos);
        }
        if (pos < hex.length()) {
            throw new NumberFormatException("Gratuitous characters starting at " + pos);
        }
        if (i < n) {
            throw new NumberFormatException("Too few characters for " + n + " nibbles. Got " + i);
        }
        msk.set(lmsk);
        val.set(lval);
    }

    public static String formatNumber(long number, int radix) {
        return NumericUtilities.formatNumber(number, radix, SignednessFormatMode.DEFAULT);
    }

    public static String formatNumber(long number, int radix, SignednessFormatMode mode) {
        if (radix == 0) {
            byte[] bytes = new byte[]{(byte)(number >> 56), (byte)(number >> 48), (byte)(number >> 40), (byte)(number >> 32), (byte)(number >> 24), (byte)(number >> 16), (byte)(number >> 8), (byte)(number >> 0)};
            return StringUtilities.toQuotedString(bytes);
        }
        IntegerRadixRenderer renderer = null;
        switch (mode) {
            case SIGNED: {
                renderer = SIGNED_INTEGER_RENDERER;
                break;
            }
            case UNSIGNED: {
                renderer = UNSIGNED_INTEGER_RENDERER;
                break;
            }
            case DEFAULT: {
                renderer = DEFAULT_INTEGER_RENDERER;
            }
        }
        return renderer.toString(number, radix);
    }

    public static byte[] convertStringToBytes(String hexString) {
        CharSequence characters = NumericUtilities.condenseByteString(hexString);
        return NumericUtilities.convertCharSequenceToBytes(characters);
    }

    /*
     * WARNING - void declaration
     */
    private static CharSequence condenseByteString(String hexString) {
        if (!StringUtils.containsAny((CharSequence)hexString, (char[])new char[]{' ', ','})) {
            return hexString;
        }
        String[] parts = hexString.split("[ ,]");
        StringBuilder buffy = new StringBuilder();
        for (String string : parts) {
            void var6_6;
            if (string.length() == 1) {
                String string2 = "0" + string;
            }
            buffy.append((String)var6_6);
        }
        return buffy;
    }

    private static byte[] convertCharSequenceToBytes(CharSequence characters) {
        if (characters.length() % 2 == 1) {
            throw new IllegalArgumentException("Hex string must be an even number of characters");
        }
        try {
            byte[] bytes = new byte[characters.length() / 2];
            for (int i = 0; i < characters.length(); i += 2) {
                CharSequence bs = characters.subSequence(i, i + 2);
                bytes[i / 2] = (byte)Integer.parseInt(bs.toString(), 16);
            }
            return bytes;
        }
        catch (Exception exception) {
            return null;
        }
    }

    public static String toString(byte b) {
        String bs = Integer.toHexString(b & 0xFF);
        if (bs.length() == 1) {
            return "0" + bs;
        }
        return bs;
    }

    public static String convertBytesToString(byte[] bytes) {
        return NumericUtilities.convertBytesToString(bytes, null);
    }

    public static String convertBytesToString(byte[] bytes, String delimeter) {
        if (bytes == null) {
            return null;
        }
        return NumericUtilities.convertBytesToString(bytes, 0, bytes.length, delimeter);
    }

    public static String convertBytesToString(byte[] bytes, int start, int len, String delimeter) {
        ResettableIterator iterator = IteratorUtils.arrayIterator((Object)bytes, (int)start, (int)(start + len));
        return NumericUtilities.convertBytesToString((Iterator<Byte>)iterator, delimeter);
    }

    public static String convertBytesToString(Iterator<Byte> bytes, String delimiter) {
        return NumericUtilities.convertBytesToString(() -> bytes, delimiter);
    }

    public static String convertBytesToString(Iterable<Byte> bytes, String delimiter) {
        return NumericUtilities.convertBytesToString(CollectionUtils.asStream(bytes), delimiter);
    }

    public static String convertBytesToString(Stream<Byte> bytes, String delimiter) {
        delimiter = delimiter == null ? "" : delimiter;
        return bytes.map(NumericUtilities::toString).collect(Collectors.joining(delimiter));
    }

    public static boolean isIntegerType(Number number) {
        Class<?> numClass = number.getClass();
        return NumericUtilities.isIntegerType(numClass);
    }

    public static boolean isIntegerType(Class<?> numClass) {
        return INTEGER_TYPES.contains(numClass);
    }

    public static boolean isFloatingPointType(Number number) {
        Class<?> numClass = number.getClass();
        return NumericUtilities.isFloatingPointType(numClass);
    }

    public static boolean isFloatingPointType(Class<?> numClass) {
        return FLOATINGPOINT_TYPES.contains(numClass);
    }

    static {
        INTEGER_TYPES.add(Byte.class);
        INTEGER_TYPES.add(Short.class);
        INTEGER_TYPES.add(Integer.class);
        INTEGER_TYPES.add(Long.class);
        FLOATINGPOINT_TYPES = new HashSet<Class<? extends Number>>();
        FLOATINGPOINT_TYPES.add(Float.class);
        FLOATINGPOINT_TYPES.add(Double.class);
        SIGNED_INTEGER_RENDERER = new SignedIntegerRadixRenderer();
        UNSIGNED_INTEGER_RENDERER = new UnsignedIntegerRadixRenderer();
        DEFAULT_INTEGER_RENDERER = new DefaultIntegerRadixRenderer();
    }

    private static interface IntegerRadixRenderer {
        public String toString(long var1, int var3);
    }

    private static final class SignedIntegerRadixRenderer
    implements IntegerRadixRenderer {
        private SignedIntegerRadixRenderer() {
        }

        @Override
        public String toString(long number, int radix) {
            switch (radix) {
                case 2: {
                    return Long.toString(number, radix) + "b";
                }
                case 8: {
                    return Long.toString(number, radix) + "o";
                }
                case 10: {
                    return Long.toString(number, radix);
                }
                case 16: {
                    return Long.toString(number, radix) + "h";
                }
            }
            throw new IllegalArgumentException("Unsupported radix");
        }
    }

    private static final class UnsignedIntegerRadixRenderer
    implements IntegerRadixRenderer {
        private UnsignedIntegerRadixRenderer() {
        }

        @Override
        public String toString(long number, int radix) {
            switch (radix) {
                case 2: {
                    return Long.toBinaryString(number) + "b";
                }
                case 8: {
                    return Long.toOctalString(number) + "o";
                }
                case 10: {
                    return Long.toUnsignedString(number);
                }
                case 16: {
                    return Long.toHexString(number) + "h";
                }
            }
            throw new IllegalArgumentException("Unsupported radix");
        }
    }

    private static final class DefaultIntegerRadixRenderer
    implements IntegerRadixRenderer {
        private DefaultIntegerRadixRenderer() {
        }

        @Override
        public String toString(long number, int radix) {
            switch (radix) {
                case 2: {
                    return new UnsignedIntegerRadixRenderer().toString(number, radix);
                }
                case 8: {
                    return new UnsignedIntegerRadixRenderer().toString(number, radix);
                }
                case 10: {
                    return new SignedIntegerRadixRenderer().toString(number, radix);
                }
                case 16: {
                    return new UnsignedIntegerRadixRenderer().toString(number, radix);
                }
            }
            throw new IllegalArgumentException("Unsupported radix");
        }
    }
}

