(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*)

open Range
open Ast
open List
open Tast
open Tastops
open Typrelns
open Env
open Lib
open Ildiag

exception MatchIncomplete of bool * string option * Range.range
exception RuleNeverMatched of Range.range

type actionOnFailure = 
  | IncompleteWarnOnLast 
  | Incomplete 
  | Throw 
  | Rethrow 
  | FailFilter

type pat =
  | TPat_const of tconst * Range.range
  | TPat_wild of Range.range  (* note = TPat_disjs([],m), but we haven't yet removed that duplication *)
  | TPat_as of  pat * pbind * Range.range (* note: can be replaced by TPat_var, i.e. equals TPat_conjs([TPat_var; pat]) *)
  | TPat_disjs of  pat list * Range.range
  | TPat_conjs of  pat list * Range.range
  | TPat_query of (expr * typ list * val_ref option * int * apinfo) * pat * Range.range
  | TPat_unionconstr of unionconstr_ref * tinst * pat list * Range.range
  | TPat_exnconstr of tycon_ref * pat list * Range.range
  | TPat_tuple of  pat list * typ list * Range.range
  | TPat_array of  pat list * typ * Range.range
  | TPat_recd of tycon_ref * tinst * (tinst * pat) list * Range.range
  | TPat_range of Nums.unichar * Nums.unichar * Range.range
  | TPat_null of Range.range
  | TPat_isinst of typ * typ * pbind option * Range.range
and pbind = PBind of val_spec * typ_scheme

and tclause =  TClause of pat * expr option * dtree_target * Range.range

let debug = false

(*---------------------------------------------------------------------------
 * Nasty stuff to permit obscure polymorphic bindings.
 *
 * [bind_subexpr] actually produces the binding
 * e.g. let v2 = \Gamma ['a,'b]. ([] : 'a ,[] : 'b) in 
 *      let (x,y) = p.  
 * When v = x, gtvs = 'a,'b.  We must bind:
 *     x --> \Gamma A. fst (v2[A,<dummy>]) 
 *     y --> \Gamma A. snd (v2[<dummy>,A]).  
 * 
 * [get_subexpr] is just used to get a concrete value from a type
 * function in the middle of the "test" part of pattern matching.
 * For example, e.g.  let [x; y] = [ (\x.x); (\x.x) ] 
 * Here the constructor test needs a real list, even though the
 * r.h.s. is actually a polymorphic type function.  To do the
 * test, we apply the r.h.s. to a dummy type - it doesn't matter
 * which (unless the r.h.s. actually looks at it's type argument...)
 *------------------------------------------------------------------------- *)

type subexpr_of_expr = SubExpr of (typar_inst -> expr -> expr) * (expr * val_spec)

let bind_subexpr g amap gtps (PBind(v,tyscheme)) m (SubExpr(accessf,(ve2,v2))) =
    let e' = 
        if isNil gtps then accessf [] ve2 else 
        let freeze_var gtp = 
             if is_being_generalized gtp tyscheme then TType_var gtp 
             else Typrelns.choose_typar_solution g amap gtp in 
             
        let tyargs = map freeze_var gtps in 
        let tinst = mk_typar_inst gtps tyargs in 
        accessf tinst (mk_appl((ve2,type_of_val v2),[tyargs],[],range_of_val v2)) in
    v,mk_poly_bind_rhs m tyscheme e'

let get_subexpr (gtps,tyargs,tinst) (SubExpr(accessf,(ve2,v2))) =
    if isNil gtps then accessf [] ve2 else
    accessf tinst (mk_appl((ve2,type_of_val v2),[tyargs],[],range_of_val v2))

(*---------------------------------------------------------------------------
 * path, frontier
 *------------------------------------------------------------------------- *)

(* path reaches into a pattern.
 * The ints record which choices taken, e.g. tuple/record fields.
 * [it may be enough that subpatterns have unique paths].
 *)
type path = 
    | PathQuery of path * int
    | PathConj of path * int
    | PathTuple of path * tinst * int
    | PathRecd of path * tycon_ref * tinst * tinst  * int
    | PathUnionConstr of path * unionconstr_ref * tinst * int
    | PathArray of path * typ * int * int
    | PathExnConstr of path * tycon_ref * int
    | PathEmpty of typ 

let rec path_eq p1 p2 = 
    match p1,p2 with
    | PathQuery(p1,n1), PathQuery(p2,n2) -> (n1 = n2) && path_eq p1 p2
    | PathConj(p1,n1), PathConj(p2,n2) -> (n1 = n2) && path_eq p1 p2
    | PathTuple(p1,_,n1), PathTuple(p2,_,n2) -> (n1 = n2) && path_eq p1 p2
    | PathRecd(p1,_,_,_,n1), PathRecd(p2,_,_,_,n2) -> (n1 = n2) && path_eq p1 p2
    | PathUnionConstr(p1,_,_,n1), PathUnionConstr(p2,_,_,n2) -> (n1 = n2) && path_eq p1 p2
    | PathArray(p1,_,_,n1), PathArray(p2,_,_,n2) -> (n1 = n2) && path_eq p1 p2
    | PathExnConstr(p1,_,n1), PathExnConstr(p2,_,n2) -> (n1 = n2) && path_eq p1 p2
    | PathEmpty(_), PathEmpty(_) -> true
    | _ -> false


(*---------------------------------------------------------------------------
 * Counter example generation 
 *------------------------------------------------------------------------- *)

type refutedSet = Refuted of path * dtree_discrim 

let notNullText = "some-non-null-value"
let otherSubtypeText = "some-other-subtype"

exception CannotRefute
let refute g m path discrim = 
    let mk_unknown ty = snd(mk_compgen_local m "_" ty) in
    let rec go path = 
        match path with 
        | PathQuery _ -> raise CannotRefute
        | PathConj (p,j) -> go p
        | PathTuple (p,tys,j) -> 
             mk_tupled g m (mk_one_known p j tys) tys
             
        | PathRecd (p,tcref,tinst,finst,j) -> 
             let flds = tcref |> typs_of_tcref_rfields (mk_tcref_inst tcref tinst) |> mk_one_known p j in
             TExpr_op(TOp_recd(RecdExpr, tcref),tinst, flds,m)

        | PathUnionConstr (p,ucref,tinst,j) -> 
             let flds = ucref |> typs_of_ucref_rfields (mk_tcref_inst (tcref_of_ucref ucref) tinst)|> mk_one_known p j in
             TExpr_op(TOp_uconstr(ucref),tinst, flds,m)

        | PathArray (p,ty,len,n) -> 
             TExpr_op(TOp_array,[ty], mk_one_known p n (replicate len ty) ,m)

        | PathExnConstr (p,ecref,n) -> 
             let flds = ecref |> typs_of_ecref_rfields |> mk_one_known p n in
             TExpr_op(TOp_exnconstr(ecref),[], flds,m)

        | PathEmpty(ty) -> 
            begin match discrim with 
            | TTest_isnull -> 
                snd(mk_compgen_local m notNullText ty)
            | TTest_isinst (srcty,tgty) -> 
                snd(mk_compgen_local m otherSubtypeText ty)
            | TTest_const c -> 
                let c' = 
                    match c with 
                    | TConst_bool(b) -> TConst_bool(not b)
                    | TConst_int8(i) -> TConst_int8(Nums.i8_to_int i + 1 |> Nums.int_to_i8)
                    | TConst_int16(i) -> TConst_int16(Nums.i16_to_int i + 1 |> Nums.int_to_i16)
                    | TConst_int32(i) -> TConst_int32(Int32.add i 1l)
                    | TConst_int64(i) -> TConst_int64(Int64.add i 1L)
                    | TConst_uint8(i) -> TConst_uint8(Nums.u8_to_int i + 1 |> Nums.int_to_u8)
                    | TConst_uint16(i) -> TConst_uint16(Nums.u16_to_int i + 1 |> Nums.int_to_u16)
                    | TConst_uint32(i) -> TConst_uint32(Nums.i32_to_u32 (Int32.add (Nums.u32_to_i32 i) 1l))
                    | TConst_uint64(i) -> TConst_uint64(Nums.i64_to_u64 (Int64.add (Nums.u64_to_i64 i) 1L))
                    | TConst_float(i) -> TConst_float(Nums.float_to_ieee64 (Nums.ieee64_to_float i +. 1.0))
                    | TConst_char(i) -> TConst_char(Nums.u16_to_unichar(Nums.u16_to_int (Nums.unichar_to_u16 i) + 1 |> Nums.int_to_u16))
                    | TConst_string(s) -> TConst_string(Bytes.append s  (Bytes.string_as_unicode_bytes "a"))
                    | TConst_decimal(s) -> TConst_decimal(Bytes.append (Bytes.string_as_unicode_bytes "1") s)
                    | TConst_bigint(s) -> TConst_bigint(Bytes.append (Bytes.string_as_unicode_bytes "1") s)
                    | TConst_bignum(s) -> TConst_bignum(Bytes.append (Bytes.string_as_unicode_bytes "1") s)
(*
                    | TConst_nativeint  of int64
                    | TConst_unativeint of Nums.u64
                    | TConst_float32    of Nums.ieee32
                    | TConst_unit
                    | TConst_default 
*)                  
                    | _ -> 
                        raise CannotRefute  in 

                (* REVIEW: we could return a better enumeration literal field here if a field matches one of the enumeration cases *)

                TExpr_const(c',m,ty)
                
            | TTest_unionconstr (ucref1,tinst) -> 
                 let tcref = (tcref_of_ucref ucref1) in 
                 (* Choose the first ucref based on ordering of names *)
                 let others = 
                     tcref 
                     |> ucrefs_of_tcref 
                     |> filter (g.ucref_eq ucref1 >> not) 
                     |> List.sort (orderOn name_of_ucref compare) in
                 begin match others with 
                 | [] -> raise CannotRefute
                 | ucref2 :: _ -> 
                   let flds = ucref2 |> typs_of_ucref_rfields (mk_tcref_inst tcref tinst) |> mk_unknowns in
                   TExpr_op(TOp_uconstr(ucref2),tinst, flds,m)
                 end
                   
            | TTest_array_length (n,ty) -> 
                 TExpr_op(TOp_array,[ty], mk_unknowns (replicate (n+1) ty) ,m)
                 
            | TTest_query (pexp,restys,apatVrefOpt,idx,apinfo) when not (total_of_apinfo apinfo) -> 
                raise CannotRefute (* TODO: 'failing active pattern' *)
            | _ -> 
                raise CannotRefute
            end
    and mk_one_known p n tys = list_mapi (fun i ty -> if i = n then go p else mk_unknown ty) tys 
    and mk_unknowns tys = list_map mk_unknown tys in 
    go path

let rec combineRefutations g r1 r2 =
   match r1,r2 with
   | TExpr_val(vref,_,_), other | other, TExpr_val(vref,_,_) when name_of_vref vref = "_" -> other 
   | TExpr_val(vref,_,_), other | other, TExpr_val(vref,_,_) when name_of_vref vref = notNullText -> other 
   | TExpr_val(vref,_,_), other | other, TExpr_val(vref,_,_) when name_of_vref vref = otherSubtypeText -> other 

   | TExpr_op((TOp_exnconstr(ecref1) as op1), tinst1,flds1,m1), TExpr_op(TOp_exnconstr(ecref2), _,flds2,_) when g.tcref_eq ecref1 ecref2 -> 
        TExpr_op(op1, tinst1,List.map2 (combineRefutations g) flds1 flds2,m1)

   | TExpr_op((TOp_uconstr(ucref1) as op1), tinst1,flds1,m1), 
     TExpr_op(TOp_uconstr(ucref2), _,flds2,_)  -> 
       if g.ucref_eq ucref1 ucref2 then 
           TExpr_op(op1, tinst1,List.map2 (combineRefutations g) flds1 flds2,m1)
       (* Choose the greater of the two ucrefs based on name ordering *)
       else if name_of_ucref ucref1 < name_of_ucref ucref2 then 
           r2
       else 
           r1
        
   | TExpr_op(op1, tinst1,flds1,m1), TExpr_op(_, _,flds2,_) -> 
        TExpr_op(op1, tinst1,List.map2 (combineRefutations g) flds1 flds2,m1)
        
   | TExpr_const(c1, m1, ty1), TExpr_const(c2,_,_) -> 
       let c12 = 

           (* Make sure longer strings are greater, not the case in the default ordinal comparison *)
           (* This is needed because the individual counter examples make longer strings *)
           let maxStrings b1 b2 = 
               let s1 = Bytes.unicode_bytes_as_string b1 in 
               let s2 = Bytes.unicode_bytes_as_string b2 in 
               let c = compare (String.length s1) (String.length s2) in 
               if c < 0 then b2 else if c > 0 then b1 else 
               if s1 < s2 then b2 else b1 in
               
           match c1,c2 with 
           | TConst_string(s1), TConst_string(s2) -> TConst_string(maxStrings s1 s2)
           | TConst_decimal(s1), TConst_decimal(s2) -> TConst_bigint(maxStrings s1 s2)
           | TConst_bigint(s1), TConst_bigint(s2) -> TConst_bigint(maxStrings s1 s2)
           | TConst_bignum(s1), TConst_bignum(s2) -> TConst_bignum(maxStrings s1 s2)
           | _ -> max c1 c2  in
           
       (* REVIEW: we couldd return a better enumeration literal field here if a field matches one of the enumeration cases *)
       TExpr_const(c12, m1, ty1)

   | _ -> r1 



let showCounterExample g denv m refuted = 
   try
      let refutations = list_map (fun (Refuted(path,discrim)) -> refute g m path discrim) refuted in
      let counterExample = 
          match refutations with 
          | [] -> raise CannotRefute
          | h :: t -> 
              if verbose then dprintf1 "h = %s\n" (Layout.showL (exprL h));
              List.fold_left (combineRefutations g) h t in
      Some(Layout.showL (NicePrint.dataExprL denv counterExample))
    with CannotRefute ->    
      None 
    | e -> 
      warning(InternalError(Printf.sprintf "<failure during counter example generation: %s>" (Printexc.to_string e),m));
      None
       
(*---------------------------------------------------------------------------
 * Basic problem specification
 *------------------------------------------------------------------------- *)
    
type ruleNumber = int

type active = Active of path * subexpr_of_expr * pat

type actives = active list

type frontier = Frontier of ruleNumber * actives * expr vspec_map

type investigationPoint = Investigation of ruleNumber * dtree_discrim * path

(* Note: actives must be a SortedDictionary *)
(* REVIEW: improve these data structures, though surprisingly these functions don't tend to show up *)
(* on profiling runs *)
let rec mem_of_actives p1 actives = 
    match actives with 
    | [] -> false 
    | (Active(p2,_,_)) :: rest -> path_eq p1 p2 or mem_of_actives p1 rest

let rec lookup_active x l = 
    match l with 
    | [] -> raise Not_found
    | (Active(h,r1,r2)::t) -> if path_eq x h then (r1,r2) else lookup_active x t

let rec remove_active x l = 
    match l with 
    | [] -> []
    | ((Active(h,_,_) as p) ::t) -> if path_eq x h then t else p:: remove_active x t

(*---------------------------------------------------------------------------
 * Utilities
 *------------------------------------------------------------------------- *)

(* tpinst is required because the pattern is specified w.r.t. generalized type variables. *)
let range_of_pat t = 
    match t with 
    | TPat_null m | TPat_isinst (_,_,_,m) | TPat_const (_,m) | TPat_unionconstr (_ ,_,_,m) 
    | TPat_exnconstr (_,_,m) | TPat_query (_,_,m) | TPat_range(_,_,m) 
    | TPat_recd(_,_,_,m) | TPat_tuple(_,_,m) | TPat_array(_,_,m) | TPat_disjs(_,m) | TPat_conjs(_,m) | TPat_as(_,_,m) | TPat_wild(m) -> m
  
let discrim_of_switchpat g tpinst t = 
    match t with 
    | TPat_null m -> 
        Some(TTest_isnull)
    | TPat_isinst (srcty,tgty,_,m) -> 
        Some(TTest_isinst (inst_type tpinst srcty,inst_type tpinst tgty))
    | TPat_exnconstr(tcref,_,m) -> 
        Some(TTest_isinst (g.exn_ty,mk_tyapp_ty tcref []))
    | TPat_const (c,m) -> 
        Some(TTest_const c)
    | TPat_unionconstr (c,tyargs',_,m) -> 
        Some(TTest_unionconstr (c,inst_types tpinst tyargs'))
    | TPat_array (args,ty,m) -> 
        Some(TTest_array_length (List.length args,ty))
    | TPat_query ((pexp,restys,apatVrefOpt,idx,apinfo),_,m) -> 
        Some(TTest_query (pexp, inst_types tpinst restys, apatVrefOpt,idx,apinfo))
    | _ -> None

let const_of_discrim discrim =
    match discrim with 
    | TTest_const x -> x
    | _ -> failwith "not a const case"

let const_of_case c = const_of_discrim(discrim_of_case c)

let getDiscrimOfCase (TCase(discrim,_)) = discrim

(*---------------------------------------------------------------------------
 * Choose a set of investigations that can be performed simultaneously 
 *------------------------------------------------------------------------- *)

let rec choose_simultaneous_edge_set prevOpt f l =
    match l with 
    | [] -> [],[]
    | h::t -> 
        match f prevOpt h with 
        | Some x,_ ->         
             let l,r = choose_simultaneous_edge_set (Some x) f t in 
             x :: l, r
        | None,cont -> 
             let l,r = choose_simultaneous_edge_set prevOpt f t in 
             l, h :: r

let canCompactConstantClass c = 
    match c with 
    | TConst_int8 _ | TConst_int16 _ | TConst_int32 _ | TConst_uint8 _ 
    | TConst_uint16 _ | TConst_uint32 _ | TConst_char _ -> true
    | _ -> false
    
                         
let discrim_same_simultaneous_class g d1 d2 =
  match d1,d2 with 
  | TTest_const _,              TTest_const _ 
  | TTest_isnull ,               TTest_isnull 
  | TTest_array_length _,   TTest_array_length _
  | TTest_unionconstr _,    TTest_unionconstr _  -> true

  | TTest_isinst _, TTest_isinst _ -> false

(*   | TTest_const (c),        TTest_const (c) -> canCompactConstantClass (c) *)

  | TTest_query (_,_,apatVrefOpt1,n1,apinfo1),        TTest_query (_,_,apatVrefOpt2,n2,apinfo2) -> 
      begin match apatVrefOpt1, apatVrefOpt2 with 
      | Some vref1, Some vref2 -> g.vref_eq vref1 vref2 
      | _ -> false (* for equality purposes these are considered different classes of discriminators! This is because adhoc computed patterns have no identity! *)
      end

  | _ -> false


(*---------------------------------------------------------------------------
 * The heuristic
 *------------------------------------------------------------------------- *)

(* Current heuristic: just choose left-to-right in the first rule. *)
let choose_l_to_r frontiers =
    match frontiers with 
    | Frontier (i,actives,_) ::t -> 
        let rec choose l = 
            match l with 
            | [] -> failwith "choose_l_to_r: no non-immediate patterns in first rule"
            | (Active(path,_,(TPat_null _ | TPat_isinst _ | TPat_exnconstr _ | TPat_unionconstr _ | TPat_array _ | TPat_const _ | TPat_query _ | TPat_range _)) as active)
                ::t -> active
            | _ :: t -> choose t in 
        choose actives
    | [] -> failwith "choose_l_to_r: no frontiers!"

(*---------------------------------------------------------------------------
 * build_switch: Build a dtree, equivalent to: TDSwitch("expr",edges,default,m) 
 *
 * Once we've chosen a particular active to investigate, we compile the
 * set of edges affected by this investigation into a switch.  
 *
 *   - For TTest_query(...,None,...) there is only one edge
 *
 *   - For TTest_isinst there are multiple edges, which we can't deal with in 
 *     one switch, so we make an iterated if-then-else to cover the cases. We
 *     should probably adjust the code to only choose one edge in this case.
 *
 *   - Compact integer switches become a single switch.  Non-compact integer
 *     switches, string switches and floating point switches are treated in the
 *     same way as TTest_isinst.
 *------------------------------------------------------------------------- *)

let rec build_switch resVarOpt g expr edges dflt m =
    if debug then  dprintf2 "--> build_switch resVarOpt@%a\n" output_range m; 
    begin match edges,dflt with 
    | [], None      -> failwith "internal error: no edges and no default"
    | [], Some dflt -> dflt      (* NOTE: first time around, edges<>[] *)
    (* Optimize the case where the match always succeeds *)
    | [TCase(_,tree)], None -> tree

    (* 'isinst' tests where we have stored the result of the 'isinst' in a variable *)
    (* In this case the 'expr' already holds the result of the 'isinst' test. *)
    
    | (TCase(TTest_isinst _,success) as edge):: edges, dflt  when isSome(resVarOpt) -> 
        TDSwitch(expr,[TCase(TTest_isnull,(build_switch None g expr edges dflt) m)],Some success,m)    
        
    | (TCase((TTest_isnull | TTest_isinst _),_) as edge):: edges, dflt  -> 
        TDSwitch(expr,[edge],Some (build_switch resVarOpt g expr edges dflt m),m)    
        (*begin match dflt with 
        | None      -> error(InternalError("exception/null/isinst matches need default cases!",m))
        | Some dflt -> TDSwitch(get_subexpr subexpr,[edge],Some (build_switch resVarOpt g subexpr edges (Some dflt) m),m)
        end*)

    (* All these should also always have default cases *)
    | TCase(TTest_const (TConst_decimal _ | TConst_bigint _ | TConst_bignum _ | TConst_string _ | TConst_float32 _ |  TConst_float _ |
                         TConst_int8 _ | TConst_uint8 _| TConst_int16 _ | TConst_uint16 _ | 
                         TConst_int32 _ | TConst_uint32 _ | TConst_int64 _ | TConst_uint64 _ | 
                         TConst_nativeint _ | TConst_unativeint _ | TConst_char _ ),_) :: _, None -> 
        error(InternalError("inexhaustive match - need a default cases!",m))

    (* Split string, float, uint64, int64, unativeint, nativeint matches into serial equality tests *)
    | TCase((TTest_array_length _ | TTest_const (TConst_float32 _ | TConst_float _ | TConst_string _ | 
                                                 TConst_decimal _ | TConst_bigint _ | TConst_bignum _ | TConst_int64 _ | 
                                                 TConst_uint64 _ | TConst_nativeint _ | 
                                                 TConst_unativeint _)),_) :: _, Some dflt -> 
        fold_right 
          (fun (TCase(discrim,tree)) sofar -> 
            let testexpr = expr in 
            let testexpr = 
              match discrim with 
              | TTest_array_length(n,ty)       -> 
                  let v,vexp,bind = mk_compgen_bind g nng "$testexpr" m testexpr in
                  mk_let_bind m bind (mk_lazy_and g m (mk_nonnull_test g m vexp) (mk_ceq g m (mk_ldlen g m vexp) (mk_int g m n)))
              | TTest_const (TConst_string _ as c)  -> 
                  mk_call_poly_equals_outer g m g.string_ty testexpr (TExpr_const(c,m,g.string_ty))
              | TTest_const (TConst_bigint _ as c)  -> 
                  mk_call_poly_equals_outer g m g.bigint_ty testexpr (TExpr_const(c,m,g.bigint_ty))
              | TTest_const (TConst_decimal _ as c)  -> 
                  mk_call_poly_equals_outer g m g.decimal_ty testexpr (TExpr_const(c,m,g.bigint_ty))
              | TTest_const (TConst_bignum _ as c)  -> 
                  mk_call_poly_equals_outer g m g.bignum_ty testexpr (TExpr_const(c,m,g.bignum_ty))
              | TTest_const ((TConst_float _ | TConst_float32 _ | TConst_int64 _ | TConst_uint64 _ | TConst_nativeint _ | TConst_unativeint _) as c)   -> 
                  mk_ceq g m testexpr (TExpr_const(c,m,type_of_expr g testexpr))
              | _ -> error(InternalError("strange switch",m)) in 
            mk_bool_switch m testexpr tree sofar)
          edges
          dflt

    (* Split integer and char matches into compact fragments which will themselves become switch *)
    (* statements. *)
    | TCase(TTest_const c,_) :: _, Some dflt when canCompactConstantClass c -> 
        let edge_compare c1 c2 = 
          match const_of_case c1,const_of_case c2 with 
          | (TConst_int8 i1),(TConst_int8 i2) -> compare i1 i2
          | (TConst_int16 i1),(TConst_int16 i2) -> compare i1 i2
          | (TConst_int32 i1),(TConst_int32 i2) -> compare i1 i2
          | (TConst_uint8 i1),(TConst_uint8 i2) -> compare i1 i2
          | (TConst_uint16 i1),(TConst_uint16 i2) -> compare i1 i2
          | (TConst_uint32 i1),(TConst_uint32 i2) -> compare i1 i2
(*
          | (TConst_int64 i1),(TConst_int64 i2) -> compare i1 i2
          | (TConst_nativeint i1),(TConst_nativeint i2) -> compare i1 i2
          | (TConst_uint64 i1),(TConst_uint64 i2) -> compare i1 i2
          | (TConst_unativeint i1),(TConst_unativeint i2) -> compare i1 i2
*)
          | (TConst_char c1),(TConst_char c2) -> compare c1 c2
          | _ -> failwith "illtyped term during pattern compilation"  in 
        let edges' = sort edge_compare edges in
        let rec compactify curr edges = 
          if debug then  dprintf2 "--> compactify@%a\n" output_range m; 
          match curr,edges with 
          | None,[] -> []
          | Some last,[] -> [rev last]
          | None,h::t -> compactify (Some [h]) t
          | Some (prev::moreprev),h::t -> 
              begin match const_of_case prev,const_of_case h with 
              | TConst_int8 iprev,TConst_int8 inext when Int32.add (Nums.i8_to_i32 iprev) (Int32.of_int 1) = Nums.i8_to_i32 inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_int16 iprev,TConst_int16 inext when Int32.add (Nums.i16_to_i32 iprev) (Int32.of_int 1) = Nums.i16_to_i32 inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_int32 iprev,TConst_int32 inext when Int32.add iprev (Int32.of_int 1) = inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_uint8 iprev,TConst_uint8 inext when Int32.add (Nums.u8_to_i32 iprev) (Int32.of_int 1) = Nums.u8_to_i32 inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_uint16 iprev,TConst_uint16 inext when Int32.add (Nums.u16_to_i32 iprev) (Int32.of_int 1) = Nums.u16_to_i32 inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_uint32 iprev,TConst_uint32 inext when Int32.add (Nums.u32_to_i32 iprev) (Int32.of_int 1) = Nums.u32_to_i32 inext -> 
                  compactify (Some (h::prev::moreprev)) t
(*
              | TConst_int64 iprev,TConst_int64 inext when Int64.add iprev (Int64.of_int 1) = inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_nativeint iprev,TConst_nativeint inext when Int64.add iprev (Int64.of_int 1) = inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_uint64 iprev,TConst_uint64 inext when Int64.add (Nums.u64_to_i64 iprev) (Int64.of_int 1) = Nums.u64_to_i64 inext -> 
                  compactify (Some (h::prev::moreprev)) t
              | TConst_unativeint iprev,TConst_unativeint inext when Int64.add (Nums.u64_to_i64 iprev) (Int64.of_int 1) = Nums.u64_to_i64 inext -> 
                  compactify (Some (h::prev::moreprev)) t
*)
              | TConst_char cprev,TConst_char cnext when (Nums.u16_to_int (Nums.unichar_to_u16 cprev) + 1) = (Nums.u16_to_int (Nums.unichar_to_u16 cnext)) -> 
                  compactify (Some (h::prev::moreprev)) t
              |       _ ->  (rev (prev::moreprev)) :: compactify None edges
              end
          | _ -> failwith "internal error: compactify" in 
        let edge_groups = compactify None edges' in 
        fold_right 
          (fun edge_group sofar ->  TDSwitch(expr,edge_group,Some sofar,m))
          edge_groups
          dflt

    (* For a total pattern match, run the active pattern, bind the result and *)
    (* recursively build a switch in the choice type *)
    | (TCase(TTest_query _,_)::rest), dflt -> 
       error(InternalError("TTest_query should have been eliminated",m));

    (* For a complete match, optimize one test to be the default *)
    | (TCase(test,tree)::rest), None -> TDSwitch (expr,rest,Some tree,m)

    (* Otherwise let codegen make the choices *)
    | _ -> TDSwitch (expr,edges,dflt,m)
    end

let rec patL pat = 
    if debug then  dprintf0 "--> patL\n"; 
    match pat with
    | TPat_query (_,pat,_) -> Layout.(--) (Layout.wordL "query") (patL pat)
    | TPat_wild _ -> Layout.wordL "wild"
    | TPat_as _ -> Layout.wordL "var"
    | TPat_tuple (pats, _, _) 
    | TPat_array (pats, _, _) -> Layout.bracketL (Layout.tupleL (List.map patL pats))
    | _ -> Layout.wordL "?" 
  
let rec pathL p = Layout.wordL "<path>"
     
let activeL (Active (path, subexpr, pat)) =
    Layout.(--) (Layout.wordL "Active") (Layout.tupleL [pathL path; patL pat]) 
     
let frontierL (Frontier (i,actives,_)) =
    Layout.(--) (Layout.wordL "Frontier") (Layout.tupleL [intL i; Layout.listL activeL actives]) 

let mk_frontiers investigations i = 
    map (fun (actives,vspec_map) -> Frontier(i,actives,vspec_map)) investigations

let get_rule_idx (Frontier (i,active,vspec_map)) = i

(*---------------------------------------------------------------------------
 * The algorithm
 *------------------------------------------------------------------------- *)

let rec is_partial p = 
    match p with 
    | TPat_query ((_,restys,apatVrefOpt,idx,apinfo),p,m) -> not (total_of_apinfo apinfo) or is_partial p
    | TPat_const _ -> false
    | TPat_wild _ -> false
    | TPat_as (p,_,_) -> is_partial p
    | TPat_disjs (ps,_) | TPat_conjs(ps,_) 
    | TPat_tuple (ps,_,_) | TPat_exnconstr(_,ps,_) 
    | TPat_array (ps,_,_) | TPat_unionconstr (_,_,ps,_)-> List.exists is_partial ps
    | TPat_recd (_,_,psl,_) -> List.exists (snd >> is_partial) psl
    | TPat_range _ -> false
    | TPat_null _ -> false
    | TPat_isinst _ -> false

let rec erase_partial inpp = 
    match inpp with 
    | TPat_query ((expr,restys,apatVrefOpt,idx,apinfo),p,m) -> 
       if (total_of_apinfo apinfo) then TPat_query ((expr,restys,apatVrefOpt,idx,apinfo),erase_partial p,m)
       else TPat_disjs ([],m) (* always fail *)
    | TPat_as (p,x,m) -> TPat_as (erase_partial p,x,m)
    | TPat_disjs (ps,m) -> TPat_disjs(erase_partials ps, m)
    | TPat_conjs(ps,m) -> TPat_conjs(erase_partials ps, m)
    | TPat_tuple (ps,x,m) -> TPat_tuple(erase_partials ps, x, m)
    | TPat_exnconstr(x,ps,m) -> TPat_exnconstr(x,erase_partials ps,m) 
    | TPat_array (ps,x,m) -> TPat_array (erase_partials ps,x,m)
    | TPat_unionconstr (x,y,ps,m) -> TPat_unionconstr (x,y,erase_partials ps,m)
    | TPat_recd (x,y,ps,m) -> TPat_recd (x,y,map (map2'2 erase_partial) ps,m)
    | TPat_const _ 
    | TPat_wild _ 
    | TPat_range _ 
    | TPat_null _ 
    | TPat_isinst _ -> inpp
and erase_partials inps = map erase_partial inps



(*---------------------------------------------------------------------------
 * The algorithm
 *------------------------------------------------------------------------- *)

type edge_discrim = EdgeDiscrim of int * dtree_discrim * range
let get_discrim (EdgeDiscrim(_,discrim,_)) = discrim

let when_of_clause (TClause(p,whenOpt,_,_)) = whenOpt
let pat_of_clause (TClause(p,whenOpt,_,_)) = p
let range_of_clause (TClause(p,whenOpt,_,m)) = m
let vs_of_clause (TClause(p,whenOpt,TTarget(vs,_),m)) = vs

let compilePatternBasic 
        nng 
        g denv amap exprm matchm 
        warnOnUnused 
        warnOnIncomplete 
        actionOnFailure 
        (topv,topgtvs) 
        (clausesL: tclause list) 
        ty =
    (* Add the targets to a match builder *)
    let mbuilder = MatchBuilder.create exprm in
    list_iteri (fun i (TClause(_,_,tg,_)) -> MatchBuilder.add_target mbuilder tg |> ignore) clausesL;
    
    (* Add the incomplete or rethrow match clause on demand, printing a *)
    (* warning if necessary (only if it is ever exercised) *)
    let incompleteMatchClauseOnce = ref None in
    let getIncompleteMatchClause (refuted) = 
        (* This is lazy because MatchBuilder.add_target is imperative and we emit a warning when the lazy thunk gets evaluated *)
        match !incompleteMatchClauseOnce with 
        | None -> 
                (* Emit the incomplete match warning *)               
                if (actionOnFailure = Incomplete) && warnOnIncomplete then 
                    warning (MatchIncomplete (false,showCounterExample g denv exprm refuted, exprm));
                let throwExpr =
                    match actionOnFailure with
                      | FailFilter  -> mk_int g matchm 0 (* return 0 from the filter *)
                      | Rethrow     -> mk_rethrow matchm ty (expr_for_val matchm topv) (* rethrow unmatched try-catch exn *)
                      | IncompleteWarnOnLast  
                      | Throw       -> mk_throw   matchm ty (expr_for_val matchm topv) (* throw instead of rethrow on computation expression unmatched try-catch *)
                      | Incomplete  -> mk_throw   matchm ty (mk_exnconstr(mk_MFCore_tcref g.fslibCcu "MatchFailureException", 
                                           (* REVIEW: use unicode names for files and get rid of this use of Bytes.string_as_unicode_bytes *)
                                               [ mk_string g matchm (Bytes.string_as_unicode_bytes (file_of_range matchm)); 
                                                 mk_int g matchm (matchm |> start_of_range |> line_of_pos); 
                                                 mk_int g matchm (matchm |> start_of_range |> col_of_pos)],matchm)) in
                let tg = TTarget([],throwExpr) in
                MatchBuilder.add_target mbuilder tg |> ignore;
                let clause = TClause(TPat_wild matchm,None,tg,matchm) in 
                incompleteMatchClauseOnce := Some(clause);
                clause
                
        | Some c -> c in

    (* Helpers to get the variables bound at a target. We conceptually add a dummy clause that will always succeed with a "throw" *)
    let clausesA = Array.of_list clausesL in 
    let nclauses = Array.length clausesA in 
    let lastMatchClauseWarnngGiven = ref false in
    let get_clause i refuted = 
        if i < nclauses then  (
            (* Seqeunce expression constructs such as '[ for (h::t) in x -> ...' ]' previously had a semantics *)
            (* where unmatched elements are skipped and no results generated. This was/is implemented by *)
            (* providing a last clause that returned no elements in the generator. However, this semantics *)
            (* is not consistent with the interpretation of sequence expressions in a generator computation expression. *)
            (* Hence we now give a warning when the last clause is referenced. *)
            if not !lastMatchClauseWarnngGiven && (actionOnFailure = IncompleteWarnOnLast) && (i = nclauses - 1) then  (
                warning (MatchIncomplete (true,showCounterExample g denv exprm refuted, exprm));
                lastMatchClauseWarnngGiven := true
            );
            clausesA.(i)  
        )
        else if i = nclauses then getIncompleteMatchClause(refuted)
        else failwith "get_clause" in 
    let get_clause_vs i refuted = vs_of_clause (get_clause i refuted) in
    let get_clause_when i refuted = when_of_clause (get_clause i refuted) in 
    
    (* Different uses of parameterized active patterns have different identities as far as paths *)
    (* are concerned. *)
    let genUniquePathId = let count= ref 10000 in fun () -> incr count; !count in

    (* Build versions of these functions which apply a dummy instantiation to the overall type arguments *)
    let get_subexpr,discrim_of_switchpat = 
        let tyargs = map (fun _ -> g.unit_ty) topgtvs in 
        let unit_tpinst = mk_typar_inst topgtvs tyargs in 
        get_subexpr (topgtvs,tyargs,unit_tpinst),
        discrim_of_switchpat g unit_tpinst in


     (* The main recursive loop of the pattern match compiler *)
    let rec compile refuted frontiers = 
        if debug then dprintf1 "frontiers = %s\n" (String.concat ";" (map (get_rule_idx >> string_of_int) frontiers));
        match frontiers with
        | [] -> failwith "compilePattern:compile - empty clauses: at least the final clause should always succeed"
        | (Frontier (i,active,vspec_map)) :: rest ->

            (* Check to see if we've got a succeeding clause.  There may still be a 'when' condition for the clause *)
            match active with
            | [] -> 
                if debug then dprintf1 "generating success node for rule %d\n" i;
                let vs2 = get_clause_vs i refuted in 
                let es2 = map (fun v -> match (vspec_map_tryfind v vspec_map) with None -> failwith ("internal error: no binding for "^(name_of_val v)) | Some res -> res) vs2 in 
                let rhs' = TDSuccess(es2, i) in 
                begin match get_clause_when i refuted with 
                | Some when_expr -> 
                    if debug then dprintf1 "generating success node for rule %d, with 'when' clause\n" i;
                    (* We duplicate both the bindings and the guard expression to ensure uniqueness of bound variables: the same vars are boun in the guard and the targets *)
                    (* REVIEW: this is also duplicating the guard when "or" patterns are used, leading to code explosion for large guards with many "or" patterns *)
                    let m = (range_of_expr when_expr) in 
                    let when_expr = copy_expr g false (mk_lets_bind m (mk_binds vs2 es2) when_expr) in
                    mk_bool_switch m when_expr rhs' (compile refuted rest)
                | None -> rhs' 
                end

            | _ -> 
                if debug then dprintf2 "Investigating based on rule %d, #active = %d\n" i (length active);
                (* Otherwise choose a point (i.e. a path) to investigate. *)
                let (Active(path,subexpr,pat))  = choose_l_to_r frontiers in 
                match pat with
                (* All these constructs should have been eliminated in bind_immediate *)
                | TPat_as _   | TPat_tuple _  | TPat_wild _      | TPat_disjs _  | TPat_conjs _          
                | TPat_recd _ -> failwith "unexpected pattern"	      
                (* Leaving the ones where we have real work to do *)
                | TPat_null _ | TPat_isinst _ | TPat_array _  | TPat_exnconstr _ 
                | TPat_unionconstr _ | TPat_const _ |  TPat_range _ 
                | TPat_query _ -> 

                    (* Select the set of discriminators which we can handle in one test, or as a series of *)
                    (* iterated tests, e.g. in the case of TPat_isinst.  Ensure we only take at most one class of TPat_query(_) at a time. *)
                    (* Record the rule numbers so we know which rule the TPat_query cam from, so that when we project through *)
                    (* the frontier we only project the right rule. *)
                    if debug then dprintf0 "choose_simultaneous_edge_set\n";
                    let simulSetOfEdgeDiscrims,fallthroughPathFrontiers = 
                      frontiers |> choose_simultaneous_edge_set None (fun prevOpt (Frontier (i',active',_)) -> 
                          if mem_of_actives path active' then 
                              let p = lookup_active path active' |> snd in 
                              match discrim_of_switchpat p with
                              | Some discrim -> 
                                  if (match prevOpt with None -> true | Some (EdgeDiscrim(_,discrimPrev,_)) -> discrim_same_simultaneous_class g discrim discrimPrev) then (
                                      if debug then dprintf1 "taking rule %d\n" i';
                                      Some (EdgeDiscrim(i',discrim,range_of_pat p)),true
                                  ) else 
                                      None,false
                                                                    
                              | None -> 
                                  None,true
                          else 
                              None,true) in 

                    let resVarOpt,bindOpt =         
                       match simulSetOfEdgeDiscrims with 
                        (* Very simple 'isinst' tests: put the result of 'isinst' in a local variable *)
                        
                       | EdgeDiscrim(i',(TTest_isinst (srcty,tgty)),m) :: rest 
                                  (* check we can use a simple 'isinst' instruction *)
                                  when not (is_typar_ty tgty) && 
                                       not (typNullIsTrueValue g tgty) &&
                                       not (typNullNever g tgty) && 
                                       isNil topgtvs ->

                           let v,vexp = mk_local m (nng.nngApply "$typeTestResult" m) tgty in
                           let argexp = get_subexpr subexpr in
                           let appexp = mk_isinst tgty argexp matchm in 
                           Some(vexp),Some(TBind(v,appexp))

                        (* Active pattern matches: check if all the cases refer to the same active pattern: give a warning if they do not. *)
                        (* Also create a variable to hold the results of executing the active pattern. *)
                       | (EdgeDiscrim(i',(TTest_query(pexp,restys,resVarOpt,_,apinfo)),m) :: rest) ->
                           if debug then dprintf0 "Building result var for active pattern...\n";
                           
                           if nonNil topgtvs then error(InternalError("unexpected generalized type variables when compiling an active pattern",m));
                           let rty = mk_apinfo_result_typ g apinfo restys in
                           let v,vexp = mk_local m (nng.nngApply "$tpres" m) rty in
                           let argexp = get_subexpr subexpr in
(*
                           if total_of_apinfo apinfo && isSome resVarOpt &&
                              not (List.for_all (get_discrim >> get_vref >> g.vref_eq (the resVarOpt)) simulSetOfEdgeDiscrims) then
                               error(Error("A set of active pattern labels used in a match resolves to different active pattern values. You may need to use qualified paths to indicate the exact discrimination set to use in this pattern match.",m));
*)
                           let appexp = mk_appl((pexp,type_of_expr g pexp), [], [argexp],m) in
                           
                           Some(vexp),Some(TBind(v,appexp))
                        | _ -> None,None in
                            
                    (* For each case, recursively compile the residue decision trees that result if that case successfully matches *)
                    let simulSetOfCases, _ = 

                        simulSetOfEdgeDiscrims  
                            |> map_concat_acc_list 
                                (fun taken (EdgeDiscrim(i',discrim,m)) -> 
                                     (* Check to see if we've already collected the edge for this case, in which case skip it. *)
                                     if List.exists (discrim_subsumed_by g amap m discrim) taken  then 
                                       (* Skip this edge: it is refuted *)
                                       ([],taken) 
                                     else 
                                       (* Convert active pattern edges to tests on results data *)
                                       let discrim' = 
                                           match discrim with 
                                           | TTest_query(pexp,restys,apatVrefOpt,idx,apinfo) -> 
                                               let aparity = length (names_of_apinfo apinfo) in 
                                               let total = total_of_apinfo apinfo in 
                                               if not total && aparity > 1 then 
                                                   error(Error("partial active patterns may only generate one result",m));
                                               
                                               if not total then TTest_unionconstr(mk_some_ucref g,restys)
                                               else if aparity <= 1 then TTest_const(TConst_unit) 
                                               else TTest_unionconstr(mk_choices_ucref g aparity idx,restys) 
                                           | _ -> discrim in 
                                       (* Project a successful edge through the frontiers. *)
                                       let investigation = Investigation(i',discrim,path) in 
                                       
                                       let frontiers = mapConcat (project resVarOpt investigation) frontiers in
                                       (* Return the edge *)
                                       [TCase(discrim',compile refuted frontiers)], (discrim :: taken) )
                                [] in
                          
                          
                    assert (nonNil(simulSetOfCases));

                    (* Check if match is complete, if so optimize the default case away. *)
                    if debug then (
                        dprintf2 "#fallthroughPathFrontiers = %d, #simulSetOfEdgeDiscrims = %d\n"  (length fallthroughPathFrontiers) (length simulSetOfEdgeDiscrims);
                        
                        dprintf0 "Making cases for each discriminator...\n";
                        
                        dprintf1 "#edges = %d\n" (length simulSetOfCases);

                        dprintf2 "Checking for completeness of edge set from earlier investigation of rule %d, #active = %d\n" i (length active);
                    );

                    (* Work out what the default/fall-through tree looks like, is any *)
                
                    let defaultTreeOpt  : dtree option = 

                        let simulSetOfDiscrims = map getDiscrimOfCase simulSetOfCases in 

                        let isRefuted (Frontier (i',active',_)) = 
                            mem_of_actives path active' &&
                            let p = lookup_active path active' |> snd in 
                            match discrim_of_switchpat p with 
                            | Some(discrim) -> List.exists (discrim_subsumed_by g amap exprm discrim) simulSetOfDiscrims 
                            | None -> false in

                        match simulSetOfDiscrims with 
                        | TTest_const (TConst_bool b) :: _ when length simulSetOfCases = 2 ->  None
                        | TTest_const (TConst_unit) :: _  ->  None
                        | TTest_unionconstr (ucref,_) :: _ when  length simulSetOfCases = Array.length (uconstrs_array_of_tcref (tcref_of_ucref ucref)) -> None                      
                        | TTest_query _ :: _ -> error(InternalError("TTest_query should have been eliminated",matchm))
                        | _ -> 
                            let fallthroughPathFrontiers = filter (isRefuted >> not) fallthroughPathFrontiers in
                            
                            (* Add to the refuted set *)
                            let refuted = (list_map (fun discrim -> Refuted(path,discrim)) simulSetOfDiscrims) @ refuted in 
                        
                            if debug then dprintf0 "Edge set was incomplete. Compiling remaining cases\n";
                            match fallthroughPathFrontiers with
                            | [] -> 
                                None
                            | _ -> 
                                Some(compile refuted fallthroughPathFrontiers) in

                    if debug then dprintf1 "Do we have a default tree: %b\n" (isNone defaultTreeOpt);
                        
                    (* OK, build the whole tree and whack on the binding if any *)
                    let finalDecisionTree = 
                        let inpExprToSwitch = (match resVarOpt with Some vexp -> vexp | None -> get_subexpr subexpr) in 
                        let tree = build_switch resVarOpt g inpExprToSwitch simulSetOfCases defaultTreeOpt matchm in 
                        match bindOpt with 
                        | None -> tree
                        | Some bind -> TDBind (bind,tree) in
                        
                    finalDecisionTree

          
    (* Build a new frontire that represents the result of a successful investigation *)
    (* at rule point (i',discrim,path) *)
    and project vexpOption (Investigation(i',discrim,path)) (Frontier (i, active,vspec_map) as frontier) =
        if debug then dprintf2 "projecting success of investigation encompassing rule %d through rule %d \n" i' i;

        if (mem_of_actives path active) then
            let (SubExpr(accessf,ve) as e),pat = lookup_active path active in 
            if debug then dprintf0 "active...\n";

            let mk_sub_frontiers path accessf' active' argpats pathBuilder = 
                let mk_sub_active j p = 
                    let newSubExpr = SubExpr(accessf' j, ve) in 
                    let newPath = pathBuilder path j in
                    Active(newPath, newSubExpr, p) in 
                let new_actives = list_mapi mk_sub_active argpats in 
                let investigations = bind_immediate_multi new_actives (active', vspec_map) in 
                mk_frontiers investigations i in

            let active' = remove_active path active in
            match pat with 
            | TPat_wild _ | TPat_as _ | TPat_tuple _ | TPat_disjs _ | TPat_conjs _ | TPat_recd _ -> failwith "unexpected pattern"
            | TPat_query ((_,restys,apatVrefOpt,idx,apinfo),p,m) -> 
            
                if total_of_apinfo apinfo then
            
                    if (isNone apatVrefOpt && i = i') 
                       || (discrim_eq g discrim (the (discrim_of_switchpat pat))) then
                        let aparity = length (names_of_apinfo apinfo) in 
                        let accessf' j tpinst e' = 
                            if aparity <= 1 then the vexpOption 
                            else
                                let ucref = mk_choices_ucref g aparity idx in 
                                mk_uconstr_field_get(the vexpOption,ucref,inst_types tpinst restys,j,exprm) in
                        mk_sub_frontiers path accessf' active' [p] (fun path j -> PathQuery(path,j))

                    else if isNone apatVrefOpt then

                        [frontier] (* unlike most other cases the frontier remains here *)
                    else
                        []
                else 
                    if i = i' then
                            let accessf' j tpinst _ =  
                                mk_uconstr_field_get(the vexpOption, mk_some_ucref g, inst_types tpinst restys, 0, exprm) in
                            mk_sub_frontiers path accessf' active' [p] (fun path j -> PathQuery(path,j))
                    else 
                        [frontier]  (* unlike other cases the frontier remains here *)

            | _ when not (discrim_eq g discrim (the (discrim_of_switchpat pat))) -> []

            | TPat_unionconstr (ucref, tyargs, argpats,_) ->
                let accessf' j tpinst e' = mk_uconstr_field_get(accessf tpinst e',ucref,inst_types tpinst tyargs,j,exprm) in 
                mk_sub_frontiers path accessf' active' argpats (fun path j -> PathUnionConstr(path,ucref,tyargs,j))

            | TPat_array (argpats,ty,_) ->
                let accessf' j tpinst e' = mk_call_array_get g exprm ty (accessf tpinst e') (mk_int g exprm j) in 
                mk_sub_frontiers path accessf' active' argpats (fun path j -> PathArray(path,ty,List.length argpats,j))

            | TPat_exnconstr (ecref, argpats,_) ->
                let accessf' j tpinst e' = mk_exnconstr_field_get(accessf tpinst e',ecref,j,exprm) in 
                mk_sub_frontiers path accessf' active' argpats (fun path j -> PathExnConstr(path,ecref,j))

            | TPat_isinst (srcty,tgty,Some pbind,_) -> 
                let accessf' tpinst e' = match vexpOption with Some e -> e | _ -> mk_unbox (inst_type tpinst tgty) (accessf tpinst e') exprm in 
                let (v,e') =  bind_subexpr g amap topgtvs pbind exprm (SubExpr(accessf',ve)) in
                [Frontier (i, active', vspec_map_add v e' vspec_map)]

            | TPat_const _ | TPat_null _ | TPat_isinst (_,_,None,_) -> 
                [Frontier (i, active',vspec_map)]

            | _ -> failwith "pattern compilation: project"
        else [frontier] 
        
    and bind_immediate (Active(path,subExpr,p) as inp) ((acc_active,acc_vmap) as s) = 
        let (SubExpr(accessf,ve)) = subExpr  in
        let mk_sub_active pathBuilder accessf'  j p'  = 
            Active(pathBuilder path j,SubExpr(accessf' j,ve),p') in
            
        match p with 
        | TPat_wild _ -> 
            bind_immediate_multi [] s 
        | TPat_as(p',pbind,m) -> 
            let (v,e') =  bind_subexpr g amap topgtvs pbind m subExpr in
            bind_immediate (Active(path,subExpr,p')) (acc_active,vspec_map_add v e' acc_vmap)
        | TPat_tuple(ps,tyargs,m) ->
            let accessf' j tpinst e' = mk_tuple_field_get(accessf tpinst e',inst_types tpinst tyargs,j,exprm) in 
            let pathBuilder path j = PathTuple(path,tyargs,j) in
            let new_actives = list_mapi (mk_sub_active pathBuilder accessf') ps in 
            bind_immediate_multi new_actives s 
        | TPat_recd(tcref,tinst,ps,m) -> 
            let new_actives = 
                list_mapi (fun j ((finst,p),fref) -> 
                    let accessf' fref j tpinst e' = mk_recd_field_get g (accessf tpinst e',fref,inst_types tpinst tinst,finst,exprm) in 
                    let pathBuilder path j = PathRecd(path,tcref,tinst,finst,j) in
                    mk_sub_active pathBuilder (accessf' fref) j p) 
                (combine ps (instance_rfrefs_of_tcref tcref)) in 
            bind_immediate_multi new_actives s 
        | TPat_disjs(ps,m) -> 
            mapConcat (fun p -> bind_immediate (Active(path,subExpr,p)) s)  ps
        | TPat_conjs(ps,m) -> 
            let new_actives = list_mapi (mk_sub_active (fun path j -> PathConj(path,j)) (fun j -> accessf)) ps in
            bind_immediate_multi new_actives s 
        
        | TPat_range (c1,c2,m) ->
            let res = ref [] in 
            for i = Nums.u16_to_int (Nums.unichar_to_u16 c1) to Nums.u16_to_int (Nums.unichar_to_u16 c2) do
                res :=  bind_immediate (Active(path,subExpr,TPat_const(TConst_char(Nums.u16_to_unichar (Nums.int_to_u16 i)),m))) s @ !res
            done;
            !res
        (* Assign an identifier to each TPat_query based on our knowledge of the 'identity' of the active pattern, if any *)
        | TPat_query ((_,_,apatVrefOpt,_,_),_,_) -> 
            let uniqId = match apatVrefOpt with None -> genUniquePathId() | Some vref -> stamp_of_vref vref in
            let inp = Active(PathQuery(path,uniqId),subExpr,p)  in
            [(inp::acc_active, acc_vmap)] 
        | _ -> 
            [(inp::acc_active, acc_vmap)] 
    and bind_immediate_multi ps s =
        fold_right (fun p sofar -> mapConcat (bind_immediate p) sofar) ps [s]  in

    (* The setup routine of the match compiler *)
    let dtree = 
      compile
        []
        (concat 
           (list_mapi 
              (fun i (TClause(p,opt_when,rhs,_)) -> 
                let initialSubExpr = SubExpr((fun tpinst x -> x),(expr_for_val (range_of_val topv) topv,topv)) in 
                let investigations = bind_immediate (Active(PathEmpty(ty),initialSubExpr,p)) ([],vspec_map_empty()) in 
                mk_frontiers investigations i)
              clausesL)
          @ 
          mk_frontiers [([],vspec_map_empty())] nclauses) in 

    let targets = MatchBuilder.close_targets mbuilder in

    (* Report unused targets *)
    let used = acc_targets_of_dtree dtree [] in 
    let _ = if warnOnUnused then list_iteri (fun i (TClause(_,_,_,patm)) ->  if not (mem i used) then warning (RuleNeverMatched patm)) clausesL in 
    dtree,targets
  
let isPartialOrWhenClause c = is_partial (pat_of_clause c) or (isSome (when_of_clause c))


let rec compilePattern nng g denv amap exprm matchm warnOnUnused actionOnFailure (topv,topgtvs) (clausesL: tclause list) ty =
  match clausesL with 
  | _ when List.exists isPartialOrWhenClause clausesL ->
        (* Partial clauses cause major code explosion if treated naively *)
        (* Hence treat any pattern matches with any partial clauses clause-by-clause *)
        
        (* First make sure we generate at least some of the obvious incomplete match warnings. *)
        let warnOnUnused = false in (* we can't turn this on since we're pretending all partial's fail in order to control the complexity of this. *)
        let warnOnIncomplete = true in
        let clausesPretendAllPartialFail = mapConcat (fun (TClause(p,whenOpt,tg,m)) -> [TClause(erase_partial p,whenOpt,tg,m)]) clausesL in 
        let _ = compilePatternBasic nng g denv amap exprm matchm warnOnUnused warnOnIncomplete actionOnFailure (topv,topgtvs) clausesPretendAllPartialFail ty in 
        let warnOnIncomplete = false in
        
        let rec atMostOnePartialAtATime clauses = 
            if debug then dprintf1 "atMostOnePartialAtATime: #clauses = %d\n" (length clauses);
            match list_take_until isPartialOrWhenClause clauses with 
            | l,[]       -> compilePatternBasic nng g denv amap exprm matchm warnOnUnused warnOnIncomplete actionOnFailure (topv,topgtvs) l ty
            | l,(h :: t) -> doGroupWithAtMostOnePartial (l @ [h]) t
        and doGroupWithAtMostOnePartial group rest = 
            if debug then dprintf1 "doGroupWithAtMostOnePartial: #group = %d\n" (length group);
            let dtree,targets = atMostOnePartialAtATime rest in 
            let expr = mk_and_optimize_match exprm matchm ty dtree targets in 
            compilePatternBasic nng g denv amap exprm matchm warnOnUnused warnOnIncomplete actionOnFailure (topv,topgtvs) (group @ [TClause(TPat_wild matchm,None,TTarget([],expr),matchm)]) ty in 
        

        atMostOnePartialAtATime clausesL
      
  | _ -> compilePatternBasic nng g denv amap exprm matchm warnOnUnused true actionOnFailure (topv,topgtvs) (clausesL: tclause list) ty
