// (c) Microsoft Corporation 2005-2007. 

namespace Microsoft.FSharp.Compatibility.FSharp

#light

#nowarn "0057";; // active patterns

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Control
open Microsoft.FSharp.Collections
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

/// LazyLists are possibly-infinite, cached sequences.  See also IEnumerable/Seq for
/// uncached sequences. Calling "get" on the same lazy list value you will keep 
/// getting the same (cached) result.  LazyLists normally involve delayed computations
/// without side-effects, and calling "get" may cause these computations to be executed.  The results 
/// of these computations are cached - evaluations will be performed 
/// only once for each element of the lazy list.  This is different to IEnumerable/Seq where
/// recomputation happens each time an enumerator is created and the sequence traversed.
///
/// LazyLists can represent cached potentially-infinite computations.  Because they are cached they may cause 
/// memory leaks if some part of your code maintains a live reference to
/// the head of an infinite or very large lazy list while iterating it, or if a reference is
/// maintained after the list is no longer required.
///
/// Although lazy lists are an abstract type you may pattern match against them using the
/// LazyList.Cons and LazyList.Nil active patterns. These may force the computation of elements
/// of the list.

// abstract type: implementation details hidden.
type 'a LazyList with 
    interface IEnumerable<'a>
    interface System.Collections.IEnumerable
    

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module LazyList =

    type 'a t      = LazyList<'a>     // For use when not opening module, e.g. Map.t
    type 'a llist  = LazyList<'a>     // More ML-friendly names

    /// Test if a stream contains at least one element.  Forces the evaluation of
    /// the first element of the stream if it is not already evaluated.
    val nonempty : LazyList<'a> -> bool

    /// Return the first element of the stream.  Raise 'Invalid_argument "hd"' if the
    /// stream is empty. Forces the evaluation of
    /// the first cell of the stream if it is not already evaluated.
    val hd       : LazyList<'a> -> 'a

    /// Return the stream corresponding to the remaining items in the sequence.  
    /// Raise 'Invalid_argument "tl"' if the stream is empty. Forces the evaluation of
    /// the first cell of the stream if it is not already evaluated.
    val tl       : LazyList<'a> -> LazyList<'a>

    /// Get the first cell of the stream.
    val get      : LazyList<'a> -> ('a * LazyList<'a>) option  

    /// Return the stream which on consumption will consist of at most 'n' elements of 
    /// the given stream.  Does not force the evaluation of any cells in the stream.
    val take     : int -> LazyList<'a> -> LazyList<'a>

    /// Return the stream without the first 'n' elements of the given stream.  Does
    /// not force the evaluation of any cells in the stream.
    val drop     : int -> LazyList<'a> -> LazyList<'a>

    /// Apply the given function to successive elements of the list, returning the first
    /// result where function returns 'Some(x)' for some x. If the funtion never returns
    /// true, 'None' is returned.
    val first    : ('a -> bool) -> LazyList<'a> -> 'a option
    /// Return the first element for which the given function returns "true".
    /// Raise 'Not_found' if no such element exists.
    val find     : ('a -> bool) -> LazyList<'a> -> 'a (* or raise Not_found *)

    /// Evaluates to the stream that contains no items
    val empty    : unit -> LazyList<'a>

    /// Return a new stream which contains on demand the given item followed by the
    /// given stream.
    val cons     : 'a -> LazyList<'a>               -> LazyList<'a>

    /// Return a new stream which contains on demand the given item followed by the
    /// stream returned by the given computation.  The computation is
    /// not executed until the elements of the stream are consumed.  The
    /// computation is only executed once.
    val consf    : 'a -> (unit -> LazyList<'a>)     -> LazyList<'a>

    /// Return the stream which on consumption will consist of an infinite sequence of the given item
    val repeat   : 'a -> LazyList<'a>

    /// Return a stream that is in effect the stream returned by the given computation.
    /// The given computation is not executed until the first element on the stream is
    /// consumed.
    val delayed  : (unit -> LazyList<'a>)           -> LazyList<'a>

    /// Return a stream that contains the elements returned by the given computation.
    /// The given computation is not executed until the first element on the stream is
    /// consumed.  The given argument is passed to the computation.  Subsequent elements
    /// in the stream are generated by again applying the residual 'b to the computation.
    val unfold   : ('b -> ('a * 'b) option) -> 'b -> LazyList<'a>

    /// Return the stream which contains on demand the elements of the first stream followed
    /// by the elements of the second list
    val append   : LazyList<'a> -> LazyList<'a> -> LazyList<'a>
    /// Return the stream which contains on demand the pair of elements of the first and second list
    val combine  : LazyList<'a> -> LazyList<'b> -> LazyList<'a * 'b>
    /// Return the stream which contains on demand the list of elements of the list of lazy lists.
    val concat   : LazyList< LazyList<'a>> -> LazyList<'a>

    /// Return a new collection which on consumption will consist of only the elements of the collection
    /// for which the given predicate returns "true"
    val filter   : ('a -> bool) -> LazyList<'a> -> LazyList<'a>

    /// Return a new stream consisting of the results of applying the given accumulating function
    /// to successive elements of the stream
    val folds    : ('b -> 'a -> 'b) -> 'b -> LazyList<'a> -> LazyList<'b>  

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    val map      : ('a -> 'b) -> LazyList<'a> -> LazyList<'b>

    /// Build a new collection whose elements are the results of applying the given function
    /// to the corresponding elements of the two collections pairwise.
    val map2     : ('a -> 'b -> 'c) -> LazyList<'a> -> LazyList<'b> -> LazyList<'c>

    /// Build a collection from the given array. This function will eagerly evaluate all of the 
    /// stream (and thus may not terminate). 
    val of_array : 'a array -> LazyList<'a>

    /// Build an array from the given collection
    val to_array : LazyList<'a> -> 'a array

    /// Build a collection from the given list. This function will eagerly evaluate all of the 
    /// stream (and thus may not terminate). 
    val of_list  : list<'a> -> LazyList<'a>

    /// Build a list from the given collection This function will eagerly evaluate all of the 
    /// stream (and thus may not terminate). 
    val to_list  : LazyList<'a> -> list<'a>

    /// Return a view of the collection as an enumerable object
    [<System.Obsolete("Consider using to_seq instead")>]
    val to_IEnumerable: LazyList<'a> -> IEnumerable<'a>

    /// Build a new collection from the given enumerable object
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_IEnumerable: #IEnumerable<'a> -> LazyList<'a>

    /// Return a view of the collection as an enumerable object
    val to_seq: LazyList<'a> -> seq<'a>

    /// Build a new collection from the given enumerable object
    val of_seq: #seq<'a> -> LazyList<'a>

    #if CLI_AT_LEAST_2_0
    /// Return a view of the collection as a .NET collection 
    val to_ICollection: LazyList<'a> -> ICollection<'a>

    ///  Build a new collection from any type that supports the .NET ICollection interface
    [<System.Obsolete("Consider using of_IEnumerable instead")>]
    val of_ICollection: #ICollection<'a> -> LazyList<'a>
    #endif

    //--------------------------------------------------------------------------
    // Lazy list active patterns

    val (|Cons|Nil|) : 'a llist -> Choice<('a * 'a llist),unit>

