
namespace Microsoft.FSharp.Compiler.Interactive

type IEventLoop =
    interface 
        /// Enter the run loop on the main thread of the event loop.
        /// Call this from the GUI thread.
        ///
        /// A return of 'true' indicates 'I was replaced mid-flight and you can try to restart a different event loop'
        /// A return of 'false' indicates a normal exit, e.g. a call to System.Windows.Forms.Application.Exit()
        abstract Run : unit -> bool  
        /// Call this from a worker thread. Runs the operation synchronously.
        abstract Invoke : (unit -> 'a) -> 'a 
        /// Call this from a worker thread. Notify the event loop that 
        /// it should exit the 'Run' loop on its main thread with a return value 
        /// of 'true' indicating an attempt to restart.  Ignored if not running.
        abstract ScheduleRestart : unit -> unit
    end
    
type InteractiveSession 
  with
    member FloatingPointFormat: string with get,set
    member FormatProvider: System.IFormatProvider  with get,set
    member PrintWidth : int  with get,set
    member PrintDepth : int  with get,set
    member PrintLength : int  with get,set
    member ShowProperties : bool  with get,set
    member ShowIEnumerable: bool  with get,set
    member ShowIDictionary: bool  with get,set
    member PrintIntercepts: (StructuredFormat.IEnvironment -> obj -> StructuredFormat.Layout option) list with get,set
    member AddPrinter: ('a -> string) -> unit
    
    /// The command line arguments after ignoring the arguments relevant to the interactive
    /// environment and replacing the first argument with the name of the last script file,
    /// if any. Thus 'fsi.exe test1.fs test2.fs -- hello goodbye' will give arguments
    /// 'test2.fs', 'hello', 'goodbye'.  This value will normally be different to those
    /// returned by System.Environment.GetCommandLineArgs and Sys.argv.
    member CommandLineArgs : string [] with get,set
    
    /// This property controls how F# Interactive executes code in the GUI context
    member EventLoop: IEventLoop with get,set
    
    /// This method is automatically called when a thread dies in an F# Interactive session
    member ReportThreadException: exn -> unit
    
    /// This event is triggered by a call to ReportThreadException. At most one listener may be active on this
    /// event, so registering additional listeners will cause previous listeners to be dropped.
    member ThreadException: IEvent<exn>
    
    /// Add a printer that shows values of type 'a as if they were values
    /// returned by the given function
    member AddPrintTransformer: ('a -> obj) -> unit
  end

module Settings : sig

  val fsi : InteractiveSession
end

/// Hooks (internal use only, may change without notice).
module Internals : sig
  val getFsiPrintOptions : unit -> Microsoft.FSharp.Core.Pervasives.FormatOptions 
  val saveIt : 'a -> unit  
  val getSavedIt : unit -> obj
  val getSavedItType : unit -> System.Type
(*    val openPaths : unit -> string[] *)
end

