
// F# Visual Studio Sample Script File
//
// This file contains some sample constructs to guid you through the
// primitives of F# Interactive.
//
// You can run this file as a standalone script using 'fsi --exec script.fsx'
//
// Contents:
//   - Simple computations in F# Interactive
//   - Using .NET libraries and F# Interactive
//   - Using Intellisense Functions on integers.  
//   - Tuples 
//   - Strings
//   - Lists
//   - Arrays
//   - Functions

// Simple computations in F# Interactive
// ---------------------------------------------------------------
// Below is a simple computation.  If you're entering these manually into
// F# Interactive then separate interactive entries with ';;'.  
// You can run this code by starting F# Interactive - go to the
// Tools->AddIn menu, and select F# Interactive, and you should see a new
// window appear. Redock the window in an appropriate place or display on 
// a second monitor.  Then select the line below (e.g. with the keyboard by 
// pressing shift and the 'down arrow') and press Alt-Enter.  The code will now
// evaluate in the interactive session.

#light

printfn "Hello World";;

// You can also use 'do': 

do printfn "Hello again"

// Using .NET libraries
// ---------------------------------------------------------------
// Try the following sample program:

open System.Windows.Forms
let form = new Form()
form.Visible <- true
form.TopMost <- true
form.Text <- "Hello WinForms!"

// You can also set properties as you construct objects:

let form2 = new Form(Visible=true, TopMost=true, Text="Hello WinForms!")

// Using Intellisense
// ---------------------------------------------------------------
// Take the above program, and try retyping it in the space below.  You will see menus
// appear while you are typing.  These give hints appropriate to the context.
// You will also see automatic typechecking reporting errors as you are working.
// You can see all the errors by finding the 'Errors' window.
//
// Try ctrl-J on (partial) identifiers.  


// The rest of this file contains some simple samples of basic F#
// programming.  Enjoy!
// ---------------------------------------------------------------

// Simple arithmetic:
let xA = 1
let xB = 2
let xC = xA + xB
let twist x = 10 - x
printfn "res = %d" (twist (xC+4))
  // note: let binds values and functions


// Functions on integers.  
// ---------------------------------------------------------------

let inc x = x + 1
let rec fact n = if n=0 then 1 else n * fact (n-1)

/// Highest-common-factor 
let rec hcf a b =           // notice: 2 arguments seperated by spaces
    if a=0 then b
    elif a<b then hcf a (b-a)           // notice: 2 arguments seperated by spaces
    else hcf (a-b) b
    // note: function arguments are typically space seperated.
    // note: let rec binds recursive functions.

      
// Tuples.  These combine values into packets.
// ---------------------------------------------------------------
let pA = (1,2,3)
let pB = (1,"fred",3.1415)
let swap (a,b) = (b,a)

/// Note: hcf2 takes one argument which is in fact a tuple.
let rec hcf2 (a,b) = 
    if a = 0 then b
    else if a<b then hcf2 (a,b-a)
    else hcf2 (a-b,a)


// Booleans.
// ---------------------------------------------------------------

let bA = true
let bB = false
let bC = not bA && (bB || false)


// Strings.
// ---------------------------------------------------------------

let sA  = "hello"
let sB  = "world"
let sC  = sA + " " + sB
let sC2 = String.concat " " [sA;sB]
printfn "sC = %s, sC2 = %s" sC sC2
sprintf "sC = %s, sC2 = %d" sC sC2.Length


// Functional, Immutable Linked Lists
// ---------------------------------------------------------------

let xsA = [ ]           // empty list
let xsB = [ 1;2;3 ]     // list of 3 ints
let xsC = 1 :: [2;3]    // :: is cons operation.

printfn "xsC = %A" xsC

// Recursive functions over lists.
// ---------------------------------------------------------------

let rec sumList xs =
    match xs with
    | []    -> 0
    | y::ys -> y + sumList ys

let y   = sumList [1;2;3]  // sum a list
let xsD = xsA @ [1;2;3]    // append
let xsE = 99 :: xsD        // cons on front


// Mutable Arrays, a primitive for efficient computations
// ---------------------------------------------------------------

let arr = Array.create 4 "hello"
arr.[1] <- "world"
arr.[3] <- "don"
let arrLength = arr.Length         // instance method on array object
let arrLength2  = Array.length arr // same, using function in Array module

let front  = arr.[0..1]          // first 2 elements, using slicing syntax
let front2 = Array.sub arr 0 2   // first 2 elements, using function in Array module

  // Trying re-typing the above line to see intellisense in action.
  // Note, ctrl-J on (partial) identifiers re-activates it.

// Other common data structures
// ---------------------------------------------------------------

// See namespaces 
//   System.Collections.Generic
//   Microsoft.FSharp.Collections
//   Hashtbl
//   Set
//   Map


// Functions
// ---------------------------------------------------------------

let inc2 x = x + 2              // as a function definition
let inc3   = fun x -> x + 3     // as a lambda expression

let ysA = List.map inc2 [1;2;3]
let ysB = List.map inc3 [1;2;3]
let ysC = List.map (fun x -> x+4) [1;2;3]

let addFour x = x + 4
let aboveFive x = x > 4

// Pipelines:
let pipe1 = [1;2;3] |> List.map (fun x -> x+4) 
let pipe2 = 
    [1;2;3] 
    |> List.map (fun x -> x+4) 
    |> List.filter (fun x -> x>5) 

let pipe3 = 
    [1;2;3] 
    |> List.map addFour 
    |> List.filter aboveFive 

// Composition pipelines:
let processor  = List.map (fun x -> x+4) >> List.filter (fun x -> x>5) 
let processor2 = List.map addFour >> List.filter aboveFive 

// Types - datatypes
// ---------------------------------------------------------------

type expr = 
    | Num of int
    | Add of expr * expr
    | Sub of expr * expr
    | Mul of expr * expr
    | Div of expr * expr
    | Var of string
  
let lookup id (env : (string * int) list) = List.assoc id env
  
let rec eval env exp = 
    match exp with
    | Num n -> n
    | Add (x,y) -> eval env x + eval env y
    | Sub (x,y) -> eval env x - eval env y
    | Mul (x,y) -> eval env x * eval env y
    | Div (x,y) -> eval env x / eval env y
    | Var id    -> lookup id env
  
let envA = [ "a",1 ; "b",2 ; "c",3 ]
             
let expT1 = Add(Var "a",Mul(Num 2,Var "b"))
let resT1 = eval envA expT1


// Types - records
// ---------------------------------------------------------------

type Card = { Name  : string;
              Phone : string;
              Ok    : bool }
              
let cardA = { Name = "Alf" ; Phone = "+44.1223.000.000" ; Ok = false }
let cardB = {cardA with Phone = "+44.1223.123.456"; Ok = true }
let string_of_card c = 
    c.Name + " Phone: " + c.Phone + (if not c.Ok then " (unchecked)" else "")


// Here's a longer construction syntax should you get name conflicts:
let cardC = { new Card 
              with Name = "Alf" 
              and Phone = "+44.1223.000.000" 
              and Ok = false }


// Types - interfaces, which are like records of functions
// ---------------------------------------------------------------

type IPeekPoke = 
    abstract member Peek: unit -> int
    abstract member Poke: int -> unit

let Counter(initialState) = 
    let state = ref initialState
    { new IPeekPoke with 
        member x.Poke(n) = state := state.Value + n
        member x.Peek() = state.Value  }
              
// Types - classes
// ---------------------------------------------------------------

type Widget(initialState:int) = 
    let mutable state = initialState
    member x.Poke(n) = state <- state + n
    member x.Peek() = state 
    member x.HasBeenPoked = (state <> 0)
              
// Types - classes with interface implementations
// ---------------------------------------------------------------

type Wodget() = 
    let mutable state = 0
    interface IPeekPoke with 
        member x.Poke(n) = state <- state + n
        member x.Peek() = state 
    member x.HasBeenPoked = (state <> 0)
              