/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package junit.framework;

import ccl.util.Test;
import ccl.util.Util;
import java.lang.reflect.Method;
import java.util.Enumeration;
import java.util.Vector;

/**
 * This class is a fake for JUnit test classes. It redirects all your test code to use the
 * test class from the ccl.util.Test class.<p>
 *
 * To execute your tests written for the JUnit API with the
 * ccl.util.Test class, write code like this:
 *
 * <pre>
 *     public void main( String[] args ) {
 *         TestCase testCase = new MyTest();
 *         testCase.setVerbose( true );
 *         testCase.setTiming ( true );
 *         testCase.run();
 *         testCase.printResults();
 * 
 *         System.exit( 0 );
 *     }
 * </pre>
 *
 * @see ccl.util.Test   for details how to build a hierarchy of
 *                      tests
 *
 * @author  Chr. Clemens Lee
 * @version $Id: TestCase.java,v 1.4 2003/05/01 16:45:00 clemens Exp clemens $
 */
public class TestCase extends Test
{
    /**
     * This method gets invoked from the ccl.util.Test class and is responsible for
     * executed all test code.
     * We do it like junit and find via reflection all public void methods without
     * parameters which start with 'test'. Alternatively, if a 'suite' method exists
     * in the implemented test case class, then of course the test methods and classes 
     * specified there have to be used.
     */
    protected void _doIt()
        throws Exception
    {
        // get suite
        // find tests
        //    get list of all public void methods starting with test and no parameters
        Vector vTestMethods = getTestMethods( this );
        //    invoke each of them
        Enumeration eMethods = vTestMethods.elements();
        while( eMethods.hasMoreElements() )
        {
            invokeTestMethod( (Method) eMethods.nextElement() );
        }
    }

    /**
     * Default constructor expected in Test.
     */
    public TestCase() 
    {
        super();
    }

    /**
     * Standard constructor expected in Test.
     */
    public TestCase( Test test ) 
    {
        super( test );
    }

    /**
     * Standard constructor expected in JUnit TestCase.
     */
    public TestCase( String testName ) 
    {
        super();
    }

    /**
     * Maps the JUnit standard assertion to Test's assertion.
     */
    public void assert( String message, boolean check )
    {
        super.Assert( check, message );
    }

    /**
     * A workaround method to be able to use reflection to invoke the super method
     * _enterSubTest.
     */
    protected void enterSubTest( String testName )
    {
        super._enterSubTest( testName );
    }

    /**
     * A workaround method to be able to use reflection to invoke the super method
     * _exitSubTest.
     */
    protected void exitSubTest()
    {
        super._exitSubTest();
    }

    /**
     * Used to invoke the test methods.
     */
    public void invokeTestMethod( Method testMethod )
        throws Exception
    {
        invokeTestMethod( testMethod, this );
    }

    /**
     * Used to invoke the test methods.
     */
    static public void invokeTestMethod( Method testMethod, Object object )
        throws Exception
    {
        testMethod.invoke( object, new Object[ 0 ] );
    }

    /**
     * Returns all methods of an object that conform to the test method convention.
     */
    static public Vector getTestMethods( Object object )
    {
        Vector vRetVal = new Vector();

        Vector vMethods = getMethods( object );
        Enumeration eMethods = vMethods.elements();
        while( eMethods.hasMoreElements() )
        {
            Method nextMethod = (Method) eMethods.nextElement();
            if ( nextMethod.getName().startsWith( "test" ) 
                 && nextMethod.getReturnType().getName().equals( "void" )
                 && nextMethod.getParameterTypes().length == 0 )
            {
                vRetVal.addElement( nextMethod );
            }
        }

        return vRetVal;
    }

    /**
     * Returns all public methods of an object.
     */
    static public Vector getMethods( Object object )
    {
        Method[] aMethods = object.getClass().getMethods();

        return Util.objectsToVector( aMethods );
    }

    /*
    public void run( TestResult testResult )
    {
        throw new RuntimeException( getClass().getName() + ".run(TestResult) not implemented" );
    }
    */
}
