/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Paddy Ramanathan.
 * Portions created by Paddy Ramanathan are Copyright (C) 1999
 * Paddy Ramanathan. All Rights Reserved.
 * Portions created by Chr. Clemens Lee are Copyright (C) 1999,
 * 2000,2001,2002 Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Paddy Ramanathan <rpaddy@concordesolutions.com>,
 *                 Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util;

import java.io.ByteArrayInputStream;
import java.util.Properties;

/** 
 * Reads the file pointed to by value of the JVM option
 * SYS_ENV as set of name value pairs 
 * (as per java.util.Properties class). Keep all the
 * envoirnment variables that you need Jacob to know in this
 * file.
 * See usage of the three public interface exposed by the
 * class.
 * Eg. to access $(HOME) in Jacob have entry
 * HOME=/users/me  in the properties file
 *
 * Before you can use this class, don't forget to initialize
 * it.
 * 
 * @author Paddy Ramanathan <rpaddy@concordesolutions.com> 1999-06-03
 *         Maintained also by Chr. Clemens Lee <clemens@kclee.com>
 * @version $Id: SysEnv.java,v 1.10 2002/10/06 16:42:11 clemens Exp clemens $
 */
public class SysEnv 
{
    /**
     * System property name used to read a file which contains
     * environment variables and their values.
     */
    public static final String S_SYS_ENV = "SYS_ENV";

    /**
     * FIX-ME  should this  be system dependent, now set to Unix compatible with make $(ENV)
     */
    private static final char C_ENV_MARKER = '$';
    private static final char C_ENV_END_MARKER = ')';

    private static Properties _propApplicationEnv = null;

    /*// FIX-ME should be called from  initialization module, dont know where
      static {
      init ();
      }*/

    /** 
     * Reads the properties files passed as parameter to 
     * JVM 
     */
    public static void initialize() 
    {
        String sPropFileName = System.getProperty( S_SYS_ENV );

        initialize( sPropFileName );
    }

    /** 
     * Reads the properties files passed as parameter to 
     * JVM 
     */
    public static void initialize( String sPropFileName_ ) 
    {
        if ( Util.isEmpty( sPropFileName_ ) ||
             (!FileUtil.existsFile( sPropFileName_ )) )
        {
            // file doesn't exist, so return.
            Util.debug( "ccl.util.SysEnv.initialize(..).NO_ENV_FILE" );
            
            return;
        }

        try
        {
            String sEnvVarsFileContent = FileUtil.readFile( sPropFileName_ );
            sEnvVarsFileContent = Util.replace( sEnvVarsFileContent,
                                                "\\", "\\\\" );

            ByteArrayInputStream inputStream = 
                   new ByteArrayInputStream( sEnvVarsFileContent.getBytes() );
            _propApplicationEnv = new Properties();
            _propApplicationEnv.load( inputStream );
            inputStream.close();
            Util.debug( "ccl.util.SysEnv.init().Listing all the envoirnment properties");
            Util.debug( "ccl.util.SysEnv.init()._propApplicationEnv: " +
                        _propApplicationEnv.propertyNames().toString() );
        }
        catch( Exception x ) 
        {
            Util.printlnErr( "ccl.util.SysEnv.init().x: " + x.getMessage() );
        }
    }

    /** 
     * Tests whether the string buffer contains any
     * envoirnment variable: syntax $(VAR_NAME) 
     */
    public static boolean isEnvEvalReqd ( String buffer ) 
    {
        Util.debug( "isEnv=" + buffer );
        return buffer.indexOf ( "$" ) == -1 ? false : true;
    }

    /**
     * Replaces all env. variables ( $(VAR_NAME) ) with
     * value from the SYS_ENV file, if not found
     * $(VAR_NAME) is replaced by empty space.
     */
    public static String evalEnvVariable  ( String buffer ) 
    {
        try 
        {
            Util.debug( "ccl.util.SysEnv.evalEnvVariable(..).buffer: " + buffer );
            int start = getStartPos( buffer );
            Util.debug("ccl.util.SysEnv.evalEnvVariable(..).start: " + start );
            if ( start == -1 ) 
            {
                return buffer;
            }
            
            String var = getVariable( buffer, start );
            Util.debug( "ccl.util.SysEnv.evalEnvVariable(..).var: " + var );
            int end = start + var.length () + 2; // the length of orig string to be replaced
            Util.debug("ccl.util.SysEnv.evalEnvVariable(..).end: " + end);
            String actualval = getActualVal ( var );
            Util.debug("actualval" + actualval);
            String eval = replace ( buffer, actualval, start, end );

            return( isEnvEvalReqd( eval ) ? evalEnvVariable( eval ) : eval );
        }
        catch ( Exception x ) 
        {
            Util.debug( "SysEnv.evalEnvVariable(..).x: " + x );

            return buffer;
        }
    }

    private static int getStartPos( String buffer ) 
    {
        return buffer.indexOf( C_ENV_MARKER );
    }

    private static String getVariable ( String buffer, int start ) throws Exception  
    {
        int end =  buffer.indexOf (  C_ENV_END_MARKER, start );
        if ( end == -1 ) 
        {
            throw new Exception ( "Env. eval failure: No " + C_ENV_END_MARKER + " for " + buffer + " after " + start );
        }

        return buffer.substring ( start + 2, end  );
    }

    private static String getActualVal ( String val ) 
    {
        Util.debug( "ccl.util.SysEnv.getActualVal(..).val: " + val );
        Util.debug( "ccl.util.SysEnv.getActualVal(..)._propApplicationEnv: " + 
                    _propApplicationEnv );
        String s = (String) _propApplicationEnv.get ( val );
        if ( s == null ) 
        {
            s = "";
        }

        return s;           
    }

    private static String replace ( String buffer, String actualval, int start, int end ) 
    {
        String pre = buffer.substring( 0, start );
        String post = buffer.substring( end + 1, buffer.length () );

        return pre + actualval + post;
    }

    /*
      public static void main ( String [] argv ) {
      init ();
      // for testing  
      }
    */
}
