/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util;

import java.io.PrintStream;
import java.util.Observable;

/**
 * Replace std out and err with special stream
 * and let listeners register themselves for
 * every occuring message to print out stuff
 * into a dialog e.g.<p>
 *
 * Here is an example of how you could use this class:
 * <pre>
 *   OutputErrorStreamManager streamManager = new OutputErrorStreamManager();
 *   streamManager.setAutoClear( false );
 *   streamManager.install();
 *   streamManager.setVisible( false );
 *
 *   doSomething();
 *
 *   String output = streamManager.getStreamsBuffer();
 *   streamManager.setVisible( true );
 *   streamManager.deinstall();
 *
 *   System.out.println( "Output of last action was: '" + output + "'" );
 * </pre>
 * 
 * @author  Chr. Clemens Lee (clemens@kclee.com)
 * @version $Id: OutputErrorStreamManager.java,v 1.7 2003/11/23 09:13:53 clemens Exp clemens $
 */
public class OutputErrorStreamManager extends Observable 
{
    /**
     * Clear internal buffer automatically on every flush
     * occasion.
     */
    private boolean _bAutoClear = true;

    /**
     * Store complete stream content in this buffer.
     */
    private StringBuffer _buffer = null;

    /**
     * Index in string of text which has not been flushed
     * so far. Used for notification.
     */
    private int _lastFlushIndex = 0;

    /**
     * Indicator to supress output into original streams.
     */
    private boolean _bVisible = true;

    /**
     * Special stream for std out.
     */
    private OutputErrorStream _newOut = null;

    /**
     * Special stream for std err.
     */
    private OutputErrorStream _newErr = null;

    /**
     * Used to restore original std out.
     */
    private PrintStream _outOrig = null;

    /**
     * Used to restore original std error.
     */
    private PrintStream _errOrig = null;

    /**
     * Create manager, but do not intall it.
     */
    public OutputErrorStreamManager() 
    {
        super();
    }

    /**
     * Clear internal buffer automatically on every flush
     * occasion.
     */
    public void setAutoClear( boolean bAutoClear_ ) 
    {
        _bAutoClear = bAutoClear_;
    }

    /**
     * Install special output stream in Java system.
     */
    public void install() 
    {
        Util.panicIf( _buffer != null );

        _outOrig = System.out;
        _errOrig = System.err;

        _buffer = new StringBuffer( 256 );
        _newOut = new OutputErrorStream( _outOrig );
        _newErr = new OutputErrorStream( _errOrig );
        try 
        {
            System.setOut( _newOut );
            System.setErr( _newErr );
        }
        catch( SecurityException exception ) 
        {
            Util.printlnErr( "Warning: an AccessControlException happened "
                             + "while trying to redirect\n"
                             + "standard output streams. Please relax your "
                             + "security policy for your jdk\n"
                             + "in order to let the application run with its "
                             + "full functionality.\n"
                             + "You should grand more permissions in file "
                             + "$JAVA_HOME/jre/lib/security/java.policy.\n"
                             + "E.g. put\n'grant {\n    permission "
                             + "java.security.AllPermission;\n};'\n"
                             + "in that file (backup the original content).\n"
                             + "If you don't want to do anything like this the "
                             + "application should still\n"
                             + "work with most of its functionality.\n" );
            /*Util.printlnErr( exception );*/
        }
    }

    /**
     * Uninstall special output stream in Java system
     * and restore back original streams.
     */
    public void deinstall() 
    {
        if ( _buffer != null ) 
        {
            try 
            {
                System.setOut( _outOrig );
                System.setErr( _errOrig );
            } 
            catch( SecurityException exception ) 
            {
                Util.println( "Again AccessControlException: please relax your "
                              + "security "
                              + "policy for your jdk\n"
                              + "in order to let Jacob run with its full "
                              + "functionality.\nYou should grand more "
                              + "permissions in file e.g. "
                              + "$JAVA_HOME/jre/lib/security/java.policy." );
                /*exception.printStackTrace();*/
            }
        }
        _buffer = null;
        _newOut = null;
        _newErr = null;
    }
    
    /**
     * Return String with content of everything which has been
     * printed into out and err streams.
     */
    public String getStreamsBuffer() 
    {
        String retVal = null;

        synchronized( _buffer )
               {
                   retVal = _buffer.toString();
               }

        return retVal;
    }

    /**
     * Clear stream output buffers.
     */
    public void clearStreamsBuffer() 
    {
        synchronized( _buffer ) 
        {
            _buffer.setLength( 0 );
            _lastFlushIndex = 0;
        }
    }

    /**
     * Indicator to supress output into original streams.
     */
    public void setVisible( boolean bVisible_ ) 
    {
        _bVisible = bVisible_;
    }

    /**
     * Indicates that this object has changed. 
     */
    protected synchronized void setChanged() 
    {
        super.setChanged();
    }

    /**
     * Special stream which hooks into out or err stream.
     */
    public class OutputErrorStream extends PrintStream 
    {
        private PrintStream _out = null;

        public OutputErrorStream( PrintStream out_ ) 
        {
            super( out_ );

            _out = out_;
        }
        
        private final void _notifyObservers() 
        {
            synchronized( _buffer ) 
            {
                int len = _buffer.length() - _lastFlushIndex;
                if ( len > 0 ) 
                {
                    char[] acNewOutput = new char[ len ];
                    _buffer.getChars( _lastFlushIndex
                                      , _buffer.length()
                                      , acNewOutput
                                      , 0 );
                    notifyObservers( new String( acNewOutput ) );
                    if ( _bAutoClear ) 
                    {
                        clearStreamsBuffer();
                    }
                    else 
                    {
                        _lastFlushIndex = _buffer.length();
                    }
                }
            }
        }
        
        public void flush() 
        {
            _notifyObservers();
            _out.flush();
        }
        
        /*public void close() {
            _out.close();
        }

        public boolean checkError() {
            return _out.checkError();
            }*/

        public void write(int b) 
        {
            synchronized( _buffer ) 
            {
                _buffer.append( (char) b );
                OutputErrorStreamManager.this.setChanged();

                if ( (char) b == '\n' ) 
                {
                    _notifyObservers();
                }
            }
            if ( _bVisible ) 
            {
                _out.write( b );
            }
        }

        public void write(byte buf[], int off, int len) 
        {
            for( int index = off; index - off < len; index++ ) 
            {
                this.write( buf[ index ] );
            }
            /*_out.write( buf, off, len );*/
        }

        /*public void print(boolean b) {
            synchronized( _buffer ) {
                _buffer.append(b ? "true" : "false");
            }
            _out.print( b );
        }

        public void print(char c) {
            synchronized( _buffer ) {
                _buffer.append( c );
            }
            _out.print( c );
        }

        public void print(int i) {
            synchronized( _buffer ) {
                _buffer.append( i );
            }
            _out.print( i );
        }

        public void print(long l) {
            synchronized( _buffer ) {
                _buffer.append( l );
            }
            _out.print( l );
        }

        public void print(float f) {
            synchronized( _buffer ) {
                _buffer.append( f );
            }
            _out.print( f );
        }

        public void print(double d) {
            synchronized( _buffer ) {
                _buffer.append( d );
            }
            _out.print( d );
        }

        public void print(char s[]) {
            synchronized( _buffer ) {
                _buffer.append( s );
            }
            _out.print( s );
        }

        public void print(String s) {
            synchronized( _buffer ) {
                _buffer.append( s );
            }
            _out.print( s );
        }

        public void print(Object obj) {
            synchronized( _buffer ) {
                _buffer.append( String.valueOf( obj ) );
            }
            _out.print( obj );
        }

        public void println() {
            synchronized( _buffer ) {
                _buffer.append( '\n' );
            }
            _out.println();
        }

        public void println(boolean x) {
            synchronized( _buffer ) {
                _buffer.append(x ? "true\n" : "false\n");
            }
            _out.println( x );
        }

        public void println(char x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(int x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(long x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(float x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(double x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(char x[]) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(String x) {
            synchronized( _buffer ) {
                _buffer.append( x ).append( "\n" );
            }
            _out.println( x );
        }

        public void println(Object x) {
            synchronized( _buffer ) {
                _buffer.append( String.valueOf( x ) )
                       .append( "\n" );
            }
            _out.println( x );
            }*/
    }
}
