/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.bibtex;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.jabref.logic.util.strings.StringSimilarity;
import org.jabref.model.EntryTypes;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.AuthorList;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.EntryType;
import org.jabref.model.entry.FieldName;
import org.jabref.model.entry.FieldProperty;
import org.jabref.model.entry.InternalBibtexFields;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateCheck {
    private static final double DUPLICATE_THRESHOLD = 0.75;
    private static final Logger LOGGER = LoggerFactory.getLogger(DuplicateCheck.class);
    private static final int NOT_EQUAL = 0;
    private static final int EQUAL = 1;
    private static final int EMPTY_IN_ONE = 2;
    private static final int EMPTY_IN_TWO = 3;
    private static final int EMPTY_IN_BOTH = 4;
    private static final double DOUBT_RANGE = 0.05;
    private static final double REQUIRED_WEIGHT = 3.0;
    private static final Map<String, Double> FIELD_WEIGHTS = new HashMap<String, Double>();

    private DuplicateCheck() {
    }

    public static boolean isDuplicate(BibEntry one, BibEntry two, BibDatabaseMode bibDatabaseMode) {
        if (DuplicateCheck.haveSameIdentifier(one, two)) {
            return true;
        }
        if (DuplicateCheck.haveDifferentEntryType(one, two) || DuplicateCheck.haveDifferentEditions(one, two) || DuplicateCheck.haveDifferentChaptersOrPagesOfTheSameBook(one, two)) {
            return false;
        }
        EntryType type = EntryTypes.getTypeOrDefault(one.getType(), bibDatabaseMode);
        double[] reqCmpResult = DuplicateCheck.compareRequiredFields(type, one, two);
        if (DuplicateCheck.isFarFromThreshold(reqCmpResult[0])) {
            return reqCmpResult[0] >= 0.75;
        }
        return DuplicateCheck.compareOptionalFields(type, one, two, reqCmpResult);
    }

    private static boolean haveSameIdentifier(BibEntry one, BibEntry two) {
        for (String name : FieldName.getIdentifierFieldNames()) {
            if (!one.getField(name).isPresent() || !one.getField(name).equals(two.getField(name))) continue;
            return true;
        }
        return false;
    }

    private static boolean haveDifferentEntryType(BibEntry one, BibEntry two) {
        return !one.getType().equals(two.getType());
    }

    private static boolean haveDifferentEditions(BibEntry one, BibEntry two) {
        Optional<String> editionOne = one.getField("edition");
        Optional<String> editionTwo = two.getField("edition");
        return editionOne.isPresent() && editionTwo.isPresent() && !editionOne.get().equals(editionTwo.get());
    }

    private static boolean haveDifferentChaptersOrPagesOfTheSameBook(BibEntry one, BibEntry two) {
        return DuplicateCheck.compareSingleField("author", one, two) == 1 && DuplicateCheck.compareSingleField("title", one, two) == 1 && (DuplicateCheck.compareSingleField("chapter", one, two) == 0 || DuplicateCheck.compareSingleField("pages", one, two) == 0);
    }

    private static double[] compareRequiredFields(EntryType type, BibEntry one, BibEntry two) {
        double[] dArray;
        Set<String> requiredFields = type.getRequiredFieldsFlat();
        if (requiredFields == null) {
            double[] dArray2 = new double[2];
            dArray2[0] = 0.0;
            dArray = dArray2;
            dArray2[1] = 0.0;
        } else {
            dArray = DuplicateCheck.compareFieldSet(requiredFields, one, two);
        }
        return dArray;
    }

    private static boolean isFarFromThreshold(double value) {
        return Math.abs(value - 0.75) > 0.05;
    }

    private static boolean compareOptionalFields(EntryType type, BibEntry one, BibEntry two, double[] req) {
        double denominator;
        Set<String> optionalFields = type.getOptionalFields();
        if (optionalFields == null) {
            return req[0] >= 0.75;
        }
        double[] opt = DuplicateCheck.compareFieldSet(optionalFields, one, two);
        double numerator = 3.0 * req[0] * req[1] + opt[0] * opt[1];
        double totValue = numerator / (denominator = req[1] * 3.0 + opt[1]);
        return totValue >= 0.75;
    }

    private static double[] compareFieldSet(Collection<String> fields, BibEntry one, BibEntry two) {
        double res = 0.0;
        double totWeights = 0.0;
        for (String field2 : fields) {
            double weight = FIELD_WEIGHTS.getOrDefault(field2, 1.0);
            totWeights += weight;
            int result = DuplicateCheck.compareSingleField(field2, one, two);
            if (result == 1) {
                res += weight;
                continue;
            }
            if (result != 4) continue;
            totWeights -= weight;
        }
        if (totWeights > 0.0) {
            return new double[]{res / totWeights, totWeights};
        }
        return new double[]{0.5, 0.0};
    }

    private static int compareSingleField(String field2, BibEntry one, BibEntry two) {
        Optional<String> optionalStringOne = one.getField(field2);
        Optional<String> optionalStringTwo = two.getField(field2);
        if (!optionalStringOne.isPresent()) {
            if (!optionalStringTwo.isPresent()) {
                return 4;
            }
            return 2;
        }
        if (!optionalStringTwo.isPresent()) {
            return 3;
        }
        String stringOne = optionalStringOne.get();
        String stringTwo = optionalStringTwo.get();
        if (InternalBibtexFields.getFieldProperties(field2).contains((Object)FieldProperty.PERSON_NAMES)) {
            return DuplicateCheck.compareAuthorField(stringOne, stringTwo);
        }
        if ("pages".equals(field2)) {
            return DuplicateCheck.comparePagesField(stringOne, stringTwo);
        }
        if ("journal".equals(field2)) {
            return DuplicateCheck.compareJournalField(stringOne, stringTwo);
        }
        if ("chapter".equals(field2)) {
            return DuplicateCheck.compareChapterField(stringOne, stringTwo);
        }
        return DuplicateCheck.compareField(stringOne, stringTwo);
    }

    private static int compareAuthorField(String stringOne, String stringTwo) {
        String authorTwo;
        String authorOne = AuthorList.fixAuthorLastNameOnlyCommas(stringOne, false).replace(" and ", " ").toLowerCase(Locale.ROOT);
        double similarity = DuplicateCheck.correlateByWords(authorOne, authorTwo = AuthorList.fixAuthorLastNameOnlyCommas(stringTwo, false).replace(" and ", " ").toLowerCase(Locale.ROOT));
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    private static int comparePagesField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = stringOne.replaceAll("[- ]+", "-");
        if (processedStringOne.equals(processedStringTwo = stringTwo.replaceAll("[- ]+", "-"))) {
            return 1;
        }
        return 0;
    }

    private static int compareJournalField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = stringOne.replace(".", "").toLowerCase(Locale.ROOT);
        double similarity = DuplicateCheck.correlateByWords(processedStringOne, processedStringTwo = stringTwo.replace(".", "").toLowerCase(Locale.ROOT));
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    private static int compareChapterField(String stringOne, String stringTwo) {
        String processedStringOne = stringOne.replaceAll("(?i)chapter", "").trim();
        String processedStringTwo = stringTwo.replaceAll("(?i)chapter", "").trim();
        return DuplicateCheck.compareField(processedStringOne, processedStringTwo);
    }

    private static int compareField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = stringOne.toLowerCase(Locale.ROOT).trim();
        double similarity = DuplicateCheck.correlateByWords(processedStringOne, processedStringTwo = stringTwo.toLowerCase(Locale.ROOT).trim());
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    public static double compareEntriesStrictly(BibEntry one, BibEntry two) {
        HashSet<String> allFields = new HashSet<String>();
        allFields.addAll(one.getFieldNames());
        allFields.addAll(two.getFieldNames());
        int score = 0;
        for (String field2 : allFields) {
            Optional<String> stringTwo;
            Optional<String> stringOne = one.getField(field2);
            if (!stringOne.equals(stringTwo = two.getField(field2))) continue;
            ++score;
        }
        if (score == allFields.size()) {
            return 1.01;
        }
        return (double)score / (double)allFields.size();
    }

    public static Optional<BibEntry> containsDuplicate(BibDatabase database, BibEntry entry, BibDatabaseMode bibDatabaseMode) {
        for (BibEntry other : database.getEntries()) {
            if (!DuplicateCheck.isDuplicate(entry, other, bibDatabaseMode)) continue;
            return Optional.of(other);
        }
        return Optional.empty();
    }

    public static double correlateByWords(String s1, String s2) {
        String[] w1 = s1.split("\\s");
        String[] w2 = s2.split("\\s");
        int n = Math.min(w1.length, w2.length);
        int misses = 0;
        for (int i = 0; i < n; ++i) {
            double corr = DuplicateCheck.similarity(w1[i], w2[i]);
            if (!(corr < 0.75)) continue;
            ++misses;
        }
        double missRate = (double)misses / (double)n;
        return 1.0 - missRate;
    }

    private static double similarity(String first, String second) {
        String shorter;
        String longer;
        if (first.length() < second.length()) {
            longer = second;
            shorter = first;
        } else {
            longer = first;
            shorter = second;
        }
        int longerLength = longer.length();
        if (longerLength == 0) {
            return 1.0;
        }
        double distanceIgnoredCase = new StringSimilarity().editDistanceIgnoreCase(longer, shorter);
        double similarity = ((double)longerLength - distanceIgnoredCase) / (double)longerLength;
        LOGGER.debug("Longer string: " + longer + " Shorter string: " + shorter + " Similarity: " + similarity);
        return similarity;
    }

    static {
        FIELD_WEIGHTS.put("author", 2.5);
        FIELD_WEIGHTS.put("editor", 2.5);
        FIELD_WEIGHTS.put("title", 3.0);
        FIELD_WEIGHTS.put("journal", 2.0);
    }
}

